"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataSource = exports.DataSourceBase = exports.DataSourceConfigurationBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const cdk = require("aws-cdk-lib");
const datazone = require("aws-cdk-lib/aws-datazone");
const resource_1 = require("./resource");
const IS_DATA_SOURCE = Symbol.for('amazon-datazone.DataSource');
class DataSourceConfigurationBase {
    static fromGlue(options) {
        const configuration = {
            glueRunConfiguration: {
                autoImportDataQualityResult: options.autoImportDataQualityResult ?? true,
                dataAccessRole: options.dataAccessRole?.roleArn,
                relationalFilterConfigurations: options.filterConfigurations.map((filter) => {
                    return {
                        databaseName: filter.databaseName,
                        filterExpressions: filter.filterExpressions?.map(expression => ({
                            expression: expression.expression,
                            type: expression.filterType ?? 'INCLUDE',
                        })),
                        schemaName: filter.schemaName,
                    };
                }),
            },
        };
        return new DataSourceConfiguration('GLUE', configuration);
    }
    static fromRedshift(options) {
        const configuration = {
            redshiftRunConfiguration: {
                redshiftStorage: 'CLUSTER' === options.redshiftType ? {
                    redshiftClusterSource: {
                        clusterName: options.name,
                    },
                } : {
                    redshiftServerlessSource: {
                        workgroupName: options.name,
                    },
                },
                redshiftCredentialConfiguration: {
                    secretManagerArn: options.credentials.secretArn,
                },
                dataAccessRole: options.dataAccessRole?.roleArn,
                relationalFilterConfigurations: options.filterConfigurations.map((filter) => {
                    return {
                        databaseName: filter.databaseName,
                        filterExpressions: filter.filterExpressions?.map(expression => ({
                            expression: expression.expression,
                            type: expression.filterType ?? 'INCLUDE',
                        })),
                        schemaName: filter.schemaName,
                    };
                }),
            },
        };
        return new DataSourceConfiguration('REDSHIFT', configuration);
    }
}
exports.DataSourceConfigurationBase = DataSourceConfigurationBase;
_a = JSII_RTTI_SYMBOL_1;
DataSourceConfigurationBase[_a] = { fqn: "cdk-data-zone.DataSourceConfigurationBase", version: "0.0.0" };
class DataSourceConfiguration {
    constructor(filterType, configuration) {
        this.filterType = filterType;
        this.configuration = configuration;
    }
}
class DataSourceBase extends resource_1.ResourceBase {
}
exports.DataSourceBase = DataSourceBase;
_b = JSII_RTTI_SYMBOL_1;
DataSourceBase[_b] = { fqn: "cdk-data-zone.DataSourceBase", version: "0.0.0" };
class DataSource extends DataSourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        // check if domain name is 64 characters or less
        if (!cdk.Token.isUnresolved(props.name) && props.name.length > 64) {
            throw new Error('Project name must be 64 characters or less');
        }
        this.description = props.description;
        this.name = props.name;
        this.filterType = props.configuration.filterType;
        this.enabled = props.enabled ?? true;
        this.publishOnImport = props.publishOnImport ?? true;
        this.recommendation = props.recommendation ?? true;
        this.environment = props.environment;
        const resource = new datazone.CfnDataSource(this, 'Resource', {
            domainIdentifier: props.project.projectDomainId,
            projectIdentifier: props.project.projectId,
            environmentIdentifier: props.environment.environmentId,
            name: props.name,
            description: props.description,
            enableSetting: this.enabled ? 'ENABLED' : 'DISABLED',
            publishOnImport: this.publishOnImport,
            recommendation: this.recommendation ? { enableBusinessNameGeneration: true } : undefined,
            schedule: props.schedule ? { schedule: props.schedule.expression, timezone: props.schedule.timezone } : undefined,
            type: props.configuration.filterType,
            configuration: props.configuration.configuration,
        });
        this.dataSourceId = resource.attrId;
        this.createdAt = resource.attrCreatedAt;
        this.updatedAt = resource.attrUpdatedAt;
        this.lastRunAssetCount = resource.attrLastRunAssetCount;
        this.lastRunAt = resource.attrLastRunAt;
        this.lastRunStatus = resource.attrLastRunStatus;
        this.status = resource.attrStatus;
        this.project = props.project;
        Object.defineProperty(this, IS_DATA_SOURCE, { value: true });
    }
}
exports.DataSource = DataSource;
_c = JSII_RTTI_SYMBOL_1;
DataSource[_c] = { fqn: "cdk-data-zone.DataSource", version: "0.0.0" };
//# sourceMappingURL=data:application/json;base64,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