"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lib_1 = require("../lib");
const aws_auth_1 = require("../lib/aws-auth");
const util_1 = require("./util");
module.exports = {
    'empty aws-auth'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'cluster');
        // WHEN
        new aws_auth_1.AwsAuth(stack, 'AwsAuth', { cluster });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesResource.RESOURCE_TYPE, {
            Manifest: JSON.stringify([{
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: { name: 'aws-auth', namespace: 'kube-system' },
                    data: { mapRoles: '[]', mapUsers: '[]', mapAccounts: '[]' }
                }])
        }));
        test.done();
    },
    'addRoleMapping and addUserMapping can be used to define the aws-auth ConfigMap'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'Cluster');
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        const user = new iam.User(stack, 'user');
        // WHEN
        cluster.awsAuth.addRoleMapping(role, { groups: ['role-group1'], username: 'roleuser' });
        cluster.awsAuth.addRoleMapping(role, { groups: ['role-group2', 'role-group3'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['user-group1', 'user-group2'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['user-group1', 'user-group2'], username: 'foo' });
        cluster.awsAuth.addAccount('112233');
        cluster.awsAuth.addAccount('5566776655');
        // THEN
        assert_1.expect(stack).to(assert_1.countResources(lib_1.KubernetesResource.RESOURCE_TYPE, 1));
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                "Fn::Join": [
                    "",
                    [
                        "[{\"apiVersion\":\"v1\",\"kind\":\"ConfigMap\",\"metadata\":{\"name\":\"aws-auth\",\"namespace\":\"kube-system\"},\"data\":{\"mapRoles\":\"[{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "ClusterDefaultCapacityInstanceRole3E209969",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"system:node:{{EC2PrivateDNSName}}\\\",\\\"groups\\\":[\\\"system:bootstrappers\\\",\\\"system:nodes\\\"]},{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "roleC7B7E775",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"roleuser\\\",\\\"groups\\\":[\\\"role-group1\\\"]},{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "roleC7B7E775",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "roleC7B7E775",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"groups\\\":[\\\"role-group2\\\",\\\"role-group3\\\"]}]\",\"mapUsers\":\"[{\\\"userarn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "user2C2B57AE",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "user2C2B57AE",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"groups\\\":[\\\"user-group1\\\",\\\"user-group2\\\"]},{\\\"userarn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "user2C2B57AE",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"foo\\\",\\\"groups\\\":[\\\"user-group1\\\",\\\"user-group2\\\"]}]\",\"mapAccounts\":\"[\\\"112233\\\",\\\"5566776655\\\"]\"}}]"
                    ]
                ]
            }
        }));
        test.done();
    },
    'imported users and roles can be also be used'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'Cluster');
        const role = iam.Role.fromRoleArn(stack, 'imported-role', 'arn:aws:iam::123456789012:role/S3Access');
        const user = iam.User.fromUserName(stack, 'import-user', 'MyUserName');
        // WHEN
        cluster.awsAuth.addRoleMapping(role, { groups: ['group1'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['group2'] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                "Fn::Join": [
                    "",
                    [
                        "[{\"apiVersion\":\"v1\",\"kind\":\"ConfigMap\",\"metadata\":{\"name\":\"aws-auth\",\"namespace\":\"kube-system\"},\"data\":{\"mapRoles\":\"[{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "ClusterDefaultCapacityInstanceRole3E209969",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"system:node:{{EC2PrivateDNSName}}\\\",\\\"groups\\\":[\\\"system:bootstrappers\\\",\\\"system:nodes\\\"]},{\\\"rolearn\\\":\\\"arn:aws:iam::123456789012:role/S3Access\\\",\\\"username\\\":\\\"arn:aws:iam::123456789012:role/S3Access\\\",\\\"groups\\\":[\\\"group1\\\"]}]\",\"mapUsers\":\"[{\\\"userarn\\\":\\\"arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::",
                        {
                            Ref: "AWS::AccountId"
                        },
                        ":user/MyUserName\\\",\\\"username\\\":\\\"arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::",
                        {
                            Ref: "AWS::AccountId"
                        },
                        ":user/MyUserName\\\",\\\"groups\\\":[\\\"group2\\\"]}]\",\"mapAccounts\":\"[]\"}}]"
                    ]
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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