"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_patch_1 = require("./k8s-patch");
const k8s_resource_1 = require("./k8s-resource");
const spot_interrupt_handler_1 = require("./spot-interrupt-handler");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSServicePolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds
            }
        };
        let resource;
        this.kubectlEnabled = props.kubectlEnabled === undefined ? true : props.kubectlEnabled;
        if (this.kubectlEnabled) {
            resource = new cluster_resource_1.ClusterResource(this, 'Resource', clusterProps);
            this._clusterResource = resource;
        }
        else {
            resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        }
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // map the IAM role to the `system:masters` group.
        if (props.mastersRole) {
            if (!this.kubectlEnabled) {
                throw new Error(`Cannot specify a "masters" role if kubectl is disabled`);
            }
            this.awsAuth.addMastersRole(props.mastersRole);
            if (props.outputMastersRoleArn) {
                new core_1.CfnOutput(this, 'MastersRoleArn', { value: props.mastersRole.roleArn });
            }
            commonCommandOptions.push(`--role-arn ${props.mastersRole.roleArn}`);
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = this.addCapacity('DefaultCapacity', { instanceType, minCapacity });
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        if (this.kubectlEnabled) {
            this.defineCoreDnsComputeType(props.coreDnsComputeType || CoreDnsComputeType.EC2);
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: new EksOptimizedImage({
                nodeType: nodeTypeForInstanceType(options.instanceType),
                kubernetesVersion: this.version,
            }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled
        });
        return asg;
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error(`Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false`);
        }
        if (bootstrapEnabled) {
            const userData = user_data_1.renderUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tag.add(autoScalingGroup, `kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true
        });
        if (options.mapRole === true && !this.kubectlEnabled) {
            throw new Error(`Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster`);
        }
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole && this.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes'
                ]
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && this.kubectlEnabled) {
            this.addResource('spot-interrupt-handler', ...spot_interrupt_handler_1.spotInterruptHandler());
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this.kubectlEnabled) {
            throw new Error(`Cannot define aws-auth mappings if kubectl is disabled`);
        }
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    _getKubectlCreationRoleArn(assumedBy) {
        if (!this._clusterResource) {
            throw new Error(`Unable to perform this operation since kubectl is not enabled for this cluster`);
        }
        return this._clusterResource.getCreationRoleArn(assumedBy);
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                subnet.node.applyAspect(new core_1.Tag(tag, "1"));
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, "kubernetes.io/role/internal-elb");
        tagAllSubnets('public', this.vpc.publicSubnets, "kubernetes.io/role/elb");
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        if (!this.kubectlEnabled) {
            throw new Error(`kubectl must be enabled in order to define the compute type for CoreDNS`);
        }
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType
                        }
                    }
                }
            }
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2)
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, "VPC", props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.nodeType = props && props.nodeType;
        this.kubernetesVersion = props && props.kubernetesVersion || LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? "amazon-linux-2/" : "")
            + (this.nodeType === NodeType.GPU ? "amazon-linux2-gpu/" : "")
            + "recommended/image_id";
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU : NodeType.STANDARD;
}
exports.nodeTypeForInstanceType = nodeTypeForInstanceType;
//# sourceMappingURL=data:application/json;base64,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