"""Модуль клиента"""
import logging
import os

import sys
import argparse
# from Crypto.PublicKey import RSA
from Cryptodome.PublicKey import RSA


from PyQt5.QtWidgets import QApplication, QMessageBox
import logs.config_client_log
from common.variables import *
from common.errors import ServerError
from common.decos import log
from client.database import ClientDatabase
from client.transport import ClientTransport
from client.main_window import ClientMainWindow
from client.start_dialog import UserNameDialog

# Инициализация клиентского логера
logger = logging.getLogger('client')



@log
def arg_parser() -> tuple:
    """
    Парсер аргументов коммандной строки
    :return: кортеж server_address, server_port, client_name, client_passwd
    """
    parser = argparse.ArgumentParser()
    parser.add_argument('addr', default=DEFAULT_IP_ADDRESS, nargs='?')
    parser.add_argument('port', default=DEFAULT_PORT, type=int, nargs='?')
    parser.add_argument('-n', '--name', default=None, nargs='?')
    parser.add_argument('-p', '--password', default='', nargs='?')
    namespace = parser.parse_args(sys.argv[1:])
    server_address = namespace.addr
    server_port = namespace.port
    client_name = namespace.name
    client_passwd = namespace.password

    # проверим подходящий номер порта
    if not 1023 < server_port < 65536:
        logger.critical(
            f'Попытка запуска клиента с неподходящим номером порта: {server_port}. '
            f'Допустимы адреса с 1024 до 65535. Клиент завершается.')
        sys.exit(1)

    return server_address, server_port, client_name, client_passwd


# Основная функция клиента
if __name__ == '__main__':
    # Загружаем параметы коммандной строки
    server_address, server_port, client_name, client_passwd = arg_parser()

    # Создаём клиентокое приложение
    client_app = QApplication(sys.argv)

    # Запрашиваем имя пользователя если не было указано
    start_dialog = UserNameDialog()

    # Если имя пользователя не было указано в командной строке то запросим его
    if not client_name or not client_passwd:

        client_app.exec_()
        # Если пользователь ввёл имя и нажал ОК, то сохраняем ведённое и
        # удаляем объект, иначе выходим
        if start_dialog.ok_pressed:
            client_name = start_dialog.client_name.text()
            client_passwd = start_dialog.client_passwd.text()
            logger.debug(
                f" USERNAME = {client_name} PASSWORD = {client_passwd}")
            # del start_dialog
        else:
            sys.exit(0)

    # Записываем логи
    logger.info(
        f'Запущен клиент с парамертами: адрес сервера: {server_address} , '
        f'порт: {server_port}, имя пользователя: {client_name}')

# загружаем ключи с файла, если же файла нет, то генерируем новую пару
    dir_path = os.getcwd()
    key_file = os.path.join(dir_path, f"{client_name}.key")
    if not os.path.exists(key_file):
        keys = RSA.generate(2048, os.urandom)
        with open(key_file, 'wb') as key:
            key.write(keys.export_key())
    else:
        with open(key_file, 'rb') as key:
            keys = RSA.import_key(key.read())

    logger.debug('Keys sucsessfully loaded')

    database = ClientDatabase(client_name)

    # Создаем объект - транспорт и запускаем транспортный поток
    try:
        transport = ClientTransport(
            server_port,
            server_address,
            database,
            client_name,
            client_passwd,
            keys
        )
    except ServerError as e:
        message = QMessageBox()
        message.critical(start_dialog, "Ошибка сервера", e.text)
        sys.exit(1)
    transport.setDaemon(True)
    transport.start()

    # удалим объект диалога
    del start_dialog

    # Создаём GUI
    main_window = ClientMainWindow(database, transport, keys)
    main_window.make_connection(transport)
    main_window.setWindowTitle(f'Чат Программа alpha release - {client_name}')
    client_app.exec_()

    # Раз графическая оболочка закрылась, закрываем транспорт
    transport.transport_shutdown()
    transport.join()
