"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CpuMonitor = void 0;
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
// Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html#metrics-cpu-utilization
class CpuMonitor {
    static setup(scope, cacheClusterId, nodeType) {
        let threshold = 90;
        let metricName = 'EngineCPUUtilization';
        if (nodeType && nodeType.vcupCount <= 2) {
            metricName = 'CPUUtilization';
            threshold = (90 / nodeType.vcupCount);
        }
        return new cloudwatch.Alarm(scope, 'alarm-elastc-cache-' + cacheClusterId + '-' + metricName, {
            alarmName: `ElasticCacheCPULoadAlarm[${cacheClusterId}]`,
            metric: new cloudwatch.Metric({
                namespace: 'AWS/ElastiCache',
                metricName,
                dimensions: { CacheClusterId: cacheClusterId },
                statistic: 'avg',
            }),
            threshold,
            period: cdk.Duration.minutes(1),
            evaluationPeriods: 1,
            alarmDescription: `The average CPU load(${metricName}) is greater than ${threshold}%.`,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: cloudwatch.TreatMissingData.BREACHING,
        });
    }
}
exports.CpuMonitor = CpuMonitor;
//# sourceMappingURL=data:application/json;base64,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