# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/03_io.ipynb (unless otherwise specified).

__all__ = ['PDSReader', 'UVPDS', 'UVISObs', 'UVIS_NetCDF', 'HSP', 'FUV_CDF', 'FUV', 'EUV_CDF']

# Cell
import datetime as dt
import warnings
from pathlib import Path
from typing import Union

import matplotlib.pyplot as plt
import numpy as np
import seaborn as sns
from astropy import units as u
from fastcore.utils import dict2obj

import holoviews as hv
import hvplot.xarray
import pandas as pd
import pvl
import xarray as xr
from planetarypy.uvis import get_data_path, get_label_path, get_user_guide
from .hsp_sensitivity import sens_df

try:
    import ffmpy

    _FFMPY_INSTALLED = True
except ImportError:
    _FFMPY_INSTALLED = False

xr.set_options(keep_attrs=True)
hv.config.image_rtol = 0.05

# Cell
class PDSReader:
    """Class to support the correct binary reading of the PDS DAT files.

    This class focuses on reading the DAT files.
    For a more complete reader, look below at the `UVPDS`
    """

    def __init__(
        self,
        datapath: Union[str, Path],  # full path to data
    ):
        self.datapath = Path(datapath)

        label = dict2obj(pvl.load(datapath.with_suffix(".LBL")))

        self.band_range = [
            label.QUBE.UL_CORNER_BAND,
            label.QUBE.LR_CORNER_BAND + 1,  # for numpy slicing + 1
        ]
        self.band_range[1] = 1024 // label.QUBE.BAND_BIN

        self.line_range = [
            label.QUBE.UL_CORNER_LINE,
            label.QUBE.LR_CORNER_LINE + 1,  # for numpy slicing + 1
        ]

        # read the data
        # Both the data file and the calibration files contain big-endian values.
        if label.QUBE.CORE_ITEM_TYPE == "MSB_UNSIGNED_INTEGER":
            data1D = (np.fromfile(str(datapath), ">H")).astype(np.uint16)
        elif label.QUBE.CORE_ITEM_TYPE == "IEEE_REAL":
            data1D = (np.fromfile(str(datapath), ">f4")).astype(np.float32)
        else:
            raise TypeError("Unhandled QUBE data type.  Not yet implemented.")

        # reshape the data and then slice out the relevant parts
        data = data1D.reshape(label.QUBE.CORE_ITEMS, order="F")[
            slice(*self.band_range), slice(*self.line_range), :
        ]

        # store variables
        self.label = label
        self.data = data

# Cell
class UVPDS:
    """
    Class to manage a PDS product.

    One PDS product has a PRODUCT_ID like FUV2003_358_05_59 and hence provides either
    EUV or FUV data.

    If the calibration matrix PDS file is available, the wavelengths are being read from
    the calmatrix label file, correctly binned for BAND_BIN, if required, otherwise
    default wavelengths are being assumed, as given per UVIS manual.

    See `UVISOBS` for a class that manages all data related to one observation, i.e. both
    EUV and FUV, and also HDAC and HSP data.
    """

    def __init__(
        self,
        # `uvis_id` can be either the PDS product_id 'FUV2005_172_03_35' or the long form
        # used within the UVIS product.
        # The attribute `pid` will carry the shortenend PDS identifier,
        # attribute `uvis_id` will just store what the user came in with.
        uvis_id: str,
    ):
        self.uvis_id = uvis_id
        self.product_id = self.pid = uvis_id[:17]
        self.path = get_data_path(self.pid)
        self.pds = PDSReader(self.path)
        self.datalabel = self.pds.label
        self.cal_data = None

        # Try to load the calibration Matrix file.  If not present, set to None.
        if self.cal_label_path.exists() and self.cal_data_path.exists():
            self.cal_data = PDSReader(self.cal_data_path)
            self.cal_matrix = self.cal_data.data
            self.caliblabel = dict2obj(pvl.load(str(self.cal_label_path)))
            self.set_cal_wavelengths()
        else:
            self.set_default_wavelengths()

    @property
    def file_id(self) -> str:  # returns the stem of the received file path
        return self.path.stem

    @property
    def label_path(self) -> Path:
        return self.path.with_suffix(".LBL")

    @property
    def data_path(self) -> Path:
        return self.path.with_suffix(".DAT")

    @property
    def cal_label_path(self):
        return self.path.parent / (self.file_id + "_CAL_3.LBL")

    @property
    def cal_data_path(self):
        return self.path.parent / (self.file_id + "_CAL_3.DAT")

    @property
    def n_bands(self):
        return self.datalabel.QUBE.CORE_ITEMS[0] // self.datalabel.QUBE.BAND_BIN

    @property
    def n_integrations(self):
        "last item of the CORE_ITEMS label field is number of integrations."
        return self.datalabel.QUBE.CORE_ITEMS[-1]

    @property
    def data(self):
        return self.pds.data

    @property
    def shape(self):
        return self.data.shape

    @property
    def line_range(self):
        return self.pds.line_range

    @property
    def detector(self):
        return self.pid[:3]

    @property
    def integration_duration(self):
        return self.datalabel.INTEGRATION_DURATION

    def create_xarray(self, data):
        arr = xr.DataArray(data)
        arr = arr.rename({"dim_0": "spectral", "dim_1": "spatial", "dim_2": "samples"})
        arr = arr.assign_coords(
            {
                "spectral": self.wavelengths,
                "spatial": np.arange(*self.line_range),
                "samples": np.arange(self.n_integrations) + 1,
            }
        )
        arr.attrs["units"] = "Counts"
        arr.attrs["long_name"] = f"{self.detector} raw data"
        arr.spectral.attrs["units"] = "nm"
        arr.spectral.attrs["long_name"] = "Wavelength"
        arr.spatial.attrs["long_name"] = "Spatial lines"
        arr.attrs["n_bands"] = self.n_bands
        arr.attrs["integration_duration"] = self.integration_duration
        arr.name = self.product_id
        return arr

    @property
    def xarray(self):
        return self.create_xarray(self.data)

    @property
    def calibrated(self):
        "Following the instructions in self.caliblabel.QUBE.DESCRIPTION"
        caldata = self.cal_data.data
        caldata[caldata == -1] = np.nan
        calibrated = (
            self.data * caldata * self.caliblabel.QUBE.CORE_MULTIPLIER
        )
        arr = self.create_xarray(calibrated)
        arr.attrs["units"] = "kiloRayleighs"
        arr.attrs["long_name"] = f"{self.detector} calibrated data"
        return arr

    @property
    def default_wave_min(self):
        if self.pid.startswith("EUV"):
            return 111.5 * u.nm
        elif self.pid.startswith("FUV"):
            return 56.12 * u.nm
        else:
            raise NotImplementedError(pid)

    @property
    def default_wave_max(self):
        if self.pid.startswith("EUV"):
            return 190.0 * u.nm
        elif self.pid.startswith("FUV"):
            return 118.1 * u.nm
        else:
            return NotImplementedError(self.PRODUCT_ID)

    def set_cal_wavelengths(self):
        """
        Retrieve the wavelengths for this observation, in nm, using cal_matrix

        Sets `wavelengths` to
            np.array of np.double values in nanometers (nm)
            The wavelengths for each spectral bin of the QUBE observation
        """

        qubelabel = self.cal_data.label.QUBE
        BAND_BIN = qubelabel.BAND_BIN

        # The Cal matrix LBL file BAND_BIN_CENTER field does not take into account
        # spectral binning. Therefore, we must take that into account here.
        wavelengths = np.array(qubelabel.BAND_BIN_CENTER, dtype=np.double)
        if BAND_BIN > 1:
            binned_wavelengths = np.zeros((self.n_bands,), dtype=np.double)
            for iwave in range(0, wavelengths.size, BAND_BIN):
                binned_wavelengths[iwave // BAND_BIN] = np.mean(
                    wavelengths[iwave : iwave + BAND_BIN]
                )
            wavelengths = binned_wavelengths / 10.0  # Convert from Angstroms to nm
        else:
            wavelengths = wavelengths / 10.0  # to nm

        self.wavelengths = wavelengths * u.nm

    def set_default_wavelengths(self):
        "If calibration matrix is not present, set to default wavelengths."
        # TODO: Check if this is always correct, using self.shape[0] ?
        wavelengths = np.linspace(
            self.default_wave_min, self.default_wave_max, self.shape[0]
        )
        self.wavelengths = wavelengths * u.nm

    def plot(
        self,
        precise: bool = False,  # switch to choose more precise quadmesh plot
        percentiles: tuple = (
            0.5,
            99.5,
        ),  # percentiles to be applied as minimum/maximum stretch
        clim: tuple = None,  # Set the visual stretch manually instead of via percentiles
        cmap: str = "viridis",  # default colormap. Other nice ones are 'plasma' or 'magma'
        calibrated = False,  # switch to control if to plot raw or calibrated data
    ):
        """Create default hvplot for the data.

        Due to non-equidistant wavelengths, one should use the quadmesh plot,
        but that is less performant than a constant raster and creates an annoying
        aliasing structure when zoomed out (however correct, though).

        I am investigating if that aliasing can be avoided, it might come from gridlines.
        So I leave it to the user to switch to the raster plot using the `precise` switch."""
        data = self.xarray if not calibrated else self.calibrated
        # define good stretch by using percentiles:
        stretch = (
            tuple(np.percentile(data, percentiles)) if clim is None else clim
        )
        # use `frameswise=False` to allow zooming in survice scrolling over samples
        kwargs = dict(
            x="spectral",
            y="spatial",
            framewise=False,
            cmap=cmap,
            clim=stretch,
            clabel=data.attrs['units'],
            title=data.attrs['long_name'],
        )
        if precise:
            kwargs["kind"] = "quadmesh"
        return data.hvplot(**kwargs).opts(gridstyle={"grid_line_color": "white"}).opts(axiswise=True)

# Cell
class UVISObs:
    def __init__(self, pid_or_timestring):
        "Timestring like 2005_172_03_35 will find both EUV and FUV."
        self.timestring = pid_or_timestring[-14:]

    @property
    def FUV_PID(self):
        return "FUV" + self.timestring

    @property
    def EUV_PID(self):
        return "EUV" + self.timestring

    @property
    def euv(self):
        return UVPDS(self.EUV_PID)

    @property
    def fuv(self):
        return UVPDS(self.FUV_PID)

# Cell
class UVIS_NetCDF:
    def __init__(self, fname, freq):
        self.path = Path(fname)
        self.fname = str(self.path)
        self.ds = xr.open_dataset(self.fname)
        self.freq = freq
        self.timestr = self.ds.start_time_str[:20] + "000"
        self.orig_dims = list(self.ds.dims.keys())

    @property
    def start_time(self):
        timestr = self.timestr
        try:
            fmt = "%Y-%j %H:%M:%S.%f"
            return dt.datetime.strptime(timestr, fmt)
        except ValueError:
            fmt = "%Y-%jT%H:%M:%S.%f"
            return dt.datetime.strptime(timestr, fmt)

    @property
    def times(self):
        times = pd.date_range(
            self.start_time, periods=self.n_integrations, freq=self.freq
        )
        return times

    @property
    def spatial_dim(self):
        for dim in self.orig_dims:
            if dim.startswith("spatial_dim"):
                return dim

    @property
    def spectral_dim(self):
        for dim in self.orig_dims:
            if dim.startswith("spectral_dim"):
                return dim

# Cell
class HSP(UVIS_NetCDF):

    """Class for reading NetCDF UVIS HSP data files.

    Parameters
    ==========
    fname: {str, pathlib.Path}
        Path to file to read
    freq: str
        String indicating the sampling frequency, e.g. '1ms', '2ms'

    Examples
    ========
    hsp = hsp('path', '1ms')

    """

    sensitivity = sens_df

    def __init__(self, fname, freq):
        super().__init__(fname, freq)

    @property
    def n_integrations(self):
        return self.ds.counts.size

    @property
    def times(self):
        """HSP times are different in that one needs to stack all rows."""
        return pd.date_range(
            self.start_time, periods=self.n_integrations, freq=self.freq
        )

    @property
    def series(self):
        s = pd.Series(self.ds.counts.values.ravel())
        s.index = self.times
        return s

    @property
    def counts_per_sec(self):
        ind = self.series.index
        td = ind[1] - ind[0]
        return self.series / td.total_seconds()

    def get_last_minutes(self, min):
        raise NotImplementedError("get_last_minutes currently not implemented")
        # ind = self.series.index
        # return self.series[ind[-1] - datetools.Minute(min) :]

    def get_first_minutes(self, min):
        raise NotImplementedError("get_first_minutes currently not implemented.")
        # ind = self.series.index
        # return self.series[: ind[0] + pd.Minute(min)]

    @property
    def resampled(self):
        return self.counts_per_sec.resample("1s")

    @property
    def cleaned_data_copy(self):
        """Filtering out 0.5, 99.5 % outliers."""
        data = self.counts_per_sec.copy()
        min, max = np.percentile(data, (0.5, 99.5))
        data[data < min] = np.nan
        data[data > max] = np.nan
        return data

    def plot_resampled_with_errors(self, binning="1s", ax=None):
        data = self.cleaned_data_copy
        resampled = data.resample(binning)
        mean = resampled.mean()
        std = resampled.std()
        if ax is None:
            fig, ax = plt.subplots()
        mean.plot(yerr=std, ax=ax)
        ax.set_xlabel("Time")
        ax.set_ylabel("Counts per second")
        ax.set_title(f"Resampled to {binning} s")

    def plot_relative_std(self, binning="1s", ax=None):
        data = self.cleaned_data_copy
        resampled = data.resample(binning)
        mean = resampled.mean()
        std = resampled.std()
        if ax is None:
            fig, ax = plt.subplots()
        (std / mean).plot(ax=ax)
        ax.set_xlabel("Time")
        ax.set_ylabel("Relative error per resample bin.")
        ax.set_title("Ratio of STD over mean value of resample bin.")

    def save_as_csv(self):
        to_save = self.resampled.mean()
        tdindex = to_save.index - to_save.index[0]
        to_save.index = tdindex.seconds
        to_save.to_csv(str(self.fname.with_suffix(".csv")))

    def __repr__(self):
        return self.ds.__repr__()

# Cell
class FUV_CDF(UVIS_NetCDF):

    """FUV NetCDF reader class.

    Parameters
    ==========
    fname: str or pathlib.Path
        Path to file to read
    freq: str
        String indicating the sampling frequency, e.g. '1s', '2s'

    Examples
    ========
    fuv = FUV_CDF('path', '1s')
    """

    wave_min = 111.5  # nm
    wave_max = 190.0  # nm

    def __init__(self, fname, freq="1s"):
        super().__init__(fname, freq)
        self.n_integrations = self.ds["integrations"].size
        self.n_spec_bins = self.ds[self.spectral_dim].size
        self.waves = np.linspace(self.wave_min, self.wave_max, self.n_spec_bins)
        self.ds["times"] = xr.DataArray(self.times.values, dims="integrations")
        self.ds["wavelengths"] = xr.DataArray(self.waves, dims=self.spectral_dim)
        self.ds = self.ds.swap_dims(
            {"integrations": "times", self.spectral_dim: "wavelengths"}
        )
        self.ds = self.ds.rename({"window_0": "counts", self.spatial_dim: "pixels"})

    @property
    def data(self):
        return self.ds.counts

    @property
    def plotfolder(self):
        f = self.path.parent / "plots"
        f.mkdir(exist_ok=True)
        return f

    def save_spectograms(self):
        sns.set_context("talk")
        vmax = self.data.max() * 1.05
        for i, spec in enumerate(self.data):
            fig, ax = plt.subplots()
            spec.plot(ax=ax, vmax=vmax)
            fig.tight_layout()
            fname = "spectogram_{}.png".format(str(i).zfill(4))
            savepath = str(self.plotfolder / fname)
            fig.savefig(savepath, dpi=150)
            plt.close(fig)
        print("Saved spectrograms in", self.plotfolder)

    def _run_ffmpy(self, inputs, output_name):
        opts = "-framerate 3 -y"
        output_options = "-c:v libx264 -pix_fmt yuv420p"
        outputs = {str(self.plotfolder / output_name): output_options}
        ff = ffmpy.FF(global_options=opts, inputs=inputs, outputs=outputs)
        print("Running", ff.cmd_str)
        ff.run()

    def create_spectogram_movie(self):
        if not _FFMPY_INSTALLED:
            print("ffmpy is not installed: 'pip install ffmpy'.")
            return
        inputs = {str(self.plotfolder / "spectogram_%04d.png"): None}
        self._run_ffmpy(inputs, "spectograms.mp4")

    def save_spectrums(self):
        "plotting spectrums over time summing all pixels"
        sns.set_context("talk")
        vmax = self.data.sum("pixels").max() * 1.05
        for i, spec in enumerate(self.data.sum("pixels")):
            fig, ax = plt.subplots()
            spec.plot()
            ax.set_ylim(0, vmax)
            fig.tight_layout()
            fname = "summed_spectrum_{}.png".format(str(i).zfill(4))
            savepath = str(self.plotfolder / fname)
            fig.savefig(savepath, dpi=150)
            plt.close(fig)
        print("Saved spectrums in", self.plotfolder)

    def create_spec_time_sequence_movie(self):
        if not _FFMPY_INSTALLED:
            print("ffmpy is not installed: 'pip install ffmpy'.")
            return
        inputs = {str(self.plotfolder / "summed_spectrum_%04d.png"): None}
        self._run_ffmpy(inputs, "summed_spectrums.mp4")

    def __repr__(self):
        return self.ds.__repr__()


FUV = FUV_CDF  # alias to prevent breaking old API

# Cell
class EUV_CDF(UVIS_NetCDF):
    "EUV NetCDF reader class."
    wave_min = 56.12  # nm
    wave_max = 118.1  # nm

    def __init__(
        self,
        fname: Union[str, Path],  # Path to file to be read
        freq: str = "1s",  # String indicating the sampling frequency
    ):

        super().__init__(fname, freq)
        self.n_integrations = self.ds["integrations"].size
        self.n_spec_bins = self.ds[self.spectral_dim].size
        self.waves = np.linspace(self.wave_min, self.wave_max, self.n_spec_bins)
        self.ds["times"] = xr.DataArray(self.times.values, dims="integrations")
        self.ds["wavelengths"] = xr.DataArray(self.waves, dims=self.spectral_dim)
        self.ds = self.ds.swap_dims(
            {"integrations": "times", self.spectral_dim: "wavelengths"}
        )
        self.ds = self.ds.rename({"window_0": "counts", self.spatial_dim: "pixels"})

    @property
    def data(self):
        return self.ds.counts

    @property
    def plotfolder(self):
        f = self.path.parent / "plots"
        f.mkdir(exist_ok=True)
        return f

    def save_spectograms(self):
        sns.set_context("talk")
        vmax = self.data.max() * 1.05
        for i, spec in enumerate(self.data):
            fig, ax = plt.subplots()
            spec.plot(ax=ax, vmax=vmax)
            fig.tight_layout()
            fname = "spectogram_{}.png".format(str(i).zfill(4))
            savepath = str(self.plotfolder / fname)
            fig.savefig(savepath, dpi=150)
            plt.close(fig)
        print("Saved spectrograms in", self.plotfolder)

    def _run_ffmpy(self, inputs, output_name):
        opts = "-framerate 3 -y"
        output_options = "-c:v libx264 -pix_fmt yuv420p"
        outputs = {str(self.plotfolder / output_name): output_options}
        ff = ffmpy.FF(global_options=opts, inputs=inputs, outputs=outputs)
        print("Running", ff.cmd_str)
        ff.run()

    def create_spectogram_movie(self):
        if not _FFMPY_INSTALLED:
            print("ffmpy is not installed: 'pip install ffmpy'.")
            return
        inputs = {str(self.plotfolder / "spectogram_%04d.png"): None}
        self._run_ffmpy(inputs, "spectograms.mp4")

    def save_spectrums(self):
        "plotting spectrums over time summing all pixels"
        sns.set_context("talk")
        vmax = self.data.sum("pixels").max() * 1.05
        for i, spec in enumerate(self.data.sum("pixels")):
            fig, ax = plt.subplots()
            spec.plot()
            ax.set_ylim(0, vmax)
            fig.tight_layout()
            fname = "summed_spectrum_{}.png".format(str(i).zfill(4))
            savepath = str(self.plotfolder / fname)
            fig.savefig(savepath, dpi=150)
            plt.close(fig)
        print("Saved spectrums in", self.plotfolder)

    def create_spec_time_sequence_movie(self):
        if not _FFMPY_INSTALLED:
            print("ffmpy is not installed: 'pip install ffmpy'.")
            return
        inputs = {str(self.plotfolder / "summed_spectrum_%04d.png"): None}
        self._run_ffmpy(inputs, "summed_spectrums.mp4")

    def __repr__(self):
        return self.ds.__repr__()