"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
        });
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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