import unittest
from copy import copy

from .data import test_data
from dataset_creator.dataset import Dataset
from dataset_creator.creator import Creator


class TestCreator(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        self.test_data = copy(test_data)

    def tearDown(self):
        self.test_data = None

    def test_nexus_header(self):
        dataset = Dataset(self.test_data, format='NEXUS', partitioning='by gene')
        creator = Creator(dataset.data, format='NEXUS')
        expected = """
#NEXUS

BEGIN DATA;
DIMENSIONS NTAX=10 NCHAR=4739;
FORMAT INTERLEAVE DATATYPE=DNA MISSING=? GAP=-;
MATRIX
"""
        result = creator.dataset_header
        self.assertEqual(expected.strip(), result)

    def test_nexus_dataset_block(self):
        dataset = Dataset(self.test_data, format='NEXUS', partitioning='by gene')
        creator = Creator(dataset.data, format='NEXUS')
        expected = """
[ArgKin]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_11_Aus_bus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_12_Aus_cus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_13_Aus_dus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_14_Aus_eus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_15_Aus_fus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_16_Aus_gus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_17_Aus_hus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_18_Aus_ius                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_19_Aus_jus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????

[COI-begin]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ??????????????????????????????????????????????????????????ATTATTCGAACAGAATTAAGTACCCCTGGATCATTAATCGGAGATGATCAAATTTATAATACTATTGTTACAGCTCATGCTTTTATTATAATTTTTTTTATGGTTATACCTATTATAATTGGAGGATTTGGTAATTGACTTATTCCCCTTATATTAGGAGCCCCTGATATAGCTTTTCCACGAATAAATAATATAAGATTTTGACTTCTCCCACCCTCTTTAATTTTATTAATTTCGAGTAGTATAGTAGAAAATGGTGCTGGCACAGGATGAACGGTCTATCCCCCCCTCTCATCTAATATTGCCCATAGAGGATCCTCAGTTGATTTAGCAATCTTTTCCTTACATTTAGCTGGAATCTCATCAATTCTTGGAGCAATTAATTTTATTACAACAATTATTAATATACGAATTAATAAAATATCTTATGATCAAATACCTTTATTTGTTTGAGCTGTAGGAATTACCGCATTATTATTATTACTTTCTTTACCTGTATTAGCTGGAGCTATCACAATACTACTCACAGATCGAAACTTAAATACATCTTTTTTTGACCCAGCAGGAGGTGGAGATCCTATTTTATATCAACATTTATTTTGATTTTTTGG
CP100_11_Aus_bus                                                       AACATTATATTTTATTTTTGGAATTTGAGCAGGTATAGTAGGTACATCTCTCAGTTTAATTATTCGAACAGAATTAGGTAATCCAGGATCTTTAATTGGAGATGATCAAATTTATAATACTATTGTCACAGCCCATGCTTTTATTATAATTTTTTTTATAGTTATACCTATTATAATTGGAGGATTCGGTAATTGACTTATTCCTATTATATTAGGAGCCCCTGATATAGCTTTCCCTCGAATAAATAATATAAGATTTTGACTCCTTCCCCCCTCTTTAATTTTATTAATTTCAAGTAGTATTGTAGAAAATGGTGCTGGTACAGGATGAACGGTTTATCCCCCCCTTTCATCTAATATTGCTCACAGTGGATCTTCTGTTGATTTAGCAATTTTCTCTTTACATTTAGCTGGAATTTCATCAATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAGTATATCTTATGATCAAATACCATTATTTGTTTGAGCTGTTGGAATTACAGCATTATTATTATTACTCTCTCTACCTGTATTAGCTGGAGCTATTACAATACTTCTTACAGATcGAAATTTAAACaCcTcTT?TTTTgaTC????????????????????????????????????????????????????
CP100_12_Aus_cus                                                       AACTTTATATTTTATTTTTGGAATTTGAGCAGGTATAGTAGGTACATCTCTTAGTCTTATTATTCGAATAGAATTAGGTAACCCTAGATTTTTAATTGGAGATGATCAAATTTATAATACTATTGTTACAGCTCATGCTTTTATTATAATTTTTTTTATAGTAATACCTATTATAATTGGAGGATTTGGTAATTGACTTGTCCCCCTTATATTAGGAGCCCCTGATATAGCTTTCCCCCGAATAAATAATATAAGATTTTGACTCCTCCCCCCTTCTTTAGTTTTACTAATTTCAAGCAGTATTGTAGAAAATGGTGCTGGTACAGGATGAACGGTTTATCCCCCCCTTTCATCTAATATTGCCCATAGTGGATCTTCTGTAGATTTAGCAATTTTCTCTTTACATTTAGCTGGAATTTCCTCAATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAACATATCTTATGATCAAATACCCCTATTTGTTTGAGCTGTAGGTATTACTGCATTATTACTATTACTTTCCTTACCTGTATTAGCTGGAGCTATTACAATACTTCTTACAGATCGAAATTTAAATACTTCTTTTTTTGACCCTGCAGGAGGTGGTGATCCTATTTTATATCAACATTTATTTTGATTTTTTGG
CP100_13_Aus_dus                                                       AaCTTTATATTTTATTTTTGGAaTTTGAGCAGGTATAGTAGGTACTTCCCTTAGTCTTATTATCCGAACAGAATTAGGTAATCCTGGATCTTTAATTGGAGATGATCAAATTTATAATACTGTTGTCACAGCTCATGCTTTCATTATAATTTTTTTTATAGTTATACCTATTATAATTGGAGGATTTGGTAATTGACTTATCCCCCTTATACTAGGAGCTCCTGATATAGCTTTCCCCCGAATAAATAATATAAGATTTTGACTTCTACCTCCCTCTTTAACCTTACTAATTTCAAGTAGTATTGTAGAAAATGGAGCTGGAACTGGATGAACGGTCTATCCCCCCCTTTCATCCAATATTGCCCATAGTGGATCTTCAGTAGATTTAGCAATTTTCTCTTTACATTTAGCTGGAATTTCCTCAATCTTAGGAGCTATCAATTTTATTACAACAATCATTAATATACGAATTAATAATATATCTTATGATCAAATACCCTTATTTGTTTGAGCTGTAGGAATTACTGCATTATTATTATTACTTTCATTACCTGTATTAGCTGGAGCTATCACAATACTTCTTACAGATCGAAATTTAAATACTTCTTTTTTTGATCCTGCAGGAGGCGGAGATCCTATTTTATATCAACATTTATTTTGATTTTTTGG
CP100_14_Aus_eus                                                       aacATTATATTTTATTTTTGGAATTTGAGCAGGTATAGTAGGTACATCTCTCAGTTTAATTATTCGAACAGAATTAGGTAATCCAGGATCTTTAATTGGAGATGATCAAATTTATAATACTATTGTCACAGCCCATGCTTTTATTATAATTTTTTTTATAGTTATACCTATTATAATTGGAGGATTCGGTAATTGACTTATTCCTATTATATTAGGAGCCCCTGATATAGCTTTCCCTCGAATAAATAATATAAGATTTTGACTCCTTCCCCCCTCTTTAATTTTATTAATTTCAAGTAGTATTGTAGAAAATGGTGCTGGTACAGGATGAACGGTTTATCCCCCCCTTTCATCTAATATTGCTCACAGTGGATCTTCTGTTGATTTAGCAATTTTCTCTTTACATTTAGCTGGAATTTCATCAATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAGTATATCTTATGATCAAATACCATTATTTGTTTGAGCTGTTGGAATTACAGCATTATTATTATTACTCTCTCTACCTGTATTAGCTGGAGCTATTACAATACTTCTTACAGATCGAAATTTAAACACCTCTTTTTTTGATCCTGCAGGAGGAGGAGATCCTATTTTATACCAACATTTATTTTGATTTTTTGG
CP100_15_Aus_fus                                                       AACTTTATATTTTATTTTTGGAGTTTGAGCAGGAATAGTAGGTACATCCCTTAGTCTTATTATTCGAACAGAATTAGGCACCCCAGGATCTTTAATTGGAAATGATCAAATTTATAATACTATTGTTACAGCTCATGCTTTTATTATAATTTTTTTTATAGTTATACCCATTATAATTGGAGGATTTGGTAATTGACTTATTCCCCTCATACTAGGAGCTCCTGACATAGCTTTCCCCCGAATAAATAATATAAGATTTTGACTTCTTCCACCTTCTTTAACCCTACTAATTTTAAGTAGTATTGTAGAAAATGGAGCTGGCACAGGATGAACGGTTTATCCCCCCCTTTCATCTAATATTGCCCATAGTGGATCCTCTGTTGATTTAGCAATTTTTTCTTTACATTTAGCTGGAATTTCATCTATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAATATAAATTATGATCAAATACCCCTATTTGTCTGAGCTGTAGGAATTACAGCATTATTATTATTACTTTCCCTACCGGTATTAGCCGGAGCTATTACAATACTTCTTACAGATCGAAATTTAAATACTTCTTTTTTTGATCCTGCAGGAGGAGGAGATCCTATTTTATATCAACATTTATTTTGATTTTTTGG
CP100_16_Aus_gus                                                       ??????????????????????????????????????????????????ttAGTcTtaTTattcGAayaGAAtTAGGTAATCCTGGATTCTTAATTGGAGATGATCAAATTTATAATACTATTGTTACAGCTAATGCTTTTATTATAATTTTTTTCATAGTTATACCTATTATAATTGGAGGATTCGGTAATTGACTTGTTCCTCTTATACTAGGAGCCCCTGACATAGCTTTCCCCCGAATAAATAATATAAGATTTTGACTTCTACCCCCCTCTTTAATTCTATTAATTTCGAGTAGTATCGTAGAAAATGGAGCTGGCACAGGATGAACGGTTTATCCCCCCCTTTCATCCAATATTGCCCACAGCGGATCTTCTGTTGATTTAGTAATTTTCTCTTTACATTTAGCCGGAATTTCATCAATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAATATATCTTATGATCAAATACCTTTATTTGTTTGAGCTGTAGGAATTACTGCATTATTATTATTACTATCCTTACCTGTATTAGCTGGAGCTATTACAATACTTCTTACAGATCGAAATTTAAATACTTCTTTTTTTGATCCTGCAGGAGGTGGAGATCCTATCTTATACCAACATTTATTTTGATTTTTTGG
CP100_17_Aus_hus                                                       ?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_18_Aus_ius                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????AATTTATAATACTATTGTTACAGCTCATGCTTTTATTATAATTTTTTTCATAgTTATACCTATTATAATTGGAGGATTTGGTAATTGACTTaTTCCCCTTaTATTAT?ATCCCCCGTTAT??CTTTCCCCCGA????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_19_Aus_jus                                                       AaCATTATATTTTATTTTTGGAATTTGAGCAGGTATAGTAGGTACATCTCTCAGTTTAATTATTCGAACAGAATTAGGTAATCCAGGATCTTTAATTGGAGATGATCAAATTTATAATACTATTGTCACAGCCCATGCTTTTATTATAATTTTTTTTATAGTTATACCTATTATAATTGGAGGATTCGGTAATTGACTTATTCCTATTATATTAGGAGCCCCTGATATAGCTTTCCCTCGAATAAATAATATAAGATTTTGACTCCTTCCCCCCTCTTTAATTTTATTAATTTCAAGTAGTATTGTAGAAAATGGTGCTGGTACAGGATGAACGGTTTATCCCCCCCTTTCATCTAATATTGCTCACAGTGGATCTTCTGTTGATTTAGCAATTTTCTCTTTACATTTAGCTGGAATTTCATCAATTCTTGGAGCTATTAATTTTATTACAACAATTATTAATATACGAATTAATAGTATATCTTATGATCAAATACCATTATTTGTTTGAGCTGTTGGAATTACAGCATTATTATTATTACTCTCTCTACCTGTATTAGCTGGAGCTATTACAATACTTCTTACAGATCGAAATTTAAACACCTCTTTTTTTGATCCTGCAGGAGGAGGAGATCCTA???????????????????????????????

[COI_end]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ??????????????????????????????????????????????????????????????????????????AAAAAGAAACTTTTGGAAATTTaggtataATTTATGCTATATTAGCAATTGGATTATTAGGATTTATTGTGTGAGCTCATCATATATTTACTGTAGGTATAGATATTGATACTCGAGCTTATTTTACCTCTGCTACAATAATTATTGCAGTCCCAACAGGAATTAAAATTTTTAGTTGATTAGCAACTCTACATGGAACACAAATTAATTATAGTCCTTCCATACTTTGAAGACTAGGATTTATTTTTTTATTTACAGTAGGAGGATTAACTGGTGTAATTTTAGCTAATTCTTCAATTGATATTGCTCTTCATGATACTTATTATGTAgTAgCCCACTTTCATTATGTATTGTCTATAGGAGCAGTATTTGCTATTTTTGGAGGATTTGTCCATTGATATCCTTTATTTACAGGATTAATATTAAATCCATATTTATTAAAAATTCAATTTATTTCAATATTTATTGGAGTTAACTTAACTTTTTTCCCACAACATTTTTTAGGTTTAGCTGGTATACCTCGACGTTACTCAGATTACCCAGATAATTTTTTATCTTGAAATATTATTTCATCATTAGGATCTTATATTTCTCTATTTTCTATAATAATAATAaTTATTATTATATGAGAATCAATAACTTATCAACGTATAATTTTATTTTCATTAAATATACctTCTTCAATTGAGTGATATCAAAAaTTACCACCTGCCGAACATTCTTATAAtGAAC
CP100_11_Aus_bus                                                       ?????????????????????????????????????????????????????????????????????????AAAAAAGAAACTTTCGGAAGCTTAGGTATAATTTACGCTATATTAGCTATTGGATTATTAGGATtTATTGTATGAGCTCATCATATATTTACAGTAGGAATAGATATTGATACCCGAGCTTATTTTACTTCTGCTACAATAATTATTGCCGTACCAACAGGAATTAAAATTTTTAGCTGATTAGCAACTCTTCACGGAACTCAAATCAATTATAGTCCTTCCATACTTTGAAGATTAGGATTTATTTTTTTATTTACAGTAGGAGGACTAACTGGTGTAATTTTAGCTAATTCTTCAATTGATATTACTCTCCATGATACTTATTATGTTGTAGCTCATTTTCATTATGTTCTATCTATAGGAGCAGTATTTGCTATTTTCGGAGGATTTATCCACTGATACCCCTTATTTACAGGATTAATATTAAACCCATATTTATTAAAAATTCAATTCATTTCAATATTTATTGGAGTTAATTTAACTTTTTTTCCACAACATTTTTTAGGGTTAGCTGGTATACCTCGTCGTTATTCAGATTACCCAGATAATTTTTTATCTTGAAATATTATTTCATCATTAGGATCTTATATTTCATTATTTTCTATAATAATAATAATTATTATTATTTGAGAATCAATAATTTATCAACGTATAATTTTATTTACATTAAATATACCCTCTTCAATTGAATGATATCAAAATTTACCTCCTGCCGAACATTCTTATAATGAAC
CP100_12_Aus_cus                                                       TAAC????????????ATTTTAATTTTACCGGGATTTGGGATAATTTCACATATTATCTCCCAAGAAAGAGGGAAAAAAGAAACTTTTGGAAGTTTAGGAATAATTTATGCCATATTAGCAATTGGATTATTAGGATTTATTGTATGAGCTCACCATATGTTTACTGTAGGAATAGATATTGATACCCGAGCTTATTTTACTTCAGCTACAATAATTATTGCAGTTCCAACAGGAATTAAAATTTTTAGTTGATTAGCAACTCTCCATGGTACTCAAATTAATTATAGACCTTCTATACTTTGAGGTTTAGGATTTATTTTTTTATTCACAGTAGGAGGATTAACAGGCGTAATTTTAGCTAACTCTTCTTTAGACATTGCCCTACATGACACTTATTATGTCGTAGCTCATTTCCATTATGTACTTTCTATGGGAGCTGTATTCGCTATCTTTGGGGGATTTGTTCACTGATACCCTCTTTTTACTGGATTAGTGTTAAATCCCTACCTATTAAAAATTCAATTTATTTCTATATTTATTGGAGTTAATTTAACTTTTTTCCCTCAACATTTTTTAGGTTTAGCAGGTATGCCTCGTCGTTATTCTGATTATCCAGACAGCTTTTTATCTTGAAATATTATTTCTTCTTTAGGGTCTTATATTTCATTATTTTCTATAATATTAATTATTATTATTGTATGAGAATCAATAATTTACCAACGTGTTATTCTATTCTCATTAAATATACCCTCATCTATTGAATGATATCAAAATTTACCTCCAGCTGAACATTCCTACAATGAAC
CP100_13_Aus_dus                                                       aCaCCCAgaGgTGgATATTCTAgTTTTaCTaGGAtTtGGGataattTCaCATATTAtTTCTCAaGAAAGCGGAAAAAAAGAAACTTTTGGATGTTTAGGAATAATTTATGCAATATTAACAATTGGATTACTAGGATTTATTGTATGAGCCCATCATATATTCACAGTGGGTATAGATATTGATACTCGAGCTTATTTTACTTCAGCCACAATAATTATTGCTGTGCCAACAGGAATTAAAATCTTTAGTTGATTGGCAACTCTTCATGGAACTCAAATTAACTACAACCCCTCAATACTTTGAAGATTGGGATTTATTTTTTTATTTACAGTAGGTGGATTAACAGGAGTAATTCTAGCAAATTCATCAATTGATATTGCCCTTCATGATACATATTATGTTGTAGCCCACTTTCACTATGTACTATCTATAGGAGCAGTATTTGCTATTTTCGGAGGATTTATTCATTGATACCCTCTATTTACAGGATTAATTTTAAATCCTTATTTACTAAAAATCCAATTTATCTCTATATTTATTGGAGTTAATTTAACTTTCTTTCCACAACATTTTCTAGGATTAGCAGGAATACCTCGACGTTACTCAGATTATCCTGACAATTTTTTATCTTGAAATATTATTTCATCCTTAGGGTCTTATATCTCCTTATTTTCTATAATAATGATAATAATAATTATTTGGGAATCAATAATTTATCAACGAATCATTTTATTTTCATTAAATATGCCCTCCTCAATTGAATGATACCAAAACTTACCCCCTGCTGAACATTCTTATAATGAAC
CP100_14_Aus_eus                                                       acacccagaAgTCTAcaTTTtaatTTTACCAGGATTTGGTATAATTTCTCACattattTCTCAAGAAAGTGGTAAAAAAGAAACTTTTGGAAATTTAGGTATAATTTATGCTATATTAGCAATTGGATTATTAGGATTTATTGTTTGAGCTCATCACATATTTACAGTAGGAATAGATATTGATACTCGAGCTTATTTTACCTCTGCCACAATAATTATTGCTGTACCAACAGGAATTAAAATTTTTAGCTGATTAGCAACTCTTCACGGAACTCAAATTAATTATAGTCCCTCTATACTTTGAAGATTAGGATTTATTTTCTTATTCACAGTAGGAGGATTAACCGGAGTAATTTTAGCTAATTCTTCAATTGATATTGCTCTTCATGATACTTACTATGTTGTAGCCCATTTTCATTATGTATTATCTATAGGAGCAGTATTTGCTATTTTTGGAGGATTTGTTCATTGATACCCTCTATTTACAGGATTAGTATTAAATCCATATTTATTAAAAATTCAATTTATTTCAATATTTATTGGAGTTAACTTAACCTTTTTCCCACAACATTTTTTAGGCCTAGCTGGAATGCCTCGACGTTACTCAGACTATCCTGATAATTTTTTATCTTGAAATATTATTTCATCATTAGGATCTTATATTTCATTATTTTCTATAATAATAATAATTATTATCATTTGAGAATCAATAATTTATCAACGTATAATTTTATTTTCATTAAACATACCTTCCTCAATTGAGTGATATCAAAACTTACCTCCTGCCGAACATTCTTATAatgaac
CP100_15_Aus_fus                                                       ??????????????????????????????????????????????????????ATATTCCCAAGAAAGAGGTAAAAAAGAAACTTTTGGATGTTTGGGTATAATTTATGCTATACTAGCAATTGGTTTACTAGGATTTATTGTTTGAGCTCACCATATATTTACAGTAGGCATAGATATTGATACCCGAGCCTATTTCACATCAGCTACAATAATTATTGCTGTACCAACAGGAATTAAAATTTTTAGTTGATTAGCAACTCTTCATGGAACTCAAATTAATTATAGTCCTTCAATACTTTGAAGATTAGGATTTATTTTTTTATTTACAGTAGGAGGATTAACAGGTGTAATTTTAGCTAATTCATCAATTGATATTGCTCTTCATGACACATATTATGTTGTTGCACACTTCCATTATGTACTATCTATAGGAGCAGTATTTGCTATTTTCGGAGGATTTGTACATTGATACCCCTTATTTACAGGATTAATTTTAAACCCTTATTTATTAAAAATTCAATTTATTTCAATATTTATTGGAGTTAATTTAACTTTTTTTCCACAACATTTTTTAGGTTTAGCAGGGATACCTCGACGTTACTCCGATTACCCTGATAATTTTTTATCTTGAAATATTATTTCATCATTAGGGTCTTACATTTCCCTATTTTCTATAATTATAATTATTATTATTATTTGAGAATCAATAATTTACCAACGTATTATTTTATTTTCATTAAACATACCCTCTTCAATTGAATGATACCAAACCTTACCCCCCGCTGA?CATTCATATAATGAAC
CP100_16_Aus_gus                                                       aCACCCAGAaGTTTATATTTTAATTTTACCAGGATTTGGAATAATTTCACATATTATTTCTCAAGAAAGTGGTAAAAAAGAAACTTTTGGAAGTTTAGGAATAATTTATGCTATATTAGCAATTGGATTGTTAGGATTTATTGTATGAGCTCATCATATATTTACAGTAGGTATGGATATTGATACTCGAGCTTATTTTACTTCCGCTACAATAATTATTGCCGTACCAACAGGAATTAAAATTTTTAGTTGATTAGCAACTCTTCATGGGACCCAAATTAATTATAGCCCCTCGATACTTTGAGGATTAGGATTTATTTTTTTATTTACAGTAGGAGGATTAACTGGTGTGATTTTAGCTAATTCTTCAATTGATATTGCTCTTCATGATACTTATTATGTTGTAGCCCATTTTCATTACGTATTATCTATAGGAGCAGTATTTGCTATTTTTGGAGGATTTATTCATTGATACCCTTTATTTACAGGATTAGTATTAAATCCATATTTATTAAAAATTCAATTTATTTCAATATTTATTGGAGTTAATTTAACTTTTTTTCCCCAACATTTTTTAGGTTTAGCTGGAATACCTCGACGTTATTCAGATTACCCAGATAGTTTTTTATCTTGAAATATTATTTCATCATTAGGATCTTATATTTCACTATTTTCTATAATAATAATTATTATTATTATTTGAGAATCAATAATTTATCAACGTATAATTTTATTTTCATTAAATATACCCTCTTCAATTGAATGATATCAAAATTTACCACCTGCCGAACATTCGTATAATGAAC
CP100_17_Aus_hus                                                       ?????????????????????????????????????????????????????????????????????????AAAAAAGAAACTTTTGGATGTttaggtataatttAtGCcaTATTAGCAATTGGATTATTAGGATTTATTGTATGAGCTCATCACATATTTACAGTAGGTATAGATATTGATACTCGAGCTTATTTTACTTCAGCTACAATAATTATTGCAGTACCAACAGGAATTAAAATTTTTAGTTGATTAGCAACTCTTCATGGTACTCAAATTAATTATAGACCCTCAATACTTTGAAGATTAGGATTTATTTTCCTATTTACAGTAGGAGGATTAACAGGAGTAATCTTAGCTAATTCATCAATTGACATTGCACTTCATGATACATATTATGTTGTAgCTCATTTTCATTATGTACTATCTATAGGAGCAGTATTTGCTATTTTTGGGGGATTTGTTCATTGATATCCTTTATTTACAGGATTAATTTTAAATCCTTATTTATTAAAAATTCAATTTATTTCTATATTTATTGGAGTTAACTTAACTttTTTCCCACAACATTTCTTAGGTTTAGCAGGAATACCTCGACGTTACTCAGATTACCCTGACAATTTCTTATCCTGAAATATTATTTCATCATTAgGATCTTATATTTCtTTATTCTCTATAATAATAATAATAATTATTATTTGAGAATCAATAATTTATCAACGTATTATTTTATTTTCATTAAATaTACCATCTTCAATTGAATGAtATCAAAAtTTAcCCCCTGCTGAAcATTCtTatAatGAAC
CP100_18_Aus_ius                                                       AcacccAgAGgtgGatATtCTAGTtttActaggAtTtGGgATaaTTTCaCATATTATTTCTCAaGAAAGCGGAAAAAAAGAAACTTTTGGATGTTTAGGAATAATTTATGCAATATTAACAATTGGATTACTAGGATTTATTGTATGAGCCCATCATATATTCACAGTGGGTATAGATATTGATACTCGAGCTTATTTTACTTCAGCCACAATAATTATTGCTGTGCCAACAGGAATTAAAATCTTTAGTTGATTGGCAACTCTTCATGGAACTCAAATTAACTACAACCCCTCAATACTTTGAAGATTGGGATTTATTTTTTTATTTACAGTAGGTGGATTAACAGGAGTAATTCTAGCAAATTCATCAATTGATATTGCCCTTCATGATACATATTATGTTGTAGCCCACTTTCACTATGTACTATCTATAGGAGCAGTATTTGCTATTTTCGGAGGATTTATTCATTGATACCCTCTATTTACAGGATTAATTTTAAATCCTTATTTACTAAAAATCCAATTTATCTCTATATTTATTGGAGTTAATTTAACTTTCTTTCCACAACATTTTCTAGGATTAGCAGGAATACCTCGACGTTACTCAGATTATCCTGACAATTTTTTATCTTGAAATATTATTTCATCCTTAGGGTCTTATATCTCCCTATTTTCTATAATAATGATAATAATAATTATTTGGGAATCAATAATTTATCAACGAATCATTTTATTTTCATTAAATATGCCCTCCTCAATTGAATGATACCAAAACTTACCCCCTGCTGAACATTCTTATAATGAAC
CP100_19_Aus_jus                                                       ??????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????

[ef1a]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_11_Aus_bus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_12_Aus_cus                                                       ACACGTCGACTCCGGCAAGTCCACTACCACAGGACATTTGATTTACAAATGTGGTGGTATCGACAAGCGTACCATCGAGAAGTTCGAGAAGGAAGCCCAGGAGATGGGCAAGGGTTCCTTCAAATACGCCTGGGTGCTGGACAAACTGAAAGCTGAGCGCGAACGTGGTATCACCATCGACATCGCCCTGTGGAAGTTTGAGACCGCCAAATACTACGTCACCATCATCGACGCTCCTGGACACAGAGATTTCATCAAGAACATGATCACGGGAACCTCGCAGGCTGACTGCGCTGTGCTGATCGTGGCCGCCGGTACTGGTGAGTTCGAAGCTGGTATCTCCAAGAACGGCCAGACCCGTGAGCACGCGCTGCTCGCCTTCACACTCGGCGTCAAGCAGTTGATCGTGGGTGTCAACAAAATGGACTCCACTGAGCCACCCTACAGTGAACCCCGTTTTGAGGAAATCAAGAAAGAAGTGTCCTCTTACATCAAGAAGATTGGTTACAACCCAGCTGCCGTCGCTTTCGTACCCATTTCTGGCTGGCACGGAGATAACATGCTGGAGCCGTCCACTAAAATGCCCTGGTTCCAAGGGTGGCAAGTGGAGCGCAAGGAGGGTAAGGCTGAAGGCAAATGCCTCATTGA?GCCCTGGATGCCATCCTTCCCCCGGCTCGTCCCACAGACCAGGCTCTGCGACTTCCCCTTCAGGATGTCTACCAAATCGGTGGTATTGGTACAGTGCCCGTAGGCA?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_13_Aus_dus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_14_Aus_eus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_15_Aus_fus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_16_Aus_gus                                                       ?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????TTTCGTACCCCTTTCTGGCTGGTACGGAGACAACATGCTGGAGCCGTCCACCAAAATGCCCTGGTTCAAAGGATGGCAAGTTGAGCGCAAGGAGGGTAAGGCTGAAGGCAAGTGCCTCATCGAAGCATTGGACGCCATCCTACCTCCAGCACGTCCTACTGACAAGGATCTGCGACTTCCCCTACAGGACGTCTACAAAATTGGTGATATTGGTACAGTGCCAGTAGGCAGAGTGGAAACTGGGGTCCTTAAACCCGGTACCATTGTTGTTTTCGCTCCCGCTAACATTACTACTGAAGTTAAGTCTGTAGAGATGCACCACGAAGCCCTCTCTGAGGCTGTGCCTGGGGACAATGTTGGTTTCAACGTCAAGAACGTCTCAGTCAAGGAATTGCGTCGTGGTTATGTTGCCGGTGACTCCAAGAACAACCCACCTAAGGGAGCTGCTGACTTCACAGCACAAGTCATTGTCCTCAACCACCCTGGTCAAATCTCCAATGGTTACACACCTGTATTGGATTGCCACACAGCTCACATTGCCTGCAAATTCGCTGAAATCAAAGAGAAGGTAGACCGTCGGTCTGGTAAATCTACTGAAGAAAATCCCAAATCAATAAAATCTGGTGACGCTGCTATTGTCAACCTTGTACCATCCAAACCTCTTTGTGTTGAGGCCTTCCAaGAATTCCCTCCTCTTGGCCGTT???????????
CP100_17_Aus_hus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_18_Aus_ius                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_19_Aus_jus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????

[RpS2]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_11_Aus_bus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_12_Aus_cus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_13_Aus_dus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_14_Aus_eus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_15_Aus_fus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_16_Aus_gus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_17_Aus_hus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_18_Aus_ius                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_19_Aus_jus                                                       ???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????

[RpS5]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss CGATGGTGTTGAAGCGGGCAGCATGGYTGTCGATTCCATGCCACTGCCACAAGCGGCTGACATTCCAGAAATCAAATTGTTCGGAAGATGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAGGAAAAGTATGCCAAATACCTGCCCCACTCGGCGGGTAGGTATGCCCACAAACGGTTCCGTAAAGCGCAGTGCCCRATTGTTGAACGTCTGACCAATTCACTGATGATGCACGGAAGAAACAACGGCAAGAAATTGATGGCCGTTCGAATCGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACMGGTGAAAATCCYTTGCAAGTACTTGTTACAGCCATCATCAACTCTGGACCCCGTGAAGACTCTACTAGAATTGGTCGTGCTGGTACTGTGCGTCGTCAGGCTGTTGACGTGTCGCCCCTACGYCGTGTGAATCAAGCTATATGGCTGTTGTGTACTGGAGCACGTGAAGCTGCYTTCAGAAACATTAAAACAATTGCAGAATGTGTTGCTGATGAACTTATTAAYGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAGAAGGATGAGYTGGAACGTG
CP100_11_Aus_bus                                                       ??????????????????????????????????????????????CCACa?GCGGCTGACATTCCA?AAATTAA??TGTTTGGAAGATGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAGGAAAAGTATGCCAAATACCTGCCCCACTCGGCGGGTAGGTATGCCCACAAACGGTTCCGTAAAGCGCAGTGCCCAATTGTTGAACGTyTGACCAATTCACTAATGATGCACGGAAGAAACAACGGCAAGAAATTGATGGCCGTTCGAATCGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACAGGTGAAAATCCCCTGCAAGTACTTGTTACAGCCATCATCAATTCTGGACCCCGTGAAGACTCTACTAGAATTGGTCGTGCGGGTACAGTGCGTCGTCAGGCTGTTGACGTGTCGCCCCTACGCCGTGTGAATCAAGCTATATGGCTGTTGTGTACTGGAGCACGTGAAGCTGCCTTCAGAAACATTAAAACAATTGCAGAATGTGTTGCTGATGAACTTATTAATGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAGAAGGATGAGCTGGAACGTG
CP100_12_Aus_cus                                                       CGATGGCGTCGAGGCGGGCAGCATGGCTGTCGAATCCATGCCACTGCCGCAAGCGGCTGACATTCCGGAAATCAAATTATTTGGAAGATGGAGCTGTTACGATGTTCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAGGAGAAGTATGCCAAATACCTGCCCCACTCAGCGGGTAGGTATGCCCACAAACGTTTCCGTAAAGCGCAGTGCCCAATTGTTGAACGTCTGACCAATTCACTGATGATGCACGGAAGAAACAACGGCAAGAAACTGATGGCCGTTCGTATCGTCAAGCACGCTTTTGAGATCATCCACCTTTTGACTGGTGAAAATCCTCTGCAGGTTCTTGTTACAGCyATTATCAACTCTGGGCCTCGTGAAGACTCTACTAGAATTGGTCGTGCGGGTACAGTGCGyCGTCAGGCTGTGGATGTGTCACCCCTACGCCGTGTGAACCAAGCCATTTGGCTATTATGTACTGGAGCGCGGGAAGCTGCATTCAGAAACATTAAAACAATTGCAGAGTGTGTTGCTGATGAACTTATCAATGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAGAAGGATGAACTTGAACGTG
CP100_13_Aus_dus                                                       ?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_14_Aus_eus                                                       CGATGGCGTTGAAGCGGGCAGCATGGCTGTTGATTCCATGCCACTGCCACAAGCGGCTGACATTCCAGAAATAAAATTGTTCGGAAGAtGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAAGAAAAGTATGCCAAATACCTGCCACACTCGGCGGGTAGGTATGCCCACAAACGGTTCCGTAAAGCGCAGTGCCCAATTGTTGAACGCCTGACTAATTCACTGATGATGCATGGAAGAAACAACGGCAAGAAATTGATGGCCGTTCGAATCGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACAGGTGAAAACCCCCTGCAAGTTCTTGTTACAGCTATCATAAACTCTGGACCCCGTGAAGACTCTACTAGAATCGGTCGTGCGGGTACAGTGCGTCGTCAGGCTGTAGACGTGTCGCCCCTACGCCGTGTCAATCAAGCTATATGGCTGTTGTGTACTGGAGCACGTGAAGCTGCCTTCAGAAATATTAAAACAATTGCCGAATGTGTTGCTGATGAAcTTATTAATGCTGCTAAGGGTTCGTCTAACTCTTAtGCCaTCAAGAAGAAGGAtgAgctg???????
CP100_15_Aus_fus                                                       cgatGGcGTTGAagCGGGCAGCATGGCTGTCGATTCCATGCCACTGCCAcAAgCggcTGACATTCCAGAAATCAAATTGTTCGGAAGATGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAGGAAAAGTATGCCAAATACCTGCCCCACTCGGCGGGTAGGTATGCCCACAAACGGTTCCGTAAAGCGCAATGCCCAATTGTTGAACGTCTGACCAATTCACTGATGATGCATGGAAGAAACAATGGCAAGAAATTGATGGCCGTTCGAATCGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACAGGTGAAAATCCCCTGCAAGTACTTGTTACAGCCATCATTAACTCTGGACCCCGTGAAGACTCTACTAGAATTGGTCGTGCrGGTACAGTACGTCGTCAGGCTGTGgAyGTGTCGCCCCTACGCCGTGTGAATCAAGCTATATGGCTGTTGTGTACTGGAGCACGTGAAGCTGCCTTCAGAAACATTAAAACAATTGCAGAATGTGTTGCTGATGAACTTATTAATGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAgAAGGATGAGCTGGAACGTG
CP100_16_Aus_gus                                                       ?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_17_Aus_hus                                                       ????????????????????????????????????????????????????????????????????AAATCAAATTGTTTGGAAGATGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGACTACATTTCTGTGAAGGAAAAGTATGCCAAATACCTGCCCCACTCGGCAGGTAGGTATGCCCACAAACGATTCCGTAAAGCGCAGTGCCCAATTGTTGAACGTCTGACCAACTCACTGATGATGCACGGAAGAAACAACGGCAAGAAATTGATGGCCGTTCGAATCGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACTGGAGAAAATCCCCTGCAAGTTCTTGTTACAGCCATCATCAATTCTGGACCCCGTGAAGATTCTACTAGAATCGGTCGAGCGGGTACAGTGCGTCGTCAGGCTGTGGACGTGTCACCCCTACGCCGTGTGAATCAAGCTATATGGCTATTGTGTACTGGAGCACGTGAAGCTGCCTTCAGAAACATTAAAACAATTGCAGAATGTGTTGCTGATGAACTTATTAATGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAGAAGGATGAGCTGGAACGTG
CP100_18_Aus_ius                                                       ?????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_19_Aus_jus                                                       cgatgGCGTTGAaGCGGGCAGCATGGCTGTTGACTCCATGGCACTGCCGCAAgCggctGACATTCCAGAAATCAAATTGTTCGGAAGATGGAGCTGTTACGATGTGCAGGTGTCCGATATGTCTTTACAAGATTACATTTCTGTGAAGGAAAAGTATGCCAAATACCTGCCCCACTCGGCGGGTAGGTATGCCCACAAACGGTTCCGTAAAGCGCAGTGCCCAATTGTTGAACGTCTGACCAATTCACTGATGATGCACGGAAGAAACAATGGCAAGAAATTGATGGCCGTTCGAATTGTCAAGCACGCTTTTGAGATCATCCACCTTCTGACAGGTGAAAATCCCCTGCAAGTACTTGTTACAGCCATCATCAACTCTGGACCCCGTGAAGACTCTACTAGAATTGGTCGTGCGGGTACAGTGCGTCGTCAGGCTGTTGACGTGTCGCCCCTACGCCGTGTGAATCAAGCTATATGGCTGTTGTGTACTGGAGCACGTGAAGCTGCCTTCAGAAACATTAAAACAATTGCAGAATGTGTTGCTGATGAACTTATTAATGCTGCTAAGGGTTCATCTAACTCTTATGCCATCAAGAAGAAGGATGAGCTGGAACGTG

[wingless]
CP100_10_Aussssssssssssssssssssssssssss_aussssssssssssssssssssssssssss ???????????????????????????????????????????TCTGTAGGCGATGCCTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTCATGATGCCCAATACGGAGTTAGAAGCGCCTGCTCAGCGAAACGACGCCGCCCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGA?TCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCCGGCACGCACGGGCGTGCCTGCAACGACACGAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGCGGCCGTGGCTACCGGACCGAGACAATGTTCGTCGTGGAGCGATGCAAC
CP100_11_Aus_bus                                                       CTgCacGgTCaagacGTGCtGgaTgaggctGccGacTTttcggtCtGTAgGCGACGCCTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTTATGATGCCCAATACGGAGTTGGAAGCACCAGCTCAGCGAAACGATGCCGCTCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTwCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCAGGCACGCACGGGCGTTCCTGCAATGACACGAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGTGGCCGTGGCTACCGGACCGAGACAATGTTCGTTGTGGAGCGATGCAAC
CP100_12_Aus_cus                                                       ????????????????????????????????????????????CTGTAGGCGATGCATTG?AAGACGGCTTCGACGGAGCGTCGCGGGTCATGATGCCCAATACGGAAATAGAAGCACCGGCTCAGCGAAACGATGCCGCTCCGCACAGAGTCCCGCGACGAGACCGGTACAGATTTCAACTTCGGCCGCACAACCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTATACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCCGGTACGCACGGGCGTGCCTGCAACGACACGAGTATCGGCGTCGATGGCTGCGACCTCATGTGCTGCGGCCGCGGCTACCGGACCGATACGATGTTCGTCGTGGAGCGATGCAAC
CP100_13_Aus_dus                                                       ????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????
CP100_14_Aus_eus                                                       ??????????????????????????????????????????????????????????????AGGACGGCTTCGACGGAGCGTCGCGGGTCATGATGCCTAACACGGAGGTAGAAGCACCGGTTCAGCGAAACGATGCCGCTCCGCACAGAGTTCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGCTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCCGGTACGCACGGGCGTGCCTGCAACGACACGAGTATCGGCGTCGATGGCTGCGACCTCATGTGCTGCGGCCGCGGCTACCGGACCGAGACAATGTTCGTCGTGGAGCGATGCAAC
CP100_15_Aus_fus                                                       CTGCAcggtCaagacGTGCTGGatGaggcTGccGActtTTcggtctGtAgGCGATGCtTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTCATGATGCCCAATACGGAGTTAGAAGCACCGGCTCAGCGAAACGATGCCGCCCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCCGGTACGCACGGGCGTGCCTGCAACGACACGAGTATCGGTGTCGACGGGTGCGACCTCATGTGCTGCGGCCGCGGCTACCGGACAGAGACAATGTTCGTCGTGGAACGATGCAAC
CP100_16_Aus_gus                                                       CtGcAcgGtCaagacGTGCTGGAtGAgGCTGCCGACTTTTcggtctgtagGCGACGCCTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTTATGATGCCCAATACGGAGTTGGAAGCACCAGCTCAGCGAAACGATGCCGCTCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCAGGCACGCACGGGCGTTCCTGCAATGATACGAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGTGGCCGTGGCTACCGGACCGAGACAATGTTCGTCGTGGAGCGATGCAAC
CP100_17_Aus_hus                                                       CtGcAcgGtcaagacgTgctGGAtGCgGCTGCCGAcktTtcggtCtgtaggcgAtGccTTGAAGGACGGCTTCGACGGAGCGTCgcGggTCATGATGCCCAATACGGAGCTAGAAGCGCCGGCTCAGCGAAACGATGCCGCCCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCCGGCACGCACGGGCGTGCCTGCAACGACACGAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGCGGCCGTGGCTACCGGACCGAGACCATGTTCGTCGTGGAGCGATGCAAC
CP100_18_Aus_ius                                                       CTGCaCggtCaagacGTGCTGGatGaggcTGccGActtTTcggtctGTAgGCGACGCCTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTTATGATGCCCAATACGGAGTTGGAAGCACCAGCTCAGCGAAACGATGCCGCCCCCCACAGAGTCCCGCGACGAGACCGCTACAGATTTCAACTTCGGCCGCAyAATCCTGACCACAAAAyACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCAGGCACGCACGGGCGTTCCTGCAACGACACGAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGTGGCCGTGGCTACCGGACCGAGACAATGTTCGTCGTGGAGCGATGCAAC
CP100_19_Aus_jus                                                       ????????????????????????????????????????????CTGTAGGCGACGCCTTGAAGGACGGCTTCGACGGAGCGTCGCGGGTTATGATGCCCAATACGGAGTTGGAAGCACCAGCTCAGCGAAACGATGCCGCTCCGCACAGAGTCCCGCGACGAGACCGATACAGATTTCAACTTCGGCCGCACAATCCTGACCACAAAACACCCGGAGTCAAGGACCTAGTGTACTTGGAATCATCGCCGGGTTTCTGCGAAAAGAACCCGCGGCTGGGCATTCCAGGCACGCACGGGCGTTCCTGCAATGACACRAGTATCGGCGTCGACGGCTGCGACCTCATGTGCTGTGGCCGTGGCTACCGGACCGAGACAATGTTCGTCGTGGAGCGATGCAAC
;
END;
"""
        result = creator.dataset_block
        self.assertEqual(expected.strip(), result)
