from typing import Type, cast

from allauth.account import app_settings as allauth_settings
from django.contrib.auth import get_user_model
from django.contrib.auth.management.commands import createsuperuser as django_createsuperuser
from django.contrib.auth.models import AbstractUser
from django.core.management import BaseCommand
from django.db.models.signals import post_save

from composed_configuration._allauth_support import (
    createsuperuser as allauth_support_createsuperuser,
)
from composed_configuration._allauth_support.receiver import verify_email_address_on_user_post_save

"""
When Allauth is configured to use a User's `email` as the `username`, override the `createsuperuser`
management command to only prompt for an email address.
"""

# If using email as username
if not allauth_settings.USERNAME_REQUIRED:
    # Expose the modified command
    Command: Type[BaseCommand] = allauth_support_createsuperuser.Command
    user_model: Type[AbstractUser] = allauth_support_createsuperuser.EmailAsUsernameProxyUser

else:
    # Expose the pristine upstream version of the command
    Command = django_createsuperuser.Command
    user_model = cast(Type[AbstractUser], get_user_model())

# Always automatically verify email addresses of newly created superusers
post_save.connect(verify_email_address_on_user_post_save, sender=user_model)
