# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['chocs',
 'chocs.dataclasses',
 'chocs.json_schema',
 'chocs.middleware',
 'chocs.serverless']

package_data = \
{'': ['*']}

install_requires = \
['pyyaml>=5.3.1,<6.0.0', 'typing_extensions>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'chocs',
    'version': '0.13.1',
    'description': 'Lightweight and powerful wsgi rest framework for rapid building applications based on wsgi servers.',
    'long_description': '# Chocs [![PyPI version](https://badge.fury.io/py/chocs.svg)](https://pypi.org/project/chocs/) ![Release](https://github.com/kodemore/chocs/workflows/Release/badge.svg) ![Linting and Tests](https://github.com/kodemore/chocs/workflows/Linting%20and%20Tests/badge.svg) [![codecov](https://codecov.io/gh/kodemore/chocs/branch/master/graph/badge.svg)](https://codecov.io/gh/kodemore/chocs) [![Maintainability](https://api.codeclimate.com/v1/badges/9e3c979283b2361a9174/maintainability)](https://codeclimate.com/github/kodemore/chocs/maintainability)\nChocs is a modern HTTP framework for building AWS HTTP API/REST API and WSGI compatible applications. \nChocs aims to be small, expressive, and robust. \nIt provides an elegant API for writing fault-proof, extensible microservices.  \n\n## Features\n\n - AWS Serverless integration\n - Open api integration  \n - Elegant and easy API\n - No additional bloat like built-in template engines, session handlers, etc.\n - Compatible with all WSGI servers\n - Loosely coupled components which can be used separately\n - Multipart body parsing\n - Graceful error handling\n - HTTP middleware support\n - Fast routing\n\n## Installation\n```\npip install chocs\n```\n\n## Quick start\n\n```python\nfrom chocs import Application\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\nfrom chocs import serve\n\n\nhttp = Application()\n\n@http.get("/hello/{name}")\ndef hello(request: HttpRequest) -> HttpResponse:\n    return HttpResponse(f"Hello {request.path_parameters.get(\'name\')}!")\n\nserve(http)\n```\n\n> Keep in mind that the `serve()` function is using the `bjoern` package, so make sure you included it in your project\n> dependencies before using it. You are able to use any WSGI compatible server.\n\n## Table of Contents\n\n- [Usage](#usage)\n  - [Running application with Gunicorn (or any other WSGI server)](#running-application-with-gunicorn-or-any-other-wsgi-server)\n  - [Running application in AWS Lambda (Http api or rest api)](#running-application-in-aws-lambda-http-api-or-rest-api)\n  - [Routing](#routing)\n    - [Parametrized routes](#parametrized-routes)\n    - [Wildcard routes](#wildcard-routes)\n    - [Route groups](#route-groups)\n  - [Middleware](#middleware)\n  - [Integration with openapi](#integration-with-openapi)\n  - [Transforming request\'s payload to custom dataclasses](#transforming-requests-payload-to-custom-dataclasses)\n    - [Strict mode](#strict-mode)\n    - [Non-strict mode, aka: auto hydration](#non-strict-mode-aka-auto-hydration)\n    - [Dataclass support library](#dataclass-support-library)\n  - [Handling validation errors with custom middleware](#handling-validation-errors-with-custom-middleware)\n  - [Defining and using a custom middleware](#defining-and-using-a-custom-middleware)\n  - [Request](#request)\n  - [Response](#response)\n    - [`chocs.HttpResponse.status_code: chocs.HttpStatus`](#chocsresponsestatus_code-chocshttpstatus)\n  - [Working with cookies](#working-with-cookies)\n    - [Reading client cookies](#reading-client-cookies)\n    - [Setting cookies](#setting-cookies)\n- [Contributing](#contributing)\n  - [Prerequisites](#prerequisites)\n  - [Installation](#installation)\n  - [Running tests](#running-tests)\n  - [Linting](#linting)\n  - [PR](#pr)\n\n# Usage\n\n## Running application with Gunicorn (or any other WSGI server)\n\n```python\n# myapp.py\nfrom chocs import Application\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\nfrom chocs import create_wsgi_handler\n\n\nhttp = Application()\n\n\n@http.get("/hello/{name}*")\ndef hello(request: HttpRequest) -> HttpResponse:\n    return HttpResponse(f"Hello {request.path_parameters.get(\'name\')}!")\n\napp = create_wsgi_handler(http, debug=False)\n```\n\n```bash\ngunicorn -w 4 myapp:app\n```\n\n## Running application in AWS Lambda (Http api or rest api)\n\n`handler.py`\n```python\nimport logging\n\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\nfrom chocs import Application\n\nlogger = logging.getLogger()\nlogger.setLevel(logging.INFO)\n\n\nhttp = Application()\n\n\n@http.get("/hello/{name}")\ndef hello_handler(request: HttpRequest) -> HttpResponse:\n    logger.info("Hello AWS!")\n    logger.info(request.attributes.get("aws_context"))\n    logger.info(request.attributes.get("aws_event"))\n    return HttpResponse(f"Hello {request.path_parameters.get(\'name\')}")\n\n\n__all__ = ["hello_handler"]\n```\n\n`serverless.yml`\n```yaml\nservice: aws-hello-name\n\nprovider:\n  name: aws\n  runtime: python3.8\n\nplugins:\n  - serverless-python-requirements\n\ncustom:\n  pythonRequirements:\n    dockerizePip: true\n\nfunctions:\n  hello_name:\n    handler: handler.hello_handler\n    events:\n      - httpApi:\n          method: GET\n          path: /hello/{name}\n```\n\n```bash\nserverless deploy\n```\n\n## Routing\nChocs is shipped with a built-in routing module. The easiest way to utilise chocs\' routing is to use `chocs.router` object.\n`chocs.router` is an instance of the module\'s internal class `chocs.application.Application`, which provides a simple API \nwhere each function is a decorator corresponding to an HTTP method.\n\n```python\nfrom chocs import Application, HttpResponse, HttpRequest\n\n\nhttp = Application()\n\n\n@http.get("/hello")\ndef hello(req: HttpRequest) -> HttpResponse:\n    ...\n```\n\nThe above example will assign the hello function to handle a `GET /hello` request. \n\nAvailable methods:\n- `delete`\n- `get`\n- `head`\n- `options`\n- `patch`\n- `post`\n- `put`\n- `trace`\n\n### Parametrized routes\n\nRoutes can contain parameterised parts. Parameters must be enclosed within `{` and `}`.\n\n```python\nfrom chocs import Application\n\nhttp = Application()\n\n\n@http.get("/pet/{id}")\ndef hello():\n    ...\n```\nWill match the following URIs:\n - `/pet/1`\n - `/pet/abc`\n - `/pet/abc1`\n \n### Wildcard routes\n\nAsterisks (`*`) can be used in the route\'s pattern to match any possible combination. Keep in mind that routes which \n_do not_ contain wildcards are prioritised over routes with wildcards.\n\n```python\nfrom chocs import Application\n\nhttp = Application()\n\n\n@http.get("/pet/*", id)\ndef hello():\n    ...\n```\n\nThe above example will match following URIs:\n- `/pet/a`\n- `/pet/a/b/c`\n- `/pet/12jd/fds`\n\n### Route groups\n\nChocs supports route groups. Route groups is implemented through [context lib interface](https://docs.python.org/3/library/contextlib.html).\nIf you need to split your application in smaller chunks with standalone req/res handlers consider the\nfollowing example:\n\n```python\nfrom threading import Thread\n\nfrom chocs.wsgi import serve \nfrom chocs import Application\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\n\nmain_app = Application()\n\nwith main_app.group("/users/{id}") as user_module:\n    \n    @user_module.post("/profile_picture")  # POST /users/{id}/profile_pictures\n    def create_profile_picture(request: HttpRequest) -> HttpResponse:\n        ...\n    \n    @user_module.get("/profile_picture")  # GET /users/{id}/profile_pictures\n    def get_profile_picture(request: HttpRequest) -> HttpResponse:\n        ...\n    \n    @user_module.get("/badges") # GET /users/{id}/badges\n    def badges(request: HttpRequest) -> HttpResponse:\n        ...\n\nwith main_app.group("/payments") as payment_module:\n\n    @payment_module.get("/analytics") # GET /payments/analytics\n    def get_analytics(request: HttpRequest) -> HttpResponse:\n        ...\n\nif __name__ == \'__main__\':\n    def wsgi_user_module():\n        serve(user_module, port=8081)\n    def wsgi_payment_module():\n        serve(payment_module, port=8082)\n\n    Thread(target=wsgi_user_module).start()\n    payment_module()\n```\n\nThe above example shows how to run two different modules, which support their own routes\non two different ports in the one process.\n\n## Middleware\n\nMiddleware are functions or classes that inherit `chocs.Middleware`. Middlewares have access to the request object\nand the `next` function which can be used to control middleware stack flow. Successful middleware execution should call\nthe `next` function which accepts a `chocs.HttpRequest` instance and returns `chocs.HttpReponse`.\n\nMiddlewares can perform various tasks:\n- Making changes in request/response objects ending\n- Validating input data\n- Authenticating users\n- End request-response cycle\n- Connecting to external data sources\n\n## Integration with openapi\n\nTo provide automatic validation for your request based on open api specification, \nuse `chocs.middleware.OpenApiMiddleware` middleware bundled with chocs:\n\n```python\nfrom chocs.middleware import OpenApiMiddleware\nfrom chocs import Application, HttpRequest, HttpResponse\nfrom os import path\nfrom dataclasses import dataclass\n\n# absolute path to file containing open api documentation; yaml and json files are supported\nopenapi_filename = path.join(path.dirname(__file__), "/openapi.yml")\n\n# instantiating application and passing open api middleware\napp = Application(OpenApiMiddleware(openapi_filename, validate_body=True, validate_query=True))\n\n# defining our dataclass for better type support\n@dataclass()\nclass Pet:\n    id: str\n    name: str\n\n# the registered route must correspond to open api route within `path` section.\n# if request body is invalid the registered function will not be executed\n@app.post("/pets") # `parsed_body` parameter can be used to map request to certain type\ndef create_pet(request: HttpRequest) -> HttpResponse:\n    try: \n        pet = Pet(**request.parsed_body)\n    except TypeError:\n        return HttpResponse(status=400)\n    \n    return HttpResponse(pet.name)\n```\n\n> Complete integration example can be [found here](./examples/input_validation_with_open_api/openapi.yml)\n\nChocs automatically validates:\n - request body, `application/json` header must be present for successful validation\n - query string parameters\n - request headers\n\n## Transforming request\'s payload to custom dataclasses\n\n```python\nfrom chocs.middleware import ParsedBodyMiddleware\nfrom chocs import Application, HttpRequest, HttpResponse\nfrom chocs.dataclasses import asdict\nfrom dataclasses import dataclass\nimport json\n\n# You can define whether to use strict mode or not for all defined routes.\napp = Application(ParsedBodyMiddleware(strict=False))\n\n@dataclass\nclass Pet:\n    id: str\n    name: str\n\n@app.post("/pets", parsed_body=Pet) # you can also override default strict mode\ndef create_pet(request: HttpRequest) -> HttpResponse:\n    pet: Pet = request.parsed_body\n    assert isinstance(pet, Pet)\n    return HttpResponse(json.dumps(asdict(pet)))\n```\n\n### Strict mode\n\nStrict mode is using initialiser defined in dataclass. Which means the request data\nis simply unpacked and passed to your dataclass, so you have to manually transform \nnested data to dataclasses in order to conform your dataclass interface, for example:\n\n```python\nfrom chocs.middleware import ParsedBodyMiddleware\nfrom chocs import Application, HttpRequest, HttpResponse\nfrom dataclasses import dataclass\nfrom typing import List\n\napp = Application(ParsedBodyMiddleware())\n\n@dataclass\nclass Tag:\n  name: str\n  id: str\n\n@dataclass\nclass Pet:\n id: str\n name: str\n age: int\n tags: List[Tag]\n \n def __post_init__(self):  # post init might be used to reformat your data\n  self.age = int(self.age)\n  tmp_tags = self.tags\n  self.tags = []\n  for tag in tmp_tags:\n   self.tags.append(Tag(**tag))\n\n@app.post("/pets", parsed_body=Pet) \ndef create_pet(request: HttpRequest) -> HttpResponse:\n pet: Pet = request.parsed_body\n assert isinstance(pet.tags[0], Tag)\n assert isinstance(pet, Pet)\n return HttpResponse(pet.name)\n\n```\n\n### Non-strict mode, aka: auto hydration\n\nIn non-strict mode `chocs` takes care of instantiating and hydrating your dataclasses. Complex and deeply\nnested structures are supported as long as used types are supported by `chocs` hydration mechanism.\nList of supported types can be found below in dataclass support library\n\n> Note: __post_init__ method is not called as a part of hydration process.\n\n### Dataclass support library\n\nDataclass support library is composed of two functions to help with daily tasks while working\nwith dataclasses. \n\n#### `chocs.dataclasses.init_dataclass(value: dict, type_name)`\n\n`init_dataclass` function is instantiating dataclass of specified `type_name` and will hydrate the instance \nwith values passed in `value` dictionary. Each of the passed dictionary\'s keys must correspond to dataclass\'\nattributes in order to be properly interpreted. \n\nThis function support complex and nested hydration, which means if your dataclass aggregates other dataclasses \nor defines complex typing, `init_dataclass` function will respect your type annotations and will cast values \nto match the defined types. \n\nIf attributes in your dataclass do not specify the type value will be hydrated in to a newly created instance as is.\n\n#### `chocs.dataclasses.asdict(value)`\n\n`asdict` is the opposite of `init_dataclass` function, it takes an instance of dataclass as argument, and\nextracts its members to a dictionary, so the returned data can be stored as json object orn easily serialised to any other format.\n\n#### Supported data types\n\n`bool`\n\nPassed value is automatically hydrated to boolean with python\'s built-in `bool` on hydration and extraction.\n\n`dict`\n\nPassed value is automatically hydrated to dict with python\'s built-in `dict` on hydration and extraction.\n\n`float`\n\nPassed value is automatically hydrated to float with python\'s built-in `float` on hydration and extraction.\n\n`frozenset`\n\nPassed value is automatically hydrated to frozen set with python\'s built-in `frozenset` and extracted to `list`.\n\n`int`\n\nPassed value is automatically hydrated to int with python\'s built-in `int` on hydration and extraction.\n\n`list`\n\nPassed value is automatically hydrated to list with python\'s built-in `list` on hydration and extraction.\n\n`set`\n\nPassed value is automatically hydrated to set with python\'s built-in `set` and extracted to `list`.\n\n`str`\n\nPassed value is automatically hydrated to string with python\'s built-in `str` on hydration and extraction.\n\n`tuple`\n\nPassed value is automatically hydrated to tuple with python\'s built-in `tuple` and extracted to `list`.\n\n`collections.namedtuple`\n\nPassed value is automatically hydrated to named tuple and extracted to `list`.\n\n`collections.deque`\n\nPassed value is automatically hydrated to an instance of `collections.deque` and extracted to `list`.\n\n`collections.OrderedDict`\n\nPassed value is automatically hydrated to an instance of `collections.OrderedDict` and extracted to `dict`.\n\n`datetime.date`\n\nPassed value must be valid ISO-8601 date string, then it is automatically hydrated to an instance of `datetime.date` \nclass and extracted to ISO-8601 format compatible string.\n\n`datetime.datetime`\n\nPassed value must be valid ISO-8601 date time string, then it is automatically hydrated to an instance of `datetime.datetime` \nclass and extracted to ISO-8601 format compatible string.\n\n`datetime.time`\n\nPassed value must be valid ISO-8601 time string, then it is automatically hydrated to an instance of `datetime.time` \nclass and extracted to ISO-8601 format compatible string.\n\n`datetime.timedelta`\n\nPassed value must be valid ISO-8601 duration string, then it is automatically hydrated to an instance of `datetime.timedelta`\nclass and extracted to ISO-8601 format compatible string.\n\n`decimal.Decimal`\n\nPassed value must be a string containing valid decimal number representation, for more please read python\'s manual\nabout [`decimal.Decimal`](https://docs.python.org/3/library/decimal.html#decimal.Decimal), on extraction value is\nextracted back to string.\n\n`enum.Enum`\n\nSupports hydration of all instances of `enum.Enum` subclasses as long as value can be assigned\nto one of the members defined in the specified `enum.Enum` subclass. During extraction the value is\nextracted to value of the enum member.\n\n`enum.IntEnum`\n\nSame as `enum.Enum`.\n\n`typing.Any`\n\nPassed value is unchanged during hydration and extraction process.\n\n`typing.AnyStr`\n\nSame as `str`\n\n`typing.Deque`\n\nSame as `collection.dequeue` with one exception, if subtype is defined, eg `typing.Deque[int]` each item inside queue\nis hydrated accordingly to subtype.\n\n`typing.Dict`\n\nSame as `dict` with exception that keys and values are respectively hydrated and extracted to match\nannotated type.\n\n`typing.FrozenSet`\n\nSame as `frozenset` with exception that values of a frozen set are respectively hydrated and extracted to\nmatch annotated type.\n\n`typing.List`\n\nSame as `list` with exception that values of a list are respectively hydrated and extracted to match annotated type.\n\n`typing.NamedTuple`\n\nSame as `namedtuple`.\n\n`typing.Optional`\n\nOptional types can carry additional `None` value which chocs\' hydration process will respect, so for example \nif your type is `typing.Optional[int]` `None` value is not hydrated to `int`.\n\n`typing.Set`\n\nSame as `set` with exception that values of a set are respectively hydrated and extracted to match annotated type.\n\n`typing.Tuple`\n\nSame as `tuple` with exception that values of a set are respectively hydrated and extracted to match annotated types.\nEllipsis operator (`...`) is also supported.\n\n`typing.TypedDict`\n\nSame as `dict` but values of a dict are respectively hydrated and extracted to match annotated types. \n\n\n## Handling validation errors with custom middleware\n\nBy default, if validation fails users will see `500 response`. This behavior can be changed if custom middleware that\ncatches validation errors is defined and used in application.\n\n## Defining and using a custom middleware\n \nThe following code defines simple function middleware to catch validation errors when they appear and notifies users:\n\n```python\nfrom chocs.middleware import OpenApiMiddleware\nfrom chocs.json_schema.errors import ValidationError\nfrom chocs import Application, HttpRequest, HttpResponse\nfrom dataclasses import dataclass\nimport json\nfrom typing import Callable\nfrom os import path\n\nopenapi_filename = path.join(path.dirname(__file__), "/openapi.yml")\n\n\n# middleware must always accept two parameters; HttpRequest and Callable and return HttpResponse\ndef handle_errors(request: HttpRequest, next: Callable) -> HttpResponse:\n    try:\n        return next(request) # we pass request further to middleware pipeline\n    except ValidationError as error: # if exception is thrown it is caught here and new response is generated instead\n        json_response = {\n            "code": error.code,\n            "message": str(error),\n        }\n        return HttpResponse(json.dumps(json_response), status=422)\n    \n# error handling middleware must go before open api one to catch errors thrown inside open api middleware\napp = Application(handle_errors, OpenApiMiddleware(openapi_filename))\n\n@dataclass()\nclass Pet:\n  id: str\n  name: str\n\n@app.post("/pets", parsed_body=Pet)\ndef create_pet(request: HttpRequest) -> HttpResponse:\n  assert isinstance(request.parsed_body, Pet)\n\n  pet = request.parsed_body\n  return HttpResponse(pet.name)\n```\n\nFull working example can be found inside [examples directory](./examples/input_validation_with_open_api)\n\n## Request\n[Moved to wiki](https://github.com/kodemore/chocs/wiki/Request)\n\n## Response\n`chocs.Response` object is a part of request-response flow, and it is required to be returned by all functions\ndecorated with `router.*` method. Instance of the response class is recognised by `chocs.Application` and used to \ngenerate real response served to your clients.\n\n#### `chocs.HttpResponse.body: io.BytesIO` \nBody served to server\'s clients.\n\n### `chocs.HttpResponse.status_code: chocs.HttpStatus`\nValid response code, instance of `chocs.HttpStatus` enum can be used or just a status code\'s number.\n\n#### `chocs.HttpResponse.cookies:chocs.HttpCookieJar` \nResponse\'s cookies\n\n#### `chocs.HttpResponse.write(body: Union[bytes, str, bytearray])`\nWrite bytes to response body\n\n#### `chocs.HttpResponse.close()`\nMakes body non-writable.\n\n#### `chocs.HttpResponse.writable: bool`\nIndicates whether response\'s body is writable.\n\n#### `chocs.HttpResponse.parsed_body:chocs.HttpMessage`\nDepending on the content type it could be one of the following:\n- `chocs.FormHttpMessage`\n- `chocs.JsonHttpMessage`\n- `chocs.MultipartHttpMessage`\n- `chocs.YamlHttpMessage`\n\n#### `chocs.HttpResponse.as_dict(): dict`\nTries to convert response body to a dict and returns it.\n\n> Note this will only work with json and yaml content types.\n\n#### `chocs.HttpResponse.as_str(): str`\nReturns response content as a string.\n\n## Working with cookies\n\n`chocs.HttpCookieJar` object takes care of cookie handling. It can be accessed in dict-like manner, when item is requested,\ninstance of `chocs.HttpCookie` is returned to user. \n\nCookies can be set either by passing string value to the `chocs.CookieJar`\'s key, or by calling `chocs.CookieJar.append` \nmethod which accepts instance of `chocs.Cookie`.\n\n### Reading client cookies\n\nCookies can be easily accessed from `chocs.Request.cookies` object which is injected as a parameter to each function \nregistered as route handler. Consider the following example:\n\n```python\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\nfrom chocs import Application\nfrom chocs import serve\n\nhttp = Application()\n\n\n@http.get("/cookies")\ndef read_cookies(request: HttpRequest) -> HttpResponse:\n\n    message = "Hello"\n    if "user_name" in request.cookies:\n        message += f", {str(request.cookies[\'user_name\'])}"\n    message += "!"\n\n    return HttpResponse(body=message)\n\nserve(http)\n```\n\n### Setting cookies\n```python\nfrom datetime import datetime\n\nfrom chocs import HttpCookie\nfrom chocs import HttpRequest\nfrom chocs import HttpResponse\nfrom chocs import Application\nfrom chocs import serve\n\nhttp = Application()\n\n\n@http.get("/cookies")\ndef read_cookies(request: HttpRequest) -> HttpResponse:\n    response = HttpResponse(body="Hi! I have baked some cookies for ya!")\n    response.cookies[\'simple-cookie\'] = "Simple cookie for simple people"\n    response.cookies.append(HttpCookie("advanced-cookie", "This cookie will expire in 2021-01-01", expires=datetime(2021, 1, 1)))\n    return response\n\nserve(http)\n```\n\n# Contributing\n\n## Prerequisites\n\n- libev\n- python 3.8\n- docker\n\n## Installation\n\n`poetry install`\n\n## Running tests\n\n`poetry run pytest`\n\n## Linting\n\n```shell\npoetry run black .\npoetry run isort .\npoetry run mypy .\n```\n\n## PR \n',
    'author': 'Dawid Kraczkowski',
    'author_email': 'dawid.kraczkowski@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kodemore/chocs',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
