from __future__ import annotations

import functools
import itertools
import locale
import operator
import platform
import re
import string
import warnings

import sqlalchemy as sa
from sqlalchemy.dialects import postgresql as pg

import ibis.common.exceptions as com
import ibis.common.geospatial as geo
import ibis.expr.datatypes as dt
import ibis.expr.operations as ops

# used for literal translate
from ibis.backends.base.sql.alchemy import (
    fixed_arity,
    get_sqla_table,
    reduction,
    sqlalchemy_operation_registry,
    sqlalchemy_window_functions_registry,
    unary,
)
from ibis.backends.base.sql.alchemy.datatypes import to_sqla_type
from ibis.backends.base.sql.alchemy.geospatial import geospatial_supported
from ibis.backends.base.sql.alchemy.registry import (
    _bitwise_op,
    _extract,
    geospatial_functions,
    get_col_or_deferred_col,
)

operation_registry = sqlalchemy_operation_registry.copy()
operation_registry.update(sqlalchemy_window_functions_registry)

if geospatial_supported:
    operation_registry.update(geospatial_functions)


_truncate_precisions = {
    'us': 'microseconds',
    'ms': 'milliseconds',
    's': 'second',
    'm': 'minute',
    'h': 'hour',
    'D': 'day',
    'W': 'week',
    'M': 'month',
    'Q': 'quarter',
    'Y': 'year',
}


def _timestamp_truncate(t, op):
    sa_arg = t.translate(op.arg)
    try:
        precision = _truncate_precisions[op.unit]
    except KeyError:
        raise com.UnsupportedOperationError(f'Unsupported truncate unit {op.unit!r}')
    return sa.func.date_trunc(precision, sa_arg)


def _typeof(t, op):
    sa_arg = t.translate(op.arg)
    typ = sa.cast(sa.func.pg_typeof(sa_arg), sa.TEXT)

    # select pg_typeof('thing') returns unknown so we have to check the child's
    # type for nullness
    return sa.case(
        [
            ((typ == 'unknown') & (op.arg.output_dtype != dt.null), 'text'),
            ((typ == 'unknown') & (op.arg.output_dtype == dt.null), 'null'),
        ],
        else_=typ,
    )


_strftime_to_postgresql_rules = {
    '%a': 'TMDy',  # TM does it in a locale dependent way
    '%A': 'TMDay',
    '%w': 'D',  # 1-based day of week, see below for how we make this 0-based
    '%d': 'DD',  # day of month
    '%-d': 'FMDD',  # - is no leading zero for Python same for FM in postgres
    '%b': 'TMMon',  # Sep
    '%B': 'TMMonth',  # September
    '%m': 'MM',  # 01
    '%-m': 'FMMM',  # 1
    '%y': 'YY',  # 15
    '%Y': 'YYYY',  # 2015
    '%H': 'HH24',  # 09
    '%-H': 'FMHH24',  # 9
    '%I': 'HH12',  # 09
    '%-I': 'FMHH12',  # 9
    '%p': 'AM',  # AM or PM
    '%M': 'MI',  # zero padded minute
    '%-M': 'FMMI',  # Minute
    '%S': 'SS',  # zero padded second
    '%-S': 'FMSS',  # Second
    '%f': 'US',  # zero padded microsecond
    '%z': 'OF',  # utf offset
    '%Z': 'TZ',  # uppercase timezone name
    '%j': 'DDD',  # zero padded day of year
    '%-j': 'FMDDD',  # day of year
    '%U': 'WW',  # 1-based week of year
    # 'W': ?,  # meh
}

try:
    _strftime_to_postgresql_rules.update(
        {
            '%c': locale.nl_langinfo(locale.D_T_FMT),  # locale date and time
            '%x': locale.nl_langinfo(locale.D_FMT),  # locale date
            '%X': locale.nl_langinfo(locale.T_FMT),  # locale time
        }
    )
except AttributeError:
    warnings.warn(
        'locale specific date formats (%%c, %%x, %%X) are not yet implemented '
        'for %s' % platform.system()
    )


# translate strftime spec into mostly equivalent PostgreSQL spec
_scanner = re.Scanner(  # type: ignore # re does have a Scanner attribute
    # double quotes need to be escaped
    [('"', lambda *_: r'\"')]
    + [
        (
            '|'.join(
                map(
                    '(?:{})'.format,
                    itertools.chain(
                        _strftime_to_postgresql_rules.keys(),
                        [
                            # "%e" is in the C standard and Python actually
                            # generates this if your spec contains "%c" but we
                            # don't officially support it as a specifier so we
                            # need to special case it in the scanner
                            '%e',
                            r'\s+',
                            fr'[{re.escape(string.punctuation)}]',
                            fr'[^{re.escape(string.punctuation)}\s]+',
                        ],
                    ),
                )
            ),
            lambda _, token: token,
        )
    ]
)


_lexicon_values = frozenset(_strftime_to_postgresql_rules.values())

_strftime_blacklist = frozenset(['%w', '%U', '%c', '%x', '%X', '%e'])


def _reduce_tokens(tokens, arg):
    # current list of tokens
    curtokens = []

    # reduced list of tokens that accounts for blacklisted values
    reduced = []

    non_special_tokens = frozenset(_strftime_to_postgresql_rules) - _strftime_blacklist

    # TODO: how much of a hack is this?
    for token in tokens:
        # we are a non-special token %A, %d, etc.
        if token in non_special_tokens:
            curtokens.append(_strftime_to_postgresql_rules[token])

        # we have a string like DD, to escape this we
        # surround it with double quotes
        elif token in _lexicon_values:
            curtokens.append(f'"{token}"')

        # we have a token that needs special treatment
        elif token in _strftime_blacklist:
            if token == '%w':
                value = sa.extract('dow', arg)  # 0 based day of week
            elif token == '%U':
                value = sa.cast(sa.func.to_char(arg, 'WW'), sa.SMALLINT) - 1
            elif token == '%c' or token == '%x' or token == '%X':
                # re scan and tokenize this pattern
                try:
                    new_pattern = _strftime_to_postgresql_rules[token]
                except KeyError:
                    raise ValueError(
                        'locale specific date formats (%%c, %%x, %%X) are '
                        'not yet implemented for %s' % platform.system()
                    )

                new_tokens, _ = _scanner.scan(new_pattern)
                value = functools.reduce(
                    sa.sql.ColumnElement.concat,
                    _reduce_tokens(new_tokens, arg),
                )
            elif token == '%e':
                # pad with spaces instead of zeros
                value = sa.func.replace(sa.func.to_char(arg, 'DD'), '0', ' ')

            reduced += [
                sa.func.to_char(arg, ''.join(curtokens)),
                sa.cast(value, sa.TEXT),
            ]

            # empty current token list in case there are more tokens
            del curtokens[:]

        # uninteresting text
        else:
            curtokens.append(token)
    # append result to r if we had more tokens or if we have no
    # blacklisted tokens
    if curtokens:
        reduced.append(sa.func.to_char(arg, ''.join(curtokens)))
    return reduced


def _strftime(arg, pattern):
    tokens, _ = _scanner.scan(pattern.value)
    reduced = _reduce_tokens(tokens, arg)
    return functools.reduce(sa.sql.ColumnElement.concat, reduced)


def _find_in_set(t, op):
    # TODO
    # this operation works with any type, not just strings. should the
    # operation itself also have this property?
    return (
        sa.func.coalesce(
            sa.func.array_position(
                pg.array(list(map(t.translate, op.values))),
                t.translate(op.needle),
            ),
            0,
        )
        - 1
    )


def _log(t, op):
    arg, base = op.args
    sa_arg = t.translate(arg)
    if base is not None:
        sa_base = t.translate(base)
        return sa.cast(
            sa.func.log(sa.cast(sa_base, sa.NUMERIC), sa.cast(sa_arg, sa.NUMERIC)),
            t.get_sqla_type(op.output_dtype),
        )
    return sa.func.ln(sa_arg)


def _regex_extract(arg, pattern, index):
    # wrap in parens to support 0th group being the whole string
    pattern = "(" + pattern + ")"
    # arrays are 1-based in postgres
    index = index + 1
    does_match = sa.func.textregexeq(arg, pattern)
    matches = sa.func.regexp_match(arg, pattern, type_=pg.ARRAY(sa.TEXT))
    return sa.case([(does_match, matches[index])], else_=None)


def _array_repeat(t, op):
    """Repeat an array."""
    arg = t.translate(op.arg)
    times = t.translate(op.times)

    array = sa.sql.elements.Grouping(arg) if isinstance(op.arg, ops.Literal) else arg

    # We still need to prefix the table name to the column name in the final
    # query, so make sure the column knows its origin
    array.table = arg.table

    array_length = sa.func.cardinality(array)

    # sequence from 1 to the total number of elements desired in steps of 1.
    # the call to greatest isn't necessary, but it provides clearer intent
    # rather than depending on the implicit postgres generate_series behavior
    start = step = 1
    stop = sa.func.greatest(times, 0) * array_length
    series = sa.func.generate_series(start, stop, step).alias()
    series_column = sa.column(series.name, type_=sa.INTEGER)

    # if our current index modulo the array's length is a multiple of the
    # array's length, then the index is the array's length
    index_expression = series_column % array_length
    index = sa.func.coalesce(sa.func.nullif(index_expression, 0), array_length)

    # tie it all together in a scalar subquery and collapse that into an ARRAY
    return sa.func.array(sa.select(array[index]).select_from(series).scalar_subquery())


def _table_column(t, op):
    ctx = t.context
    table = op.table

    sa_table = get_sqla_table(ctx, table)
    out_expr = get_col_or_deferred_col(sa_table, op.name)

    if op.output_dtype.is_timestamp():
        timezone = op.output_dtype.timezone
        if timezone is not None:
            out_expr = out_expr.op('AT TIME ZONE')(timezone).label(op.name)

    # If the column does not originate from the table set in the current SELECT
    # context, we should format as a subquery
    if t.permit_subquery and ctx.is_foreign_expr(table):
        return sa.select(out_expr)

    return out_expr


def _round(t, op):
    arg, digits = op.args
    sa_arg = t.translate(arg)

    if digits is None:
        return sa.func.round(sa_arg)

    # postgres doesn't allow rounding of double precision values to a specific
    # number of digits (though simple truncation on doubles is allowed) so
    # we cast to numeric and then cast back if necessary
    result = sa.func.round(sa.cast(sa_arg, sa.NUMERIC), t.translate(digits))
    if digits is not None and arg.output_dtype.is_decimal():
        return result
    result = sa.cast(result, pg.DOUBLE_PRECISION())
    return result


def _mod(t, op):
    left, right = map(t.translate, op.args)

    # postgres doesn't allow modulus of double precision values, so upcast and
    # then downcast later if necessary
    if not op.output_dtype.is_integer():
        left = sa.cast(left, sa.NUMERIC)
        right = sa.cast(right, sa.NUMERIC)

    result = left % right
    if op.output_dtype.is_float64():
        return sa.cast(result, pg.DOUBLE_PRECISION())
    else:
        return result


def _neg_idx_to_pos(array, idx):
    return sa.case([(idx < 0, sa.func.cardinality(array) + idx)], else_=idx)


def _array_slice(*, index_converter, array_length):
    def translate(t, op):
        arg = t.translate(op.arg)

        arg_length = array_length(arg)

        if (start := op.start) is None:
            start = 0
        else:
            start = t.translate(start)
            start = sa.func.least(arg_length, index_converter(arg, start))

        if (stop := op.stop) is None:
            stop = arg_length
        else:
            stop = index_converter(arg, t.translate(stop))

        return arg[start + 1 : stop]

    return translate


def _array_index(*, index_converter):
    def translate(t, op):
        sa_array = t.translate(op.arg)
        sa_index = t.translate(op.index)
        if isinstance(op.arg, ops.Literal):
            sa_array = sa.sql.elements.Grouping(sa_array)
        return sa_array[index_converter(sa_array, sa_index) + 1]

    return translate


def _literal(_, op):
    dtype = op.output_dtype
    value = op.value

    if dtype.is_interval():
        return sa.text(f"INTERVAL '{value} {dtype.resolution}'")
    elif dtype.is_set():
        return list(map(sa.literal, value))
    # geo spatial data type
    elif dtype.is_geospatial():
        # inline_metadata ex: 'SRID=4326;POINT( ... )'
        return sa.literal_column(geo.translate_literal(op, inline_metadata=True))
    elif isinstance(value, tuple):
        return sa.literal_column(
            str(pg.array(value).compile(compile_kwargs=dict(literal_binds=True))),
            type_=to_sqla_type(dtype),
        )
    else:
        return sa.literal(value)


def _string_agg(t, op):
    agg = sa.func.string_agg(t.translate(op.arg), t.translate(op.sep))
    if (where := op.where) is not None:
        return agg.filter(t.translate(where))
    return agg


def _corr(t, op):
    if op.how == "sample":
        raise ValueError(
            f"{t.__class__.__name__} only implements population correlation "
            "coefficient"
        )
    return _binary_variance_reduction(sa.func.corr)(t, op)


def _covar(t, op):
    suffix = {"sample": "samp", "pop": "pop"}
    how = suffix.get(op.how, "samp")
    func = getattr(sa.func, f"covar_{how}")
    return _binary_variance_reduction(func)(t, op)


def _mode(t, op):
    arg = op.arg
    if (where := op.where) is not None:
        arg = ops.Where(where, arg, None)
    return sa.func.mode().within_group(t.translate(arg))


def _quantile(t, op):
    if op.interpolation is not None:
        warnings.warn(
            f"`{t.__module__.rsplit(',', 1)[0]}` backend does not support the "
            "`interpolation` argument"
        )
    arg = op.arg
    if (where := op.where) is not None:
        arg = ops.Where(where, arg, None)
    return sa.func.percentile_cont(t.translate(op.quantile)).within_group(
        t.translate(arg)
    )


def _binary_variance_reduction(func):
    def variance_compiler(t, op):
        x = op.left
        if (x_type := x.output_dtype).is_boolean():
            x = ops.Cast(x, dt.Int32(nullable=x_type.nullable))

        y = op.right
        if (y_type := y.output_dtype).is_boolean():
            y = ops.Cast(y, dt.Int32(nullable=y_type.nullable))

        result = func(t.translate(x), t.translate(y))

        if (where := op.where) is not None:
            return result.filter(t.translate(where))

        return result

    return variance_compiler


operation_registry.update(
    {
        ops.Literal: _literal,
        # We override this here to support time zones
        ops.TableColumn: _table_column,
        # types
        ops.TypeOf: _typeof,
        # Floating
        ops.IsNan: fixed_arity(lambda arg: arg == float('nan'), 1),
        ops.IsInf: fixed_arity(
            lambda arg: sa.or_(arg == float('inf'), arg == float('-inf')), 1
        ),
        # null handling
        ops.IfNull: fixed_arity(sa.func.coalesce, 2),
        # boolean reductions
        ops.Any: reduction(sa.func.bool_or),
        ops.All: reduction(sa.func.bool_and),
        ops.NotAny: reduction(lambda x: sa.not_(sa.func.bool_or(x))),
        ops.NotAll: reduction(lambda x: sa.not_(sa.func.bool_and(x))),
        # strings
        ops.GroupConcat: _string_agg,
        ops.Capitalize: unary(sa.func.initcap),
        ops.RegexSearch: fixed_arity(lambda x, y: x.op("~")(y), 2),
        # postgres defaults to replacing only the first occurrence
        ops.RegexReplace: fixed_arity(
            lambda string, pattern, replacement: sa.func.regexp_replace(
                string, pattern, replacement, 'g'
            ),
            3,
        ),
        ops.Translate: fixed_arity(sa.func.translate, 3),
        ops.RegexExtract: fixed_arity(_regex_extract, 3),
        ops.StringSplit: fixed_arity(
            lambda col, sep: sa.func.string_to_array(
                col, sep, type_=sa.ARRAY(col.type)
            ),
            2,
        ),
        ops.FindInSet: _find_in_set,
        # math
        ops.Log: _log,
        ops.Log2: unary(lambda x: sa.func.log(2, x)),
        ops.Log10: unary(sa.func.log),
        ops.Round: _round,
        ops.Modulus: _mod,
        # dates and times
        ops.DateFromYMD: fixed_arity(sa.func.make_date, 3),
        ops.DateTruncate: _timestamp_truncate,
        ops.TimestampTruncate: _timestamp_truncate,
        ops.IntervalFromInteger: (
            lambda t, op: t.translate(op.arg)
            * sa.text(f"INTERVAL '1 {op.output_dtype.resolution}'")
        ),
        ops.DateAdd: fixed_arity(operator.add, 2),
        ops.DateSub: fixed_arity(operator.sub, 2),
        ops.DateDiff: fixed_arity(operator.sub, 2),
        ops.TimestampAdd: fixed_arity(operator.add, 2),
        ops.TimestampSub: fixed_arity(operator.sub, 2),
        ops.TimestampDiff: fixed_arity(operator.sub, 2),
        ops.Strftime: fixed_arity(_strftime, 2),
        ops.ExtractEpochSeconds: fixed_arity(
            lambda arg: sa.cast(sa.extract('epoch', arg), sa.INTEGER), 1
        ),
        ops.ExtractDayOfYear: _extract('doy'),
        ops.ExtractWeekOfYear: _extract('week'),
        # extracting the second gives us the fractional part as well, so smash that
        # with a cast to SMALLINT
        ops.ExtractSecond: fixed_arity(
            lambda arg: sa.cast(sa.func.floor(sa.extract('second', arg)), sa.SMALLINT),
            1,
        ),
        # we get total number of milliseconds including seconds with extract so we
        # mod 1000
        ops.ExtractMillisecond: fixed_arity(
            lambda arg: sa.cast(
                sa.func.floor(sa.extract('millisecond', arg)) % 1000,
                sa.SMALLINT,
            ),
            1,
        ),
        ops.DayOfWeekIndex: fixed_arity(
            lambda arg: sa.cast(
                sa.cast(sa.extract('dow', arg) + 6, sa.SMALLINT) % 7, sa.SMALLINT
            ),
            1,
        ),
        ops.DayOfWeekName: fixed_arity(
            lambda arg: sa.func.trim(sa.func.to_char(arg, 'Day')), 1
        ),
        # now is in the timezone of the server, but we want UTC
        ops.TimestampNow: lambda *_: sa.func.timezone('UTC', sa.func.now()),
        ops.TimeFromHMS: fixed_arity(sa.func.make_time, 3),
        ops.CumulativeAll: unary(sa.func.bool_and),
        ops.CumulativeAny: unary(sa.func.bool_or),
        # array operations
        ops.ArrayLength: unary(sa.func.cardinality),
        ops.ArrayCollect: reduction(sa.func.array_agg),
        ops.ArrayColumn: (lambda t, op: pg.array(list(map(t.translate, op.cols)))),
        ops.ArraySlice: _array_slice(
            index_converter=_neg_idx_to_pos,
            array_length=sa.func.cardinality,
        ),
        ops.ArrayIndex: _array_index(index_converter=_neg_idx_to_pos),
        ops.ArrayConcat: fixed_arity(sa.sql.expression.ColumnElement.concat, 2),
        ops.ArrayRepeat: _array_repeat,
        ops.Unnest: unary(sa.func.unnest),
        ops.Covariance: _covar,
        ops.Correlation: _corr,
        ops.BitwiseXor: _bitwise_op("#"),
        ops.Mode: _mode,
        ops.Quantile: _quantile,
        ops.MultiQuantile: _quantile,
    }
)
