# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['jwskate',
 'jwskate.jwa',
 'jwskate.jwa.encryption',
 'jwskate.jwa.key_mgmt',
 'jwskate.jwa.signature',
 'jwskate.jwe',
 'jwskate.jwk',
 'jwskate.jws',
 'jwskate.jwt',
 'tests',
 'tests.test_jwk']

package_data = \
{'': ['*']}

install_requires = \
['backports.cached-property>=1.0.1,<2.0.0',
 'binapy>=0.6.0,<0.7.0',
 'cryptography>=3.4',
 'typing-extensions>=4.2.0,<5.0.0']

setup_kwargs = {
    'name': 'jwskate',
    'version': '0.4.0',
    'description': 'A Pythonic implementation of Json Web Signature, Keys, Algorithms, Tokens and Encryption (RFC7514 to 7519), on top of the `cryptography` module.',
    'long_description': '# JwSkate\n\n[![PyPi](https://img.shields.io/pypi/v/jwskate.svg)](https://pypi.python.org/pypi/jwskate)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nA Pythonic implementation of the JOSE set of IETF specifications: [Json Web Signature][rfc7515], [Keys][rfc7517], [Algorithms][rfc7518], [Tokens][rfc7519]\nand [Encryption][rfc7516] (RFC7515 to 7519), and their extensions [ECDH Signatures][rfc8037] (RFC8037), [JWK Thumbprints][rfc7638] (RFC7638),\nand [JWK Thumbprint URI][rfc9278] (RFC9278).\n\n- Free software: MIT\n- Documentation: <https://guillp.github.io/jwskate/>\n\nA quick usage example, generating an RSA private key, signing some data, then validating that signature:\n\n```python\nfrom jwskate import Jwk\n\n# let\'s generate a random private key\nrsa_private_jwk = (\n    Jwk.generate_for_alg(  # generated key will automatically be RSA, based on the required \'alg\'\n        alg="RS256", key_size=2048\n    )\n    .with_kid_thumbprint()  # include an RFC7638 thumbprint as key id\n    .with_usage_parameters()  # include the appropriate \'use\' and \'key_ops\' parameters in the JWK, based on the \'alg\'\n)\n\ndata = b"Signing is easy!"\nsignature = rsa_private_jwk.sign(data)\n\n# extract the public key, and verify the signature with it\nrsa_public_jwk = rsa_private_jwk.public_jwk()\nassert rsa_public_jwk.verify(data, signature)\n\n# let\'s see what a Jwk looks like:\nassert isinstance(rsa_private_jwk, dict)  # Jwk are dict\n\nprint(rsa_private_jwk)\n```\n\nThe result of this print will look like this (with the random parts abbreviated to `...` for display purposes only):\n\n```\n{\'kty\': \'RSA\',\n \'n\': \'...\',\n \'e\': \'AQAB\',\n \'d\': \'...\',\n \'p\': \'...\',\n \'q\': \'...\',\n \'dp\': \'...\',\n \'dq\': \'...\',\n \'qi\': \'...\',\n \'alg\': \'RS256\',\n \'kid\': \'...\',\n \'use\': \'sig\',\n \'key_ops\': [\'sign\']}\n```\n\nNow let\'s sign a JWT containing arbitrary claims, this time using an EC key:\n\n```python\nfrom jwskate import Jwk, Jwt\n\nprivate_jwk = Jwk.generate_for_alg(\n    "ES256", kid="my_key"\n)  # let\'s specify a kid manually\nclaims = {"sub": "some_sub", "claim1": "value1"}\n\njwt = Jwt.sign(claims, private_jwk)\n# that\'s it! we have a signed JWT\nassert jwt.claims == claims  # claims can be accessed as a dict\nassert jwt.sub == "some_sub"  # or individual claims can be accessed as attributes\nassert jwt["claim1"] == "value1"  # or as dict items\nassert jwt.alg == "ES256"  # alg and kid headers are also accessible as attributes\nassert jwt.kid == private_jwk.kid\nassert jwt.verify_signature(private_jwk.public_jwk())\n\nprint(jwt)\n# eyJhbGciOiJFUzI1NiIsImtpZCI6Im15a2V5In0.eyJzdWIiOiJzb21lX3N1YiIsImNsYWltMSI6InZhbHVlMSJ9.C1KcDyDT8qXwUqcWzPKkQD7f6xai-gCgaRFMdKPe80Vk7XeYNa8ovuLwvdXgGW4ZZ_lL73QIyncY7tHGXUthag\n# This will output the full JWT compact representation. You can inspect it for example at <https://jwt.io>\n\nprint(jwt.headers)\n# {\'alg\': \'ES256\', \'kid\': \'my_key\'}\n```\n\nNote above that the JWT headers are automatically generated with the appropriate values.\n\nOr let\'s sign a JWT with the standardised lifetime, subject, audience and ID claims:\n\n```python\nfrom jwskate import Jwk, JwtSigner\n\nprivate_jwk = Jwk.generate_for_alg("ES256")\nsigner = JwtSigner(issuer="https://myissuer.com", jwk=private_jwk)\njwt = signer.sign(\n    subject="some_sub",\n    audience="some_aud",\n    extra_claims={"custom_claim1": "value1", "custom_claim2": "value2"},\n)\n\nprint(jwt.claims)\n```\n\nThe generated JWT claims will include the standardised claims:\n\n```\n{\'custom_claim1\': \'value1\',\n \'custom_claim2\': \'value2\',\n \'iss\': \'https://myissuer.com\',\n \'aud\': \'some_aud\',\n \'sub\': \'some_sub\',\n \'iat\': 1648823184,\n \'exp\': 1648823244,\n \'jti\': \'3b400e27-c111-4013-84e0-714acd76bf3a\'\n}\n```\n\n## Features\n\n- Simple, Clean, Pythonic interface\n- Convenience wrappers around `cryptography` for all algorithms described in JWA\n- Json Web Keys (JWK) loading and generation\n- Arbitrary data signature and verification using Json Web Keys\n- Json Web Signatures (JWS) signing and verification\n- Json Web Encryption (JWE) encryption and decryption\n- Json Web Tokens (JWT) signing, verification and validation\n- 100% type annotated\n- nearly 100% code coverage\n- Relies on [cryptography](https://cryptography.io) for all cryptographic operations\n- Relies on [BinaPy](https://guillp.github.io/binapy/) for binary data manipulations\n\n### Supported Token Types\n\n| Token Type                | Support                                                                  |\n|---------------------------|--------------------------------------------------------------------------|\n| Json Web Signature (JWS)  | ☑ Compact <br/> ☑ JSON Flat <br/> ☑ JSON General <br/> |\n| Json Web Encryption (JWE) | ☑ Compact <br/> ☐ JSON Flat <br/> ☐ JSON General <br/> |\n| Json Web Tokens (JWT)     | ☑ Signed <br/> ☑ Signed and Encrypted                        |\n\n### Supported Signature algorithms\n\n`jwskate` supports the following signature algorithms:\n\n| Signature Alg | Description                                    | Key Type | Reference                          | Note                  |\n|---------------|------------------------------------------------|----------|------------------------------------|-----------------------|\n| HS256         | HMAC using SHA-256                             | oct      | [RFC7518, Section 3.2]             |                       |\n| HS384         | HMAC using SHA-384                             | oct      | [RFC7518, Section 3.2]             |                       |\n| HS512         | HMAC using SHA-512                             | oct      | [RFC7518, Section 3.2]             |                       |\n| RS256         | RSASSA-PKCS1-v1_5 using SHA-256                | RSA      | [RFC7518, Section 3.3]             |                       |\n| RS384         | RSASSA-PKCS1-v1_5 using SHA-384                | RSA      | [RFC7518, Section 3.3]             |                       |\n| RS512         | RSASSA-PKCS1-v1_5 using SHA-512                | RSA      | [RFC7518, Section 3.3]             |                       |\n| ES256         | ECDSA using P-256 and SHA-256                  | EC       | [RFC7518, Section 3.4]             |                       |\n| ES384         | ECDSA using P-384 and SHA-384                  | EC       | [RFC7518, Section 3.4]             |                       |\n| ES512         | ECDSA using P-521 and SHA-512                  | EC       | [RFC7518, Section 3.4]             |                       |\n| PS256         | RSASSA-PSS using SHA-256 and MGF1 with SHA-256 | RSA      | [RFC7518, Section 3.5]             |                       |\n| PS384         | RSASSA-PSS using SHA-384 and MGF1 with SHA-384 | RSA      | [RFC7518, Section 3.5]             |                       |\n| PS512         | RSASSA-PSS using SHA-512 and MGF1 with SHA-512 | RSA      | [RFC7518, Section 3.5]             |                       |\n| EdDSA         | EdDSA signature algorithms                     | OKP      | [RFC8037, Section 3.1]             |                       |\n| ES256K        | ECDSA using secp256k1 curve and SHA-256        | EC       | [RFC8812, Section 3.2]             |                       |\n| HS1           | HMAC using SHA-1                               | oct      | https://www.w3.org/TR/WebCryptoAPI | Validation Only       |\n| RS1           | RSASSA-PKCS1-v1_5 with SHA-1                   | oct      | https://www.w3.org/TR/WebCryptoAPI | Validation Only       |\n| none          | No digital signature or MAC performed          |          | [RFC7518, Section 3.6]             | Not usable by mistake |\n\n### Supported Key Management algorithms\n\n`jwskate` supports the following key management algorithms:\n\n| Signature Alg      | Description                                     | Key Type | Reference                          | Note        |\n|--------------------|-------------------------------------------------|----------|------------------------------------|-------------|\n| RSA1_5             | RSAES-PKCS1-v1_5                                | RSA      | [RFC7518, Section 4.2]             | Unwrap Only |\n| RSA-OAEP           | RSAES OAEP using default parameters             | RSA      | [RFC7518, Section 4.3]             |             |\n| RSA-OAEP-256       | RSAES OAEP using SHA-256 and MGF1 with SHA-256  | RSA      | [RFC7518, Section 4.3]             |             |\n| RSA-OAEP-384       | RSA-OAEP using SHA-384 and MGF1 with SHA-384    | RSA      | https://www.w3.org/TR/WebCryptoAPI |             |\n| RSA-OAEP-512       | RSA-OAEP using SHA-512 and MGF1 with SHA-512    | RSA      | https://www.w3.org/TR/WebCryptoAPI |             |\n| A128KW             | AES Key Wrap using 128-bit key                  | oct      | [RFC7518, Section 4.4]             |             |\n| A192KW             | AES Key Wrap using 192-bit key                  | oct      | [RFC7518, Section 4.4]             |             |\n| A256KW             | AES Key Wrap using 256-bit key                  | oct      | [RFC7518, Section 4.4]             |             |\n| dir                | Direct use of a shared symmetric key            | oct      | [RFC7518, Section 4.5]             |             |\n| ECDH-ES            | ECDH-ES using Concat KDF                        | EC       | [RFC7518, Section 4.6]             |             |\n| ECDH-ES+A128KW     | ECDH-ES using Concat KDF and "A128KW" wrapping  | EC       | [RFC7518, Section 4.6]             |             |\n| ECDH-ES+A192KW     | ECDH-ES using Concat KDF and "A192KW" wrapping  | EC       | [RFC7518, Section 4.6]             |             |\n| ECDH-ES+A256KW     | ECDH-ES using Concat KDF and "A256KW" wrapping  | EC       | [RFC7518, Section 4.6]             |             |\n| A128GCMKW          | Key wrapping with AES GCM using 128-bit key     | oct      | [RFC7518, Section 4.7]             |             |\n| A192GCMKW          | Key wrapping with AES GCM using 192-bit key     | oct      | [RFC7518, Section 4.7]             |             |\n| A256GCMKW          | Key wrapping with AES GCM using 256-bit key     | oct      | [RFC7518, Section 4.7]             |             |\n| PBES2-HS256+A128KW | PBES2 with HMAC SHA-256 and "A128KW" wrapping   | password | [RFC7518, Section 4.8]             |             |\n| PBES2-HS384+A192KW | PBES2 with HMAC SHA-384 and "A192KW" wrapping   | password | [RFC7518, Section 4.8]             |             |\n| PBES2-HS512+A256KW | PBES2 with HMAC SHA-512 and "A256KW" wrapping   | password | [RFC7518, Section 4.8]             |             |\n\n### Supported Encryption algorithms\n\n`jwskate` supports the following encryption algorithms:\n\n| Signature Alg  | Description                                                 | Reference                |\n|----------------|-------------------------------------------------------------|--------------------------|\n| A128CBC-HS256  | AES_128_CBC_HMAC_SHA_256 authenticated encryption algorithm | [RFC7518, Section 5.2.3] |\n| A192CBC-HS384  | AES_192_CBC_HMAC_SHA_384 authenticated encryption algorithm | [RFC7518, Section 5.2.4] |\n| A256CBC-HS512  | AES_256_CBC_HMAC_SHA_512 authenticated encryption algorithm | [RFC7518, Section 5.2.5] |\n| A128GCM        | AES GCM using 128-bit key                                   | [RFC7518, Section 5.3]   |\n| A192GCM        | AES GCM using 192-bit key                                   | [RFC7518, Section 5.3]   |\n| A256GCM        | AES GCM using 256-bit key                                   | [RFC7518, Section 5.3]   |\n\n### Supported Elliptic Curves\n\n`jwskate` supports the following Elliptic Curves:\n\n| Curve     | Description                           | Key Type | Usage                 | Reference                  |\n|-----------|---------------------------------------|----------|-----------------------|----------------------------|\n| P-256     | P-256 Curve                           | EC       | signature, encryption | [RFC7518, Section 6.2.1.1] |\n| P-384     | P-384 Curve                           | EC       | signature, encryption | [RFC7518, Section 6.2.1.1] |\n| P-521     | P-521 Curve                           | EC       | signature, encryption | [RFC7518, Section 6.2.1.1] |\n| Ed25519   | Ed25519 signature algorithm key pairs | OKP      | signature             | [RFC8037, Section 3.1]     |\n| Ed448     | Ed448 signature algorithm key pairs   | OKP      | signature             | [RFC8037, Section 3.1]     |\n| X25519    | X25519 function key pairs             | OKP      | encryption            | [RFC8037, Section 3.2]     |\n| X448      | X448 function key pairs               | OKP      | encryption            | [RFC8037, Section 3.2]     |\n| secp256k1 | SECG secp256k1 curve                  | EC       | signature, encryption | [RFC8812, Section 3.1]     |\n\n## Why a new lib ?\n\nThere are already multiple modules implementing JOSE and Json Web Crypto related specifications in Python. However, I have\nbeen dissatisfied by all of them so far, so I decided to come up with my own module.\n\n- [PyJWT](https://pyjwt.readthedocs.io)\n- [JWCrypto](https://jwcrypto.readthedocs.io/)\n- [Python-JOSE](https://python-jose.readthedocs.io/)\n- [AuthLib](https://docs.authlib.org/en/latest/jose/)\n\nNot to say that those are _bad_ libs (I actually use `jwcrypto` myself for `jwskate` unit tests), but they either don\'t\nsupport some important features, lack documentation, or generally have APIs that don\'t feel easy-enough, Pythonic-enough to use.\n\n## Design\n\n### JWK are dicts\n\nJWK are specified as JSON objects, which are parsed as `dict` in Python. The `Jwk` class in `jwskate` is actually a\n`dict` subclass, so you can use it exactly like you would use a dict: you can access its members, dump it back as JSON, etc.\nThe same is true for Signed or Encrypted Json Web tokens in JSON format.\n\n### JWA Wrappers\n\nYou can use `cryptography` to do the cryptographic operations that are described in [JWA](https://www.rfc-editor.org/info/rfc7518),\nbut since `cryptography` is a general purpose library, its usage is not straightforward and gives you plenty of options\nto carefully select and combine, leaving room for errors.\nTo work around this, `jwskate` comes with a set of wrappers that implement the exact JWA specifications, with minimum\nrisk of mistakes.\n\n### Safe Signature Verification\n\nFor every signature verification method in `jwskate`, the expected signature(s) algorithm(s) must be specified.\nThat is to avoid a security flaw where your application accepts tokens with a weaker encryption scheme than what\nyour security policy mandates; or even worse, where it accepts unsigned tokens, or tokens that are symmetrically signed\nwith an improperly used public key, leaving your application exposed to exploitation by attackers.\n\nTo specify which signature algorithms are accepted, each signature verification method accepts, in order of preference:\n\n- an `alg` parameter which contains the expected algorithm, or an `algs` parameter which contains a list of acceptable algorithms\n- the `alg` parameter from the signature verification `Jwk`, if present. This `alg` is the algorithm intended for use with that key.\n\nNote that you cannot use `alg` and `algs` at the same time. If your `Jwk` contains an `alg` parameter, and you provide\nan `alg` or `algs` which does not match that value, a `Warning` will be emitted.\n\n## Credits\n\nAll cryptographic operations are handled by [cryptography](https://cryptography.io).\n\n[rfc7515]: https://www.rfc-editor.org/rfc/rfc7515.html\n[rfc7516]: https://www.rfc-editor.org/rfc/rfc7516.html\n[rfc7517]: https://www.rfc-editor.org/rfc/rfc7517.html\n[rfc7518]: https://www.rfc-editor.org/rfc/rfc7518.html\n[rfc7518, section 3.2]: https://www.rfc-editor.org/rfc/rfc7518.html#section-3.2\n[rfc7518, section 3.3]: https://www.rfc-editor.org/rfc/rfc7518.html#section-3.3\n[rfc7518, section 3.4]: https://www.rfc-editor.org/rfc/rfc7518.html#section-3.4\n[rfc7518, section 3.5]: https://www.rfc-editor.org/rfc/rfc7518.html#section-3.5\n[rfc7518, section 3.6]: https://www.rfc-editor.org/rfc/rfc7518.html#section-3.6\n[rfc7518, section 4.2]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.2\n[rfc7518, section 4.3]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.3\n[rfc7518, section 4.4]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.4\n[rfc7518, section 4.5]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.5\n[rfc7518, section 4.6]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.6\n[rfc7518, section 4.7]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.7\n[rfc7518, section 4.8]: https://www.rfc-editor.org/rfc/rfc7518.html#section-4.8\n[rfc7518, section 5.2.3]: https://www.rfc-editor.org/rfc/rfc7518.html#section-5.2.3\n[rfc7518, section 5.2.4]: https://www.rfc-editor.org/rfc/rfc7518.html#section-5.2.4\n[rfc7518, section 5.2.5]: https://www.rfc-editor.org/rfc/rfc7518.html#section-5.2.5\n[rfc7518, section 5.3]: https://www.rfc-editor.org/rfc/rfc7518.html#section-5.3\n[rfc7518, section 6.2.1.1]: https://www.rfc-editor.org/rfc/rfc7518.html#section-6.2.1.1\n[rfc7519]: https://www.rfc-editor.org/rfc/rfc7519.html\n[rfc7638]: https://www.rfc-editor.org/rfc/rfc7638.html\n[rfc8037]: https://www.rfc-editor.org/rfc/rfc8037.html\n[rfc8037, section 3.1]: https://www.rfc-editor.org/rfc/rfc8037.html#section-3.1\n[rfc8037, section 3.2]: https://www.rfc-editor.org/rfc/rfc8037.html#section-3.2\n[rfc8812, section 3.1]: https://www.rfc-editor.org/rfc/rfc8812.html#section-3.1\n[rfc8812, section 3.2]: https://www.rfc-editor.org/rfc/rfc8812.html#name-ecdsa-signature-with-secp25\n[rfc9278]: https://www.rfc-editor.org/rfc/rfc9278.html\n',
    'author': 'Guillaume Pujol',
    'author_email': 'guill.p.linux@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/guillp/jwskate',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
