import * as cdk from '@aws-cdk/core';
import { CfnRoute } from './appmesh.generated';
import { HeaderMatch } from './header-match';
import { HttpRouteMethod } from './http-route-method';
import { HttpRoutePathMatch } from './http-route-path-match';
import { QueryParameterMatch } from './query-parameter-match';
import { GrpcTimeout, HttpTimeout, TcpTimeout } from './shared-interfaces';
import { IVirtualNode } from './virtual-node';
import { Construct } from '@aws-cdk/core';
/**
 * Properties for the Weighted Targets in the route.
 *
 * @stability stable
 */
export interface WeightedTarget {
    /**
     * The VirtualNode the route points to.
     *
     * @stability stable
     */
    readonly virtualNode: IVirtualNode;
    /**
     * The weight for the target.
     *
     * @default 1
     * @stability stable
     */
    readonly weight?: number;
}
/**
 * The criterion for determining a request match for this Route.
 *
 * @stability stable
 */
export interface HttpRouteMatch {
    /**
     * Specifies how is the request matched based on the path part of its URL.
     *
     * @default - matches requests with all paths
     * @stability stable
     */
    readonly path?: HttpRoutePathMatch;
    /**
     * Specifies the client request headers to match on.
     *
     * All specified headers
     * must match for the route to match.
     *
     * @default - do not match on headers
     * @stability stable
     */
    readonly headers?: HeaderMatch[];
    /**
     * The HTTP client request method to match on.
     *
     * @default - do not match on request method
     * @stability stable
     */
    readonly method?: HttpRouteMethod;
    /**
     * The client request protocol to match on.
     *
     * Applicable only for HTTP2 routes.
     *
     * @default - do not match on HTTP2 request protocol
     * @stability stable
     */
    readonly protocol?: HttpRouteProtocol;
    /**
     * The query parameters to match on.
     *
     * All specified query parameters must match for the route to match.
     *
     * @default - do not match on query parameters
     * @stability stable
     */
    readonly queryParameters?: QueryParameterMatch[];
}
/**
 * Supported :scheme options for HTTP2.
 *
 * @stability stable
 */
export declare enum HttpRouteProtocol {
    /**
     * Match HTTP requests.
     *
     * @stability stable
     */
    HTTP = "http",
    /**
     * Match HTTPS requests.
     *
     * @stability stable
     */
    HTTPS = "https"
}
/**
 * The criterion for determining a request match for this Route.
 *
 * At least one match type must be selected.
 *
 * @stability stable
 */
export interface GrpcRouteMatch {
    /**
     * Create service name based gRPC route match.
     *
     * @default - do not match on service name
     * @stability stable
     */
    readonly serviceName?: string;
    /**
     * Create metadata based gRPC route match.
     *
     * All specified metadata must match for the route to match.
     *
     * @default - do not match on metadata
     * @stability stable
     */
    readonly metadata?: HeaderMatch[];
    /**
     * The method name to match from the request.
     *
     * If the method name is specified, service name must be also provided.
     *
     * @default - do not match on method name
     * @stability stable
     */
    readonly methodName?: string;
}
/**
 * Base options for all route specs.
 *
 * @stability stable
 */
export interface RouteSpecOptionsBase {
    /**
     * The priority for the route.
     *
     * When a Virtual Router has multiple routes, route match is performed in the
     * order of specified value, where 0 is the highest priority, and first matched route is selected.
     *
     * @default - no particular priority
     * @stability stable
     */
    readonly priority?: number;
}
/**
 * Properties specific for HTTP Based Routes.
 *
 * @stability stable
 */
export interface HttpRouteSpecOptions extends RouteSpecOptionsBase {
    /**
     * The criterion for determining a request match for this Route.
     *
     * @default - matches on '/'
     * @stability stable
     */
    readonly match?: HttpRouteMatch;
    /**
     * List of targets that traffic is routed to when a request matches the route.
     *
     * @stability stable
     */
    readonly weightedTargets: WeightedTarget[];
    /**
     * An object that represents a http timeout.
     *
     * @default - None
     * @stability stable
     */
    readonly timeout?: HttpTimeout;
    /**
     * The retry policy.
     *
     * @default - no retry policy
     * @stability stable
     */
    readonly retryPolicy?: HttpRetryPolicy;
}
/**
 * HTTP retry policy.
 *
 * @stability stable
 */
export interface HttpRetryPolicy {
    /**
     * Specify HTTP events on which to retry.
     *
     * You must specify at least one value
     * for at least one types of retry events.
     *
     * @default - no retries for http events
     * @stability stable
     */
    readonly httpRetryEvents?: HttpRetryEvent[];
    /**
     * The maximum number of retry attempts.
     *
     * @stability stable
     */
    readonly retryAttempts: number;
    /**
     * The timeout for each retry attempt.
     *
     * @stability stable
     */
    readonly retryTimeout: cdk.Duration;
    /**
     * TCP events on which to retry.
     *
     * The event occurs before any processing of a
     * request has started and is encountered when the upstream is temporarily or
     * permanently unavailable. You must specify at least one value for at least
     * one types of retry events.
     *
     * @default - no retries for tcp events
     * @stability stable
     */
    readonly tcpRetryEvents?: TcpRetryEvent[];
}
/**
 * HTTP events on which to retry.
 *
 * @stability stable
 */
export declare enum HttpRetryEvent {
    /**
     * HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511.
     *
     * @stability stable
     */
    SERVER_ERROR = "server-error",
    /**
     * HTTP status codes 502, 503, and 504.
     *
     * @stability stable
     */
    GATEWAY_ERROR = "gateway-error",
    /**
     * HTTP status code 409.
     *
     * @stability stable
     */
    CLIENT_ERROR = "client-error",
    /**
     * Retry on refused stream.
     *
     * @stability stable
     */
    STREAM_ERROR = "stream-error"
}
/**
 * TCP events on which you may retry
 */
export declare enum TcpRetryEvent {
    /**
     * TCP events on which you may retry.
     *
     * @stability stable
     */
    CONNECTION_ERROR = "connection-error"
}
/**
 * Properties specific for a TCP Based Routes.
 *
 * @stability stable
 */
export interface TcpRouteSpecOptions extends RouteSpecOptionsBase {
    /**
     * List of targets that traffic is routed to when a request matches the route.
     *
     * @stability stable
     */
    readonly weightedTargets: WeightedTarget[];
    /**
     * An object that represents a tcp timeout.
     *
     * @default - None
     * @stability stable
     */
    readonly timeout?: TcpTimeout;
}
/**
 * Properties specific for a GRPC Based Routes.
 *
 * @stability stable
 */
export interface GrpcRouteSpecOptions extends RouteSpecOptionsBase {
    /**
     * The criterion for determining a request match for this Route.
     *
     * @stability stable
     */
    readonly match: GrpcRouteMatch;
    /**
     * An object that represents a grpc timeout.
     *
     * @default - None
     * @stability stable
     */
    readonly timeout?: GrpcTimeout;
    /**
     * List of targets that traffic is routed to when a request matches the route.
     *
     * @stability stable
     */
    readonly weightedTargets: WeightedTarget[];
    /**
     * The retry policy.
     *
     * @default - no retry policy
     * @stability stable
     */
    readonly retryPolicy?: GrpcRetryPolicy;
}
/**
 * gRPC retry policy.
 *
 * @stability stable
 */
export interface GrpcRetryPolicy extends HttpRetryPolicy {
    /**
     * gRPC events on which to retry.
     *
     * You must specify at least one value
     * for at least one types of retry events.
     *
     * @default - no retries for gRPC events
     * @stability stable
     */
    readonly grpcRetryEvents?: GrpcRetryEvent[];
}
/**
 * gRPC events.
 *
 * @stability stable
 */
export declare enum GrpcRetryEvent {
    /**
     * Request was cancelled.
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     * @stability stable
     */
    CANCELLED = "cancelled",
    /**
     * The deadline was exceeded.
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     * @stability stable
     */
    DEADLINE_EXCEEDED = "deadline-exceeded",
    /**
     * Internal error.
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     * @stability stable
     */
    INTERNAL_ERROR = "internal",
    /**
     * A resource was exhausted.
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     * @stability stable
     */
    RESOURCE_EXHAUSTED = "resource-exhausted",
    /**
     * The service is unavailable.
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     * @stability stable
     */
    UNAVAILABLE = "unavailable"
}
/**
 * All Properties for Route Specs.
 *
 * @stability stable
 */
export interface RouteSpecConfig {
    /**
     * The spec for an http route.
     *
     * @default - no http spec
     * @stability stable
     */
    readonly httpRouteSpec?: CfnRoute.HttpRouteProperty;
    /**
     * The spec for an http2 route.
     *
     * @default - no http2 spec
     * @stability stable
     */
    readonly http2RouteSpec?: CfnRoute.HttpRouteProperty;
    /**
     * The spec for a grpc route.
     *
     * @default - no grpc spec
     * @stability stable
     */
    readonly grpcRouteSpec?: CfnRoute.GrpcRouteProperty;
    /**
     * The spec for a tcp route.
     *
     * @default - no tcp spec
     * @stability stable
     */
    readonly tcpRouteSpec?: CfnRoute.TcpRouteProperty;
    /**
     * The priority for the route.
     *
     * When a Virtual Router has multiple routes, route match is performed in the
     * order of specified value, where 0 is the highest priority, and first matched route is selected.
     *
     * @default - no particular priority
     * @stability stable
     */
    readonly priority?: number;
}
/**
 * Used to generate specs with different protocols for a RouteSpec.
 *
 * @stability stable
 */
export declare abstract class RouteSpec {
    /**
     * Creates an HTTP Based RouteSpec.
     *
     * @stability stable
     */
    static http(options: HttpRouteSpecOptions): RouteSpec;
    /**
     * Creates an HTTP2 Based RouteSpec.
     *
     * @stability stable
     */
    static http2(options: HttpRouteSpecOptions): RouteSpec;
    /**
     * Creates a TCP Based RouteSpec.
     *
     * @stability stable
     */
    static tcp(options: TcpRouteSpecOptions): RouteSpec;
    /**
     * Creates a GRPC Based RouteSpec.
     *
     * @stability stable
     */
    static grpc(options: GrpcRouteSpecOptions): RouteSpec;
    /**
     * Called when the RouteSpec type is initialized.
     *
     * Can be used to enforce
     * mutual exclusivity with future properties
     *
     * @stability stable
     */
    abstract bind(scope: Construct): RouteSpecConfig;
}
