"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const k8s_manifest_1 = require("./k8s-manifest");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 * @stability stable
 */
class AwsAuth extends core_2.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_manifest_1.KubernetesManifest(this, 'manifest', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add.
     * @param username Optional user (defaults to the role ARN).
     * @stability stable
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map.
     * @param mapping Mapping to k8s user name and groups.
     * @stability stable
     */
    addRoleMapping(role, mapping) {
        this.assertSameStack(role);
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map.
     * @param mapping Mapping to k8s user name and groups.
     * @stability stable
     */
    addUserMapping(user, mapping) {
        this.assertSameStack(user);
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     *
     * @param accountId account number.
     * @stability stable
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    assertSameStack(construct) {
        const thisStack = core_1.Stack.of(this);
        if (core_1.Stack.of(construct) !== thisStack) {
            // aws-auth is always part of the cluster stack, and since resources commonly take
            // a dependency on the cluster, allowing those resources to be in a different stack,
            // will create a circular dependency. granted, it won't always be the case,
            // but we opted for the more causious and restrictive approach for now.
            throw new Error(`${construct.node.path} should be defined in the scope of the ${thisStack.stackName} stack to prevent circular dependencies`);
        }
    }
    synthesizeMapRoles() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _b;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_b = m.mapping.username) !== null && _b !== void 0 ? _b : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _b;
                return ({
                    userarn: m.user.userArn,
                    username: (_b = m.mapping.username) !== null && _b !== void 0 ? _b : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
_a = JSII_RTTI_SYMBOL_1;
AwsAuth[_a] = { fqn: "@aws-cdk/aws-eks.AwsAuth", version: "1.93.0" };
//# sourceMappingURL=data:application/json;base64,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