"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Semaphore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
/**
 * Generates a semaphore for a StepFunction job (or chained set of jobs) to limit parallelism across executions.
 */
class Semaphore extends aws_stepfunctions_1.StateMachineFragment {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The DynamoDB table used to store semaphores.
         */
        this.tableName = 'StateMachineSempahoreTable920751a65a584e8ab7583460f6db686a';
        const stackTracker = this.setUpMap();
        this.semaphoreTable = this.ensureTable(props);
        let lockInfo = stackTracker.get(props.lockName);
        if (lockInfo) {
            if (props.reuseLock) {
                if (lockInfo.limit != props.limit) {
                    throw new Error(`The reused \`lockName\` "${props.lockName}" was given a different \`limit\` than previously defined. Given: ${props.limit}, Previous: ${lockInfo.limit}.`);
                }
                else {
                    lockInfo = { limit: lockInfo.limit, timesUsed: lockInfo.timesUsed + 1 };
                    stackTracker.set(props.lockName, lockInfo);
                }
            }
            else {
                throw new Error(`The \`lockName\` "${props.lockName}" was reused without explicitly allowing reuse. Set \`reuseLock\` to \`true\` if you want to reuse the lock.`);
            }
        }
        else {
            lockInfo = { limit: props.limit, timesUsed: 1 };
            stackTracker.set(props.lockName, lockInfo);
        }
        const getLock = new aws_stepfunctions_1.Parallel(this, `Get ${props.lockName} Lock: ${lockInfo.timesUsed}`, { resultPath: aws_stepfunctions_1.JsonPath.DISCARD });
        const acquireLock = new aws_stepfunctions_tasks_1.DynamoUpdateItem(this, `Acquire ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `Acquire a lock using a conditional update to DynamoDB. This update will do two things:
          1) increment a counter for the number of held locks
          2) add an attribute to the DynamoDB Item with a unique key for this execution and with a value of the time when the lock was Acquired
          The Update includes a conditional expression that will fail under two circumstances:
          1) if the maximum number of locks have already been distributed
          2) if the current execution already owns a lock. The latter check is important to ensure the same execution doesn't increase the counter more than once
          If either of these conditions are not met, then the task will fail with a DynamoDB.ConditionalCheckFailedException error, retry a few times, then if it is still not successful \
          it will move off to another branch of the workflow. If this is the first time that a given lockname has been used, there will not be a row in DynamoDB \
          so the update will fail with DynamoDB.AmazonDynamoDBException. In that case, this state sends the workflow to state that will create that row to initialize.
          ` : undefined,
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: {
                '#currentlockcount': 'currentlockcount',
                '#lockownerid.$': '$$.Execution.Id',
            },
            expressionAttributeValues: {
                ':increase': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(1),
                ':limit': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(props.limit),
                ':lockacquiredtime': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(aws_stepfunctions_1.JsonPath.stringAt('$$.State.EnteredTime')),
            },
            updateExpression: 'SET #currentlockcount = #currentlockcount + :increase, #lockownerid = :lockacquiredtime',
            conditionExpression: 'currentlockcount <> :limit and attribute_not_exists(#lockownerid)',
            returnValues: aws_stepfunctions_tasks_1.DynamoReturnValues.UPDATED_NEW,
            resultPath: '$.lockinfo.acquirelock',
        });
        const initializeLockItem = new aws_stepfunctions_tasks_1.DynamoPutItem(this, `Initialize ${props.lockName} Lock Item: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `This state handles the case where an item hasn't been created for this lock yet. \
      In that case, it will insert an initial item that includes the lock name as the key and currentlockcount of 0. \ 
      The Put to DynamoDB includes a conditonal expression to fail if the an item with that key already exists, which avoids a race condition if multiple executions start at the same time. \ 
      There are other reasons that the previous state could fail and end up here, so this is safe in those cases too.` : undefined,
            table: this.semaphoreTable,
            item: {
                LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName),
                currentlockcount: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(0),
            },
            conditionExpression: 'LockName <> :lockname',
            expressionAttributeValues: {
                ':lockname': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName),
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const getCurrentLockRecord = new aws_stepfunctions_tasks_1.DynamoGetItem(this, `Get Current ${props.lockName} Lock Record: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'This state is called when the execution is unable to acquire a lock because there limit has either been exceeded or because this execution already holds a lock. \
      In that case, this task loads info from DDB for the current lock item so that the right decision can be made in subsequent states.' : undefined,
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: { '#lockownerid.$': '$$.Execution.Id' },
            projectionExpression: [new aws_stepfunctions_tasks_1.DynamoProjectionExpression().withAttribute('#lockownerid')],
            resultSelector: {
                'Item.$': '$.Item',
                'ItemString.$': 'States.JsonToString($.Item)',
            },
            resultPath: '$.lockinfo.currentlockitem',
        });
        const checkIfLockAcquired = new aws_stepfunctions_1.Choice(this, `Check if ${props.lockName} Lock Already Acquired: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `This state checks to see if the current execution already holds a lock. It can tell that by looking for Z, which will be indicative of the timestamp value. \ 
      That will only be there in the stringified version of the data returned from DDB if this execution holds a lock.` : undefined,
        });
        const continueBecauseLockWasAlreadyAcquired = new aws_stepfunctions_1.Pass(this, `Continue Because ${props.lockName} Lock Was Already Acquired: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'In this state, we have confimed that lock is already held, so we pass the original execution input into the the function that does the work.' : undefined,
        });
        const waitToGetLock = new aws_stepfunctions_1.Wait(this, `Wait to Get ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'If the lock indeed not been succesfully Acquired, then wait for a bit before trying again.' : undefined,
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(3)),
        });
        acquireLock.addRetry({ errors: ['DynamoDB.AmazonDynamoDBException'], maxAttempts: 0 })
            .addRetry({ maxAttempts: 6, backoffRate: 2 })
            .addCatch(initializeLockItem, { errors: ['DynamoDB.AmazonDynamoDBException'], resultPath: '$.lockinfo.acquisitionerror' })
            .addCatch(getCurrentLockRecord, { errors: ['DynamoDB.ConditionalCheckFailedException'], resultPath: '$.lockinfo.acquisitionerror' });
        initializeLockItem.addCatch(acquireLock, { resultPath: aws_stepfunctions_1.JsonPath.DISCARD });
        getCurrentLockRecord.next(checkIfLockAcquired);
        checkIfLockAcquired.when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.isPresent('$.lockinfo.currentlockitem.ItemString'), aws_stepfunctions_1.Condition.stringMatches('$.lockinfo.currentlockitem.ItemString', '*Z')), continueBecauseLockWasAlreadyAcquired);
        checkIfLockAcquired.otherwise(waitToGetLock);
        waitToGetLock.next(acquireLock);
        const releaseLock = new aws_stepfunctions_tasks_1.DynamoUpdateItem(this, `Release ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: {
                '#currentlockcount': 'currentlockcount',
                '#lockownerid.$': '$$.Execution.Id',
            },
            expressionAttributeValues: {
                ':decrease': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(1),
            },
            updateExpression: 'SET #currentlockcount = #currentlockcount - :decrease REMOVE #lockownerid',
            conditionExpression: 'attribute_exists(#lockownerid)',
            returnValues: aws_stepfunctions_tasks_1.DynamoReturnValues.UPDATED_NEW,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        releaseLock.addRetry({ errors: ['DynamoDB.ConditionalCheckFailedException'], maxAttempts: 0 })
            .addRetry({ maxAttempts: 5, backoffRate: 1.5 })
            .addCatch(props.nextState, { errors: ['DynamoDB.ConditionalCheckFailedException'], resultPath: '$.lockinfo.acquisitionerror' })
            .next(props.nextState);
        getLock.branch(acquireLock);
        getLock.endStates.forEach(j => j.next(props.job));
        props.job.next(releaseLock);
        this.startState = getLock;
        this.endStates = props.nextState.endStates;
    }
    setUpMap() {
        const stackId = aws_cdk_lib_1.Names.uniqueId(aws_cdk_lib_1.Stack.of(this));
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(this.tableName);
        if (existing) {
            return (Semaphore.semaphoreTracker.get(stackId));
        }
        else {
            const m = new Map();
            Semaphore.semaphoreTracker.set(stackId, m);
            return m;
        }
    }
    ensureTable(props) {
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(this.tableName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        else {
            return new aws_dynamodb_1.Table(aws_cdk_lib_1.Stack.of(this), this.tableName, {
                partitionKey: {
                    name: 'LockName',
                    type: aws_dynamodb_1.AttributeType.STRING,
                },
                readCapacity: props.tableReadWriteCapacity?.readCapacity,
                writeCapacity: props.tableReadWriteCapacity?.writeCapacity,
                billingMode: props.tableReadWriteCapacity ? aws_dynamodb_1.BillingMode.PROVISIONED : aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            });
        }
    }
}
exports.Semaphore = Semaphore;
_a = JSII_RTTI_SYMBOL_1;
Semaphore[_a] = { fqn: "@dontirun/state-machine-semaphore.Semaphore", version: "0.1.0" };
/**
 * The names and associated concurrency limits and number of uses of the sempahores.
 */
Semaphore.semaphoreTracker = new Map();
//# sourceMappingURL=data:application/json;base64,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