# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['weo']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.22,<0.24', 'iso3166>=2.0.2,<3.0.0', 'pandas>=1.3.5']

setup_kwargs = {
    'name': 'weo',
    'version': '0.7.4',
    'description': 'Python client to download IMF World Economic Outlook (WEO) dataset as pandas dataframes.',
    'long_description': '# weo-reader\n\n[![PyPI](https://img.shields.io/pypi/v/weo)](https://pypi.org/project/weo/)\n[![pytest](https://github.com/epogrebnyak/weo-reader/workflows/pytest/badge.svg)](https://github.com/epogrebnyak/weo-reader/actions)\n[![Downloads](https://pepy.tech/badge/weo/week)](https://pepy.tech/project/weo/week)\n\n[![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/drive/1Dvl1N3c9uRUEulm-Kto9zqW_VCj0YtzV)\n[![Open in Streamlit](https://static.streamlit.io/badges/streamlit_badge_black_white.svg)](https://share.streamlit.io/epogrebnyak/weo-reader)\n\n\nThis is a Python client to download [IMF World Economic Outlook Report][weo] dataset as [pandas](https://pandas.pydata.org/) dataframes by release dates. You can explore:\n\n- single country macroeconomic data and forecast,\n- macro variables across countries for a given year,\n- country-year panel for single macro variable.\n\n## Dataset releases (vintages)\n\nDataset releases (vintages) are available back to 2007, the reported data goes back to 1980, forecast is three years ahead.\n\n| Release          | Date         |\n| :--------------- | :-----------:|\n| Latest confirmed | April 2022   |\n| First            | October 2007 |\n\nConfirmed release is tested to be processed with `weo`.\nUsually, if something breaks in a new release users raise an [issue here](https://github.com/epogrebnyak/weo-reader/issues).\n\n[weo]: https://www.imf.org/en/Publications/WEO\n\n![изображение](https://user-images.githubusercontent.com/9265326/103473902-8c64da00-4dae-11eb-957c-4737f56abdce.png)\n\n## Install\n\nThe program is tested to run with Python 3.7.13 (as in Google Colab) and higher.\n\nTo install:\n\n```\npip install weo\n```\n\nLatest version:\n\n```\npip install git+https://github.com/epogrebnyak/weo-reader.git\n```\n\n## First glance\n\nGet US inflation forecast from April 2022 semiannual WEO release.\n\n```python\nfrom weo import download, WEO\n\npath, url = download(2022, 1)\n# weo_2022_1.csv 18.8Mb\n# Downloaded 2022-Apr WEO dataset\n\ndf_cpi = WEO(path).inflation()\nprint(df_cpi.USA.tail(8))\n#         USA\n# 2020  1.549\n# 2021  7.426\n# 2022  5.329\n# 2023  2.337\n# 2024  2.096\n# 2025  1.970\n# 2026  1.983\n# 2027  2.017\n```\n\n## Step 1. Download data\n\nSave data from IMF web site as local file. Specify year\nand release:\n\n```python\nimport weo\n\nweo.download(year=2020, release="Oct", filename="weo.csv")\n```\n\n- You can access WEO releases starting October 2007 with this client.\n- WEO is normally released in April and October, one exception is September 2011.\n- Release is referenced by:\n  - number `1` or `2`;\n  - month `\'Apr\'` or `\'Oct\'`, and `\'Sep\'` in 2011.\n\nYour can list all years and releases available for download with `weo.all_releases()`.\nCombine to create local dataset of WEO vintages from 2007 to present:\n\n```python\nimport pathlib\nimport weo\n\n# create folder\npathlib.Path("weo_data").mkdir(parents=False, exist_ok=True)\n\n# download all releases\nfor (year, release) in weo.all_releases():\n  weo.download(year, release, directory="weo_data")\n```\n\n## Step 2. Inspect data\n\nUse `WEO` class to view and extract data. `WEO` is a wrapper around a pandas dataframe that ensures proper data import and easier access and slicing of data across time-country-variable dimensions.\n\nTry code below:\n\n```python\nfrom weo import WEO\n\nw = WEO("weo.csv")\n```\n\nWhat variables and measurements are inside?\n\n```python\n# variable listing\nw.variables()\n\n# units\nw.units()\nw.units("Gross domestic product, current prices")\n\n# variable codes\nw.codes\nw.from_code("LUR")\n\n# countries\nw.countries("United")      # Dataframe with United Arab Emirates, United Kingdom\n                           # and United States\nw.iso_code3("Netherlands") # \'NLD\'\n```\n\nThe dataset is year-country-variable-value cube, you can fix any dimension to get a table.\n\n```python\n\nw.get("General government gross debt", "Percent of GDP")\nw.getc("NGDP_RPCH")\nw.country("DEU")\nw.fix_year(1994)\n```\n\nPlot a chart with the projected 12 largest economies in 2024 (current prices):\n\n```python\n(w.gdp_usd(2024)\n  .dropna()\n  .sort_values()\n  .tail(12)\n  .plot\n  .barh(title="GDP by country, USD billion (2024)")\n)\n```\n\nGet GDP per capita data from 2000 to 2020:\n\n```python\nw.gdp_pc_usd(start_year=2000, end_year=2020)\n```\n\n## Code documentation\n\n`weo` package documentation is [here](https://epogrebnyak.github.io/weo-reader/).\n\n## Alternative data sources\n\n1\\. If you need the latest data as time series and not the vintages of WEO releases, and you know variables that you are looking for, DBnomics is a good choice:\n\n- <https://db.nomics.world/IMF/WEO>\n- <https://db.nomics.world/IMF/WEOAGG>\n\nExample:\n\n```python\nfrom dbnomics import fetch_series_by_api_link\nts1 = fetch_series_by_api_link("https://api.db.nomics.world/v22/"\n                               "series/IMF/WEO:latest/DEU.PCPI"\n                               "?observations=1")\n```\n\n[![dbnomics](https://user-images.githubusercontent.com/9265326/168478113-00fb4d3f-11c3-43ad-9c19-28e2204f89c1.png)](https://db.nomics.world/IMF/WEO:2021-10/DEU.PCPI.idx)\n\nMore on DBnomics:\n\n- [DBnomics Web API](https://db.nomics.world/docs/web-api/)\n- [Introduction to DBnomics in Python](https://notes.quantecon.org/submission/5bd32515f966080015bafbcd)\n\n2\\. Similar dataset, not updated since 2018, but with earlier years than `weo-reader`:\nhttps://github.com/datasets/imf-weo\n\n## Development notes\n\n- You can download the WEO file in command line with `curl` command:\n\n```\ncurl -o weo.csv https://www.imf.org/-/media/Files/Publications/WEO/WEO-Database/2020/02/WEOOct2020all.xls\n```\n\n- `WEOOct2020all.xls` from the web site is really a CSV file, not an Excel file.\n- There is an update of GDP figures in [June 2020](jun2020), but the file structure is incompatible with regular releases.\n- Prior to 2020 the URL structure was similar to `https://www.imf.org/external/pubs/ft/weo/2019/02/weodata/WEOOct2019all.xls`\n\n[jun2020]: https://www.imf.org/en/Publications/WEO/Issues/2020/06/24/WEOUpdateJune2020\n',
    'author': 'Evgeny Pogrebnyak',
    'author_email': 'e.pogrebnyak@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/epogrebnyak/weo-reader',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.13',
}


setup(**setup_kwargs)
