# -*- coding: utf-8 -*-
'''
*FormFactors: Scattering Data*
------------------------------

Contains atomic scattering factors from 
"New Analytical Scattering Factor Functions for Free Atoms
and Ions for Free Atoms and Ions", 
D. Waasmaier & A. Kirfel,
*Acta Cryst.* **(1995).** A51, 416-413.

Also, tabulated coefficients for calculation of Compton Cross Section as a function of sin(theta)/lambda from
"Analytic Approximations to Incoherently Scattered X-Ray Intensities", 
H. H. M. Balyuzi, *Acta Cryst.* **(1975).** A31, 600.

'''
#  
#  The atomic scattering factor is calculated using the method developed by  D. Waasmaier & A. Kirfel
#
#          New Analytical Scattering Factor Functions for Free Atoms
#                    and Ions for Free Atoms and Ions
#                       D. Waasmaier & A. Kirfel
#                    Acta Cryst. (1995). A51, 416-413
#
#  fo the non-dispersive part of the atomic scattering factor is a
#  function of the selected element and of sin(theta)/lambda, where          
#  lambda is the photon wavelengh and theta is incident angle.               
#  This function can be approximated by a function:                          
#                                                                          
#    f0[k] = c + [SUM a_i*EXP(-b_i*(k^2)) ]                                  
#                i=1,5                                                       
#                                                                          
#  where k = sin(theta) / lambda and c, a_i and b_i                          
#  are the coefficients tabulated in this file (in columns:                  
#  a1  a2  a3  a4  a5  c  b1  b2  b3  b4  b5                                 
#
FFac5term = {
    'H' :[     0.413048,  0.294953,  0.187491,  0.080701,   0.023736,   0.000049, 15.569946, 32.398468,  5.711404,  61.889874,   1.334118],
    'He':[     0.732354,  0.753896,  0.283819,  0.190003,   0.039139,   0.000487, 11.553918,  4.595831,  1.546299,  26.463964,   0.377523],
    'Li':[     0.974637,  0.158472,  0.811855,  0.262416,   0.790108,   0.002542,  4.334946,  0.342451, 97.102966, 201.363831,   1.409234],
    'Be':[     1.533712,  0.638283,  0.601052,  0.106139,   1.118414,   0.002511, 42.662079,  0.595420, 99.106499,   0.151340,   1.843093],
    'B' :[     2.085185,  1.064580,  1.062788,  0.140515,   0.641784,   0.003823, 23.494068,  1.137894, 61.238976,   0.114886,   0.399036],
    'C' :[     2.657506,  1.078079,  1.490909, -4.241070,   0.713791,   4.297983, 14.780758,  0.776775, 42.086842,  -0.000294,   0.239535],
    'N' :[    11.893780,  3.277479,  1.858092,  0.858927,   0.912985,  -11.80490,  0.000158, 10.232723, 30.344690,   0.656065,   0.217287],
    'O' :[     2.960427,  2.508818,  0.637853,  0.722838,   1.142756,   0.027014, 14.182259,  5.936858,  0.112726,  34.958481,   0.390240],
    'F' :[     3.511943,  2.772244,  0.678385,  0.915159,   1.089261,   0.032557, 10.687859,  4.380466,  0.093982,  27.255203,   0.313066],
    'Ne':[     4.183749,  2.905726,  0.520513,  1.135641,   1.228065,   0.025576,  8.175457,  3.252536,  0.063295,  21.813910,   0.224952],
    'Na':[     4.910127,  3.081783,  1.262067,  1.098938,   0.560991,   0.079712,  3.281434,  9.119178,  0.102763, 132.013947,   0.405878],
    'Mg':[     4.708971,  1.194814,  1.558157,  1.170413,   3.239403,   0.126842,  4.875207,108.506081,  0.111516,  48.292408,   1.928171],
    'Al':[     4.730796,  2.313951,  1.541980,  1.117564,   3.154754,   0.139509,  3.628931, 43.051167,  0.095960, 108.932388,   1.555918],
    'Si':[     5.275329,  3.191038,  1.511514,  1.356849,   2.519114,   0.145073,  2.631338, 33.730728,  0.081119,  86.288643,   1.170087],
    'P' :[     1.950541,  4.146930,  1.494560,  1.522042,   5.729711,   0.155233,  0.908139, 27.044952,  0.071280,  67.520187,   1.981173],
    'S' :[     6.372157,  5.154568,  1.473732,  1.635073,   1.209372,   0.154722,  1.514347, 22.092527,  0.061373,  55.445175,   0.646925],
    'Cl':[     1.446071,  6.870609,  6.151801,  1.750347,   0.634168,   0.146773,  0.052357,  1.193165, 18.343416,  46.398396,   0.401005],
    'Ar':[     7.188004,  6.638454,  0.454180,  1.929593,   1.523654,   0.265954,  0.956221, 15.339877, 15.339862,  39.043823,   0.062409],
    'K' :[     8.163991,  7.146945,  1.070140,  0.877316,   1.486434,   0.253614, 12.816323,  0.808945,210.327011,  39.597652,   0.052821],
    'Ca':[     8.593655,  1.477324,  1.436254,  1.182839,   7.113258,   0.196255, 10.460644,  0.041891, 81.390381, 169.847839,   0.688098],
    'Sc':[     1.476566,  1.487278,  1.600187,  9.177463,   7.099750,   0.157765, 53.131023,  0.035325,137.319489,   9.098031,   0.602102],
    'Ti':[     9.818524,  1.522646,  1.703101,  1.768774,   7.082555,   0.102473,  8.001879,  0.029763, 39.885422, 120.157997,   0.532405],
    'V' :[    10.473575,  1.547881,  1.986381,  1.865616,   7.056250,   0.067744,  7.081940,  0.026040, 31.909672, 108.022842,   0.474882],
    'Cr':[    11.007069,  1.555477,  2.985293,  1.347855,   7.034779,   0.065510,  6.366281,  0.023987, 23.244839, 105.774498,   0.429369],
    'Mn':[    11.709542,  1.733414,  2.673141,  2.023368,   7.003180,  -0.147293,  5.597120,  0.017800, 21.788420,  89.517914,   0.383054],
    'Fe':[    12.311098,  1.876623,  3.066177,  2.070451,   6.975185,  -0.304931,  5.009415,  0.014461, 18.743040,  82.767876,   0.346506],
    'Co':[    12.914510,  2.481908,  3.466894,  2.106351,   6.960892,  -0.936572,  4.507138,  0.009126, 16.438129,  76.987320,   0.314418],
    'Ni':[    13.521865,  6.947285,  3.866028,  2.135900,   4.284731,  -2.762697,  4.077277,  0.286763, 14.622634,  71.966080,   0.004437],
    'Cu':[    14.014192,  4.784577,  5.056806,  1.457971,   6.932996,  -3.254477,  3.738280,  0.003744, 13.034982,  72.554794,   0.265666],
    'Zn':[    14.741002,  6.907748,  4.642337,  2.191766,  38.424042, -36.915829,  3.388232,  0.243315, 11.903689,  63.312130,   0.000397],
    'Ga':[    15.758946,  6.841123,  4.121016,  2.714681,   2.395246,  -0.847395,  3.121754,  0.226057, 12.482196,  66.203621,   0.007238],
    'Ge':[    16.540613,  1.567900,  3.727829,  3.345098,   6.785079,   0.018726,  2.866618,  0.012198, 13.432163,  58.866047,   0.210974],
    'As':[    17.025642,  4.503441,  3.715904,  3.937200,   6.790175,  -2.984117,  2.597739,  0.003012, 14.272119,  50.437996,   0.193015],
    'Se':[    17.354071,  4.653248,  4.259489,  4.136455,   6.749163,  -3.160982,  2.349787,  0.002550, 15.579460,  45.181202,   0.177432],
    'Br':[    17.550570,  5.411882,  3.937180,  3.880645,   6.707793,  -2.492088,  2.119226, 16.557184,  0.002481,  42.164009,   0.162121],
    'Kr':[    17.655279,  6.848105,  4.171004,  3.446760,   6.685200,  -2.810592,  1.908231, 16.606236,  0.001598,  39.917473,   0.146896],
    'Rb':[     8.123134,  2.138042,  6.761702,  1.156051,  17.679546,   1.139548, 15.142385, 33.542667,  0.129372, 224.132507,   1.713368],
    'Sr':[    17.730219,  9.795867,  6.099763,  2.620025,   0.600053,   1.140251,  1.563060, 14.310868,  0.120574, 135.771317,   0.120574],
    'Y' :[    17.792040, 10.253252,  5.714949,  3.170516,   0.918251,   1.131787,  1.429691, 13.132816,  0.112173, 108.197029,   0.112173],
    'Zr':[    17.859772, 10.911038,  5.821115,  3.512513,   0.746965,   1.124859,  1.310692, 12.319285,  0.104353,  91.777542,   0.104353],
    'Nb':[    17.958399, 12.063054,  5.007015,  3.287667,   1.531019,   1.123452,  1.211590, 12.246687,  0.098615,  75.011948,   0.098615],
    'Mo':[     6.236218, 17.987711, 12.973127,  3.451426,   0.210899,   1.108770,  0.090780,  1.108310, 11.468720,  66.684151,   0.090780],
    'Tc':[    17.840963,  3.428236,  1.373012, 12.947364,   6.335469,   1.074784,  1.005729, 41.901382,119.320541,   9.781542,   0.083391],
    'Ru':[     6.271624, 17.906738, 14.123269,  3.746008,   0.908235,   1.043992,  0.077040,  0.928222,  9.555345,  35.860680, 123.552246],
    'Rh':[     6.216648, 17.919739,  3.854252,  0.840326,  15.173498,   0.995452,  0.070789,  0.856121, 33.889484, 121.686691,   9.029517],
    'Pd':[     6.121511,  4.784063, 16.631683,  4.318258,  13.246773,   0.883099,  0.062549,  0.784031,  8.751391,  34.489983,   0.784031],
    'Ag':[     6.073874, 17.155437,  4.173344,  0.852238,  17.988686,   0.756603,  0.055333,  7.896512, 28.443739, 110.376106,   0.716809],
    'Cd':[     6.080986, 18.019468,  4.018197,  1.303510,  17.974669,   0.603504,  0.048990,  7.273646, 29.119284,  95.831207,   0.661231],
    'In':[     6.196477, 18.816183,  4.050479,  1.638929,  17.962912,   0.333097,  0.042072,  6.695665, 31.009790, 103.284348,   0.610714],
    'Sn':[    19.325171,  6.281571,  4.498866,  1.856934,  17.917318,   0.119024,  6.118104,  0.036915, 32.529045,  95.037186,   0.565651],
    'Sb':[     5.394956,  6.549570, 19.650681,  1.827820,  17.867832,  -0.290506, 33.326523,  0.030974,  5.564929,  87.130966,   0.523992],
    'Te':[     6.660302,  6.940756, 19.847015,  1.557175,  17.802427,  -0.806668, 33.031654,  0.025750,  5.065547,  84.101616,   0.487660],
    'I' :[    19.884502,  6.736593,  8.110516,  1.170953,  17.548716,  -0.448811,  4.628591,  0.027754, 31.849096,  84.406387,   0.463550],
    'Xe':[    19.978920, 11.774945,  9.332182,  1.244749,  17.737501,  -6.065902,  4.143356,  0.010142, 28.796200,  75.280685,   0.413616],
    'Cs':[    17.418674,  8.314444, 10.323193,  1.383834,  19.876251,  -2.322802,  0.399828,  0.016872, 25.605827, 233.339676,   3.826915],
    'Ba':[    19.747343, 17.368477, 10.465718,  2.592602,  11.003653,  -5.183497,  3.481823,  0.371224, 21.226641, 173.834274,   0.010719],
    'La':[    19.966019, 27.329655, 11.018425,  3.086696,  17.335455, -21.745489,  3.197408,  0.003446, 19.955492, 141.381973,   0.341817],
    'Ce':[    17.355122, 43.988499, 20.546650,  3.130670,  11.353665, -38.386017,  0.328369,  0.002047,  3.088196, 134.907654,  18.832960],
    'Pr':[    21.551311, 17.161730, 11.903859,  2.679103,   9.564197,  -3.871068,  2.995675,  0.312491, 17.716705, 152.192825,   0.010468],
    'Nd':[    17.331244, 62.783924, 12.160097,  2.663483,  22.239950, -57.189842,  0.300269,  0.001320, 17.026001, 148.748993,   2.910268],
    'Pm':[    17.286388, 51.560162, 12.478557,  2.675515,  22.960947, -45.973682,  0.286620,  0.001550, 16.223755, 143.984512,   2.796480],
    'Sm':[    23.700363, 23.072214, 12.777782,  2.684217,  17.204367, -17.452166,  2.689539,  0.003491, 15.495437, 139.862473,   0.274536],
    'Eu':[    17.186195, 37.156837, 13.103387,  2.707246,  24.419271, -31.586687,  0.261678,  0.001995, 14.787360, 134.816299,   2.581883],
    'Gd':[    24.898117, 17.104952, 13.222581,  3.266152,  48.995213, -43.505684,  2.435028,  0.246961, 13.996325, 110.863091,   0.001383],
    'Tb':[    25.910013, 32.344139, 13.765117,  2.751404,  17.064405, -26.851971,  2.373912,  0.002034, 13.481969, 125.836510,   0.236916],
    'Dy':[    26.671785, 88.687576, 14.065445,  2.768497,  17.067781, -83.279831,  2.282593,  0.000665, 12.920230, 121.937187,   0.225531],
    'Ho':[    27.150190, 16.999819, 14.059334,  3.386979,  46.546471, -41.165253,  2.169660,  0.215414, 12.213148, 100.506783,   0.001211],
    'Er':[    28.174887, 82.493271, 14.624002,  2.802756,  17.018515, -77.135223,  2.120995,  0.000640, 11.915256, 114.529938,   0.207519],
    'Tm':[    28.925894, 76.173798, 14.904704,  2.814812,  16.998117, -70.839813,  2.046203,  0.000656, 11.465375, 111.411980,   0.199376],
    'Yb':[    29.676760, 65.624069, 15.160854,  2.830288,  16.997850, -60.313812,  1.977630,  0.000720, 11.044622, 108.139153,   0.192110],
    'Lu':[    30.122866, 15.099346, 56.314899,  3.540980,  16.943729, -51.049416,  1.883090, 10.342764,  0.000780,  89.559250,   0.183849],
    'Hf':[    30.617033, 15.145351, 54.933548,  4.096253,  16.896156, -49.719837,  1.795613,  9.934469,  0.000739,  76.189705,   0.175914],
    'Ta':[    31.066359, 15.341823, 49.278297,  4.577665,  16.828321, -44.119026,  1.708732,  9.618455,  0.000760,  66.346199,   0.168002],
    'W' :[    31.507900, 15.682498, 37.960129,  4.885509,  16.792112, -32.864574,  1.629485,  9.446448,  0.000898,  59.980675,   0.160798],
    'Re':[    31.888456, 16.117104, 42.390297,  5.211669,  16.767591, -37.412682,  1.549238,  9.233474,  0.000689,  54.516373,   0.152815],
    'Os':[    32.210297, 16.678440, 48.559906,  5.455839,  16.735533, -43.677956,  1.473531,  9.049695,  0.000519,  50.210201,   0.145771],
    'Ir':[    32.004436,  1.975454, 17.070105, 15.939454,   5.990003,   4.018893,  1.353767, 81.014175,  0.128093,   7.661196,  26.659403],
    'Pt':[    31.273891, 18.445440, 17.063745,  5.555933,   1.575270,   4.050394,  1.316992,  8.797154,  0.124741,  40.177994,   1.316997],
    'Au':[    16.777390, 19.317156, 32.979683,  5.595453,  10.576854,  -6.279078,  0.122737,  8.621570,  1.256902,  38.008820,   0.000601],
    'Hg':[    16.839890, 20.023823, 28.428564,  5.881564,   4.714706,   4.076478,  0.115905,  8.256927,  1.195250,  39.247227,   1.195250],
    'Tl':[    16.630795, 19.386616, 32.808571,  1.747191,   6.356862,   4.066939,  0.110704,  7.181401,  1.119730,  90.660263,  26.014978],
    'Pb':[    16.419567, 32.738590,  6.530247,  2.342742,  19.916475,   4.049824,  0.105499,  1.055049, 25.025890,  80.906593,   6.664449],
    'Bi':[    16.282274, 32.725136,  6.678302,  2.694750,  20.576559,   4.040914,  0.101180,  1.002287, 25.714146,  77.057549,   6.291882],
    'Po':[    16.289164, 32.807171, 21.095163,  2.505901,   7.254589,   4.046556,  0.098121,  0.966265,  6.046622,  76.598068,  28.096128],
    'At':[    16.011461, 32.615547,  8.113899,  2.884082,  21.377867,   3.995684,  0.092639,  0.904416, 26.543257,  68.372963,   5.499512],
    'Rn':[    16.070229, 32.641106, 21.489658,  2.299218,   9.480184,   4.020977,  0.090437,  0.876409,  5.239687,  69.188477,  27.632641],
    'Fr':[    16.007385, 32.663830, 21.594351,  1.598497,  11.121192,   4.003472,  0.087031,  0.840187,  4.954467, 199.805801,  26.905106],
    'Ra':[    32.563690, 21.396671, 11.298093,  2.834688,  15.914965,   3.981773,  0.801980,  4.590666, 22.758972, 160.404388,   0.083544],
    'Ac':[    15.914053, 32.535042, 21.553976, 11.433394,   3.612409,   3.939212,  0.080511,  0.770669,  4.352206,  21.381622, 130.500748],
    'Th':[    15.784024, 32.454899, 21.849222,  4.239077,  11.736191,   3.922533,  0.077067,  0.735137,  4.097976, 109.464111,  20.512138],
    'Pa':[    32.740208, 21.973675, 12.957398,  3.683832,  15.744058,   3.886066,  0.709545,  4.050881, 19.231543, 117.255005,   0.074040],
    'U' :[    15.679275, 32.824306, 13.660459,  3.687261,  22.279434,   3.854444,  0.071206,  0.681177, 18.236156, 112.500038,   3.930325],
    'Np':[    32.999901, 22.638077, 14.219973,  3.672950,  15.683245,   3.769391,  0.657086,  3.854918, 17.435474, 109.464485,   0.068033],
    'Pu':[    33.281178, 23.148544, 15.153755,  3.031492,  15.704215,   3.664200,  0.634999,  3.856168, 16.849735, 121.292038,   0.064857],
    'Am':[    33.435162, 23.657259, 15.576339,  3.027023,  15.746100,   3.541160,  0.612785,  3.792942, 16.195778, 117.757004,   0.061755],
    'Cm':[    15.804837, 33.480801, 24.150198,  3.655563,  15.499866,   3.390840,  0.058619,  0.590160,  3.674720, 100.736191,  15.408296],
    'Bk':[    15.889072, 33.625286, 24.710381,  3.707139,  15.839268,   3.213169,  0.055503,  0.569571,  3.615472,  97.694786,  14.754303],
    'Cf':[    33.794075, 25.467693, 16.048487,  3.657525,  16.008982,   3.005326,  0.550447,  3.581973, 14.357388,  96.064972,   0.052450],	
    'H1- ':[   0.702260,  0.763666,  0.248678,  0.261323,   0.023017,   0.000425, 23.945604, 74.897919,  6.773289, 233.583450,   1.337],
    'Li1+':[   0.432724,  0.549257,  0.376575, -0.336481,   0.976060,   0.001764,  0.260367,  1.042836,  7.885294,   0.260368,   3.042],
    'Be2+':[   3.055430, -2.372617,  1.044914,  0.544233,   0.381737,  -0.653773,  0.001226,  0.001227,  1.542106,   0.456279,   4.047],
    'Cval':[   1.258489,  0.728215,  1.119856,  2.168133,   0.705239,   0.019722, 10.683769,  0.208177,  0.836097,  24.603704,  58.954],
    'O1- ':[   3.106934,  3.235142,  1.148886,  0.783981,   0.676953,   0.046136, 19.868080,  6.960252,  0.170043,  65.693512,   0.630],
    'O2- ':[   3.990247,  2.300563,  0.607200,  1.907882,   1.167080,   0.025429, 16.639956,  5.636819,  0.108493,  47.299709,   0.379],
    'F1- ':[   0.457649,  3.841561,  1.432771,  0.801876,   3.395041,   0.069525,  0.917243,  5.507803,  0.164955,  51.076206,  15.821],
    'Na1+':[   3.148690,  4.073989,  0.767888,  0.995612,   0.968249,   0.045300,  2.594987,  6.046925,  0.070139,  14.122657,   0.217],
    'Mg2+':[   3.062918,  4.135106,  0.853742,  1.036792,   0.852520,   0.058851,  2.015803,  4.417941,  0.065307,   9.669710,   0.187],
    'Al3+':[   4.132015,  0.912049,  1.102425,  0.614876,   3.219136,   0.019397,  3.528641,  7.378344,  0.133708,   0.039065,   1.644],
    'Siva':[   2.879033,  3.072960,  1.515981,  1.390030,   4.995051,   0.146030,  1.239713, 38.706276,  0.081481,  93.616333,   2.770],
    'Si4+':[   3.676722,  3.828496,  1.258033,  0.419024,   0.720421,   0.097266,  1.446851,  3.013144,  0.064397,   0.206254,   5.970],
    'Cl1-':[   1.061802,  7.139886,  6.524271,  2.355626,  35.829403, -34.916603,  0.144727,  1.171795, 19.467655,  60.320301,   0.000],
    'K1+ ':[ -17.609339,  1.494873,  7.150305, 10.899569,  15.808228,   0.257164, 18.840979,  0.053453,  0.812940,  22.264105,  14.351],
    'Ca2+':[   8.501441, 12.880483,  9.765095,  7.156669,   0.711160, -21.013187, 10.525848, -0.004033,  0.010692,   0.684443,  27.231],
    'Sc3+':[   7.104348,  1.511488,-53.669773, 38.404816,  24.532240,   0.118642,  0.601957,  0.033386, 12.572138,  10.859736,  14.125],
    'Ti2+':[   7.040119,  1.496285,  9.657304,  0.006534,   1.649561,   0.150362,  0.537072,  0.031914,  8.009958, 201.800293,  24.039],
    'Ti3+':[  36.587933,  7.230255, -9.086077,  2.084594,  17.294008, -35.111282,  0.000681,  0.522262,  5.262317,  15.881716,   6.149],
    'Ti4+':[  45.355537,  7.092900,  7.483858,-43.498817,   1.678915,  -0.110628,  9.252186,  0.523046, 13.082852,  10.193876,   0.023],
    'V2+ ':[   7.754356,  2.064100,  2.576998,  2.011404,   7.126177,  -0.533379,  7.066315,  0.014993,  7.066308,  22.055786,   0.467],
    'V3+ ':[   9.958480,  1.596350,  1.483442,-10.846044,  17.332867,   0.474921,  6.763041,  0.056895, 17.750029,   0.328826,   0.388],
    'V5+ ':[  15.575018,  8.448095,  1.612040, -9.721855,   1.534029,   0.552676,  0.682708,  5.566640, 10.527077,   0.907961,   0.066],
    'Cr2+':[  10.598877,  1.565858,  2.728280,  0.098064,   6.959321,   0.049870,  6.151846,  0.023519, 17.432816,  54.002388,   0.426],
    'Cr3+':[   7.989310,  1.765079,  2.627125,  1.829380,   6.980908,  -0.192123,  6.068867,  0.018342,  6.068887,  16.309284,   0.420],
    'Mn2+':[  11.287712, 26.042414,  3.058096,  0.090258,   7.088306, -24.566132,  5.506225,  0.000774, 16.158575,  54.766354,   0.375],
    'Mn3+':[   6.926972,  2.081342, 11.128379,  2.375107,  -0.419287,  -0.093713,  0.378315,  0.015054,  5.379957,  14.429586,   0.004],
    'Mn4+':[  12.409131,  7.466993,  1.809947,-12.138477,  10.780248,   0.672146,  0.300400,  0.112814, 12.520756,   0.168653,   5.173],
    'Fe2+':[  11.776765, 11.165097,  3.533495,  0.165345,   7.036932,  -9.676919,  4.912232,  0.001748, 14.166556,  42.381958,   0.341],
    'Fe3+':[   9.721638, 63.403847,  2.141347,  2.629274,   7.033846, -61.930725,  4.869297,  0.000293,  4.867602,  13.539076,   0.338],
    'Co2+':[   6.993840, 26.285812, 12.254289,  0.246114,   4.017407, -24.796852,  0.310779,  0.000684,  4.400528,  35.741447,  12.536],
    'Co3+':[   6.861739,  2.678570, 12.281889,  3.501741,  -0.179384,  -1.147345,  0.309794,  0.008142,  4.331703,  11.914167,  11.914],
    'Ni2+':[  12.519017, 37.832058,  4.387257,  0.661552,   6.949072, -36.344471,  3.933053,  0.000442, 10.449184,  23.860998,   0.283],
    'Ni3+':[  13.579366,  1.902844, 12.859268,  3.811005,  -6.838595,  -0.317618,  0.313140,  0.012621,  3.906407,  10.894311,   0.344],
    'Cu1+':[  12.960763, 16.342150,  1.110102,  5.520682,   6.915452, -14.849320,  3.576010,  0.000975, 29.523218,  10.114283,   0.261],
    'Cu2+':[  11.895569, 16.344978,  5.799817,  1.048804,   6.789088, -14.878383,  3.378519,  0.000924,  8.133653,  20.526524,   0.254],
    'Zn2+':[  13.340772, 10.428857,  5.544489,  0.762295,   6.869172,  -8.945248,  3.215913,  0.001413,  8.542680,  21.891756,   0.239],
    'Ga3+':[  13.123875, 35.288189,  6.126979,  0.611551,   6.724807, -33.875122,  2.809960,  0.000323,  6.831534,  16.784311,   0.212],
    'Ge4+':[   6.876636,  6.779091,  9.969591,  3.135857,   0.152389,   1.08654 ,  2.025174,  0.176650,  3.573822,   7.685848,  16.677],
    'Br1-':[  17.714310,  6.466926,  6.947385,  4.402674,  -0.697279,   1.152674,  2.122554, 19.050768,  0.152708,  58.690361,  58.690],
    'Rb1+':[  17.684320,  7.761588,  6.680874,  2.668883,   0.070974,   1.133263,  1.710209, 14.919863,  0.128542,  31.654478,   0.128],
    'Sr2+':[  17.694973,  1.275762,  6.154252,  9.234786,   0.515995,   1.125309,  1.550888, 30.133041,  0.118774,  13.821799,   0.118],
    'Y3+ ':[  46.660366, 10.369686,  4.623042,-62.170834,  17.471146,  19.023842, -0.019971, 13.180257,  0.176398,  -0.016727,   1.467],
    'Zr4+':[   6.802956, 17.699253, 10.650647, -0.248108,   0.250338,   0.827902,  0.096228,  1.296127, 11.240715,  -0.219259,  -0.219],
    'Nb3+':[  17.714323,  1.675213,  7.483963,  8.322464,  11.143573,  -8.339573,  1.172419, 30.102791,  0.080255,  -0.002983,  10.456],
    'Nb5+':[  17.580206,  7.633277, 10.793497,  0.180884,  67.837921, -68.024780,  1.165852,  0.078558,  9.507652,  31.621656,  -0.000],
    'Mo3+':[   7.447050, 17.778122, 11.886068,  1.997905,   1.789626,  -1.898764,  0.072000,  1.073145,  9.834720,  28.221746,  -0.011],
    'Mo5+':[   7.929879, 17.667669, 11.515987,  0.500402,  77.444084, -78.056595,  0.068856,  1.068064,  9.046229,  26.558945,  -0.000],
    'Mo6+':[  34.757683,  9.653037,  6.584769,-18.628115,   2.490594,   1.141916,  1.301770,  7.123843,  0.094097,   1.617443,  12.335],
    'Ru3+':[  17.894758, 13.579529, 10.729251,  2.474095,  48.227997, -51.905243,  0.902827,  8.740579,  0.045125,  24.764954,  -0.001],
    'Ru4+':[  17.845776, 13.455084, 10.229087,  1.653524,  14.059795, -17.241762,  0.901070,  8.482392,  0.045972,  23.015272,  -0.004],
    'Rh3+':[  17.758621, 14.569813,  5.298320,  2.533579,   0.879753,   0.960843,  0.841779,  8.319533,  0.069050,  23.709131,   0.069],
    'Rh4+':[  17.716188, 14.446654,  5.185801,  1.703448,   0.989992,   0.959941,  0.840572,  8.100647,  0.068995,  22.357307,   0.068],
    'Pd2+':[   6.122282, 15.651012,  3.513508,  9.060790,   8.771199,   0.879336,  0.062424,  8.018296, 24.784275,   0.776457,   0.776],
    'Pd4+':[   6.152421,-96.069023,-31.622141,-81.578255, -17.801403,  -0.915874, -0.063951,-11.090354,-13.466152,  -9.758302,  -0.783],
    'Ag1+':[   6.091192,  4.019526, 16.948174,  4.258638,  13.889437,   0.785127,  0.056305,  0.719340,  7.758938,  27.368349,   0.719],
    'Ag2+':[   6.401808, 48.699802,  4.799859,-32.332523,  16.356710,   1.068247,  0.068167,  0.942270, 20.639496,   1.100365,   6.883],
    'Cd2+':[   6.093711, 43.909691, 17.041306,-39.675117,  17.958918,   0.664795,  0.050624,  8.654143, 15.621396,  11.082067,   0.667],
    'In3+':[   6.206277, 18.497746,  3.078131, 10.524613,   7.401234,   0.293677,  0.041357,  6.605563, 18.792250,   0.608082,   0.608],
    'Sn2+':[   6.353672,  4.770377, 14.672025,  4.235959,  18.002131,  -0.042519,  0.034720,  6.167891,  6.167879,  29.006456,   0.561],
    'Sn4+':[  15.445732,  6.420892,  4.562980,  1.713385,  18.033537,  -0.172219,  6.280898,  0.033144,  6.280899,  17.983601,   0.557],
    'Sb3+':[  10.189171, 57.461918, 19.356573,  4.862206, -45.394096,   1.516108,  0.089485,  0.375256,  5.357987,  22.153736,   0.297],
    'Sb5+':[  17.920622,  6.647932, 12.724075,  1.555545,   7.600591,  -0.445371,  0.522315,  0.029487,  5.718210,  16.433775,   5.718],
    'I1- ':[  20.010330, 17.835524,  8.104130,  2.231118,   9.158548,  -3.341004,  4.565931,  0.444266, 32.430672,  95.149040,   0.014],
    'Cs1+':[  19.939056, 24.967621, 10.375884,  0.454243,  17.660248, -19.394306,  3.770511,  0.004040, 25.311275,  76.537766,   0.384],
    'Ba2+':[  19.750200, 17.513683, 10.884892,  0.321585,  65.149834, -59.618172,  3.430748,  0.361590, 21.358307,  70.309402,   0.001],
    'La3+':[  19.688887, 17.345703, 11.356296,  0.099418,  82.358124, -76.846909,  3.146211,  0.339586, 18.753832,  90.345459,   0.001],
    'Ce3+':[  26.593231, 85.866432, -6.677695, 12.111847,  17.401903, -80.313423,  3.280381,  0.001012,  4.313575,  17.868504,   0.326],
    'Ce4+':[  17.457533, 25.659941, 11.691037, 19.695251, -16.994749,  -3.515096,  0.311812, -0.003793, 16.568687,   2.886395,  -0.008],
    'Pr3+':[  20.879841, 36.035797, 12.135341,  0.283103,  17.167803, -30.500784,  2.870897,  0.002364, 16.615236,  53.909359,   0.306],
    'Pr4+':[  17.496082, 21.538509, 20.403114, 12.062211,  -7.492043,  -9.016722,  0.294457, -0.002742,  2.772886,  15.804613,  -0.013],
    'Nd3+':[  17.120077, 56.038139, 21.468307, 10.000671,   2.905866, -50.541992,  0.291295,  0.001421,  2.743681,  14.581367,  22.485],
    'Pm3+':[  22.221066, 17.068142, 12.805423,  0.435687,  52.238770, -46.767181,  2.635767,  0.277039, 14.927315,  45.768017,   0.001],
    'Sm3+':[  15.618565, 19.538092, 13.398946, -4.358811,  24.490461,  -9.714854,  0.006001,  0.306379, 14.979594,   0.748825,   2.454],
    'Eu2+':[  23.899035, 31.657497, 12.955752,  1.700576,  16.992199, -26.204315,  2.467332,  0.002230, 13.625002,  35.089481,   0.253],
    'Eu3+':[  17.758327, 33.498665, 24.067188, 13.436883,  -9.019134, -19.768026,  0.244474, -0.003901,  2.487526,  14.568011,  -0.015],
    'Gd3+':[  24.344999, 16.945311, 13.866931,  0.481674,  93.506378, -88.147179,  2.333971,  0.239215, 12.982995,  43.876347,   0.000],
    'Tb3+':[  24.878252, 16.856016, 13.663937,  1.279671,  39.271294, -33.950317,  2.223301,  0.227290, 11.812528,  29.910065,   0.001],
    'Dy3+':[  16.864344, 90.383461, 13.675473,  1.687078,  25.540651, -85.150650,  0.216275,  0.000593, 11.121207,  26.250975,   2.135],
    'Ho3+':[  16.837524, 63.221336, 13.703766,  2.061602,  26.202621, -58.026505,  0.206873,  0.000796, 10.500283,  24.031883,   2.055],
    'Er3+':[  16.810127, 22.681061, 13.864114,  2.294506,  26.864477, -17.513460,  0.198293,  0.002126,  9.973341,  22.836388,   1.979],
    'Tm3+':[  16.787500, 15.350905, 14.182357,  2.299111,  27.573771, -10.192087,  0.190852,  0.003036,  9.602934,  22.526880,   1.912],
    'Yb2+':[  28.443794, 16.849527, 14.165081,  3.445311,  28.308853, -23.214935,  1.863896,  0.183811,  9.225469,  23.691355,   0.001],
    'Yb3+':[  28.191629, 16.828087, 14.167848,  2.744962,  23.171774, -18.103676,  1.842889,  0.182788,  9.045957,  20.799847,   0.001],
    'Lu3+':[  28.828693, 16.823227, 14.247617,  3.079559,  25.647667, -20.626528,  1.776641,  0.175560,  8.575531,  19.693701,   0.001],
    'Hf4+':[  29.267378, 16.792543, 14.785310,  2.184128,  23.791996, -18.820383,  1.697911,  0.168313,  8.190025,  18.277578,   0.001],
    'Ta5+':[  29.539469, 16.741854, 15.182070,  1.642916,  16.437447, -11.542459,  1.612934,  0.160460,  7.654408,  17.070732,   0.001],
    'W6+ ':[  29.729357, 17.247808, 15.184488,  1.154652,   0.739335,   3.945157,  1.501648,  0.140803,  6.880573,  14.299601,  14.299],
    'Os4+':[  17.113485, 15.792370, 23.342392,  4.090271,   7.671292,   3.988390,  0.131850,  7.288542,  1.389307,  19.629425,   1.389],
    'Ir3+':[  31.537575, 16.363338, 15.597141,  5.051404,   1.436935,   4.009459,  1.334144,  7.451918,  0.127514,  21.705648,   0.127],
    'Ir4+':[  30.391249, 16.146996, 17.019068,  4.458904,   0.975372,   4.006865,  1.328519,  7.181766,  0.127337,  19.060146,   1.328], 
    'Pt2+':[  31.986849, 17.249048, 15.269374,  5.760234,   1.694079,   4.032512,  1.281143,  7.625512,  0.123571,  24.190826,   0.123],
    'Pt4+':[  41.932713, 16.339224, 17.653894,  6.012420, -12.036877,   4.094551,  1.111409,  6.466086,  0.128917,  16.954155,   0.778],
    'Au1+':[  32.124306, 16.716476, 16.814100,  7.311565,   0.993064,   4.040792,  1.216073,  7.165378,  0.118715,  20.442486,  53.095],
    'Au3+':[  31.704271, 17.545767, 16.819551,  5.522640,   0.361725,   4.042679,  1.215561,  7.220506,  0.118812,  20.050970,   1.215],
    'Hg1+':[  28.866837, 19.277540, 16.776051,  6.281459,   3.710289,   4.068430,  1.173967,  7.583842,  0.115351,  29.055994,   1.173],
    'Hg2+':[  32.411079, 18.690371, 16.711773,  9.974835,  -3.847611,   4.052869,  1.162980,  7.329806,  0.114518,  22.009489,  22.009],
    'Tl1+':[  32.295044, 16.570049, 17.991013,  1.535355,   7.554591,   4.054030,  1.101544,  0.110020,  6.528559,  52.495068,  20.338],
    'Tl3+':[  32.525639, 19.139185, 17.100321,  5.891115,  12.599463,  -9.256075,  1.094966,  6.900992,  0.103667,  18.489614,  -0.001],
    'Pb2+':[  27.392647, 16.496822, 19.984501,  6.813923,   5.233910,   4.065623,  1.058874,  0.106305,  6.708123,  24.395554,   1.058],
    'Pb4+':[  32.505657, 20.014240, 14.645661,  5.029499,   1.760138,   4.044678,  1.047035,  6.670321,  0.105279,  16.525040,   0.105],
    'Bi3+':[  32.461437, 19.438683, 16.302486,  7.322662,   0.431704,   4.043703,  0.997930,  6.038867,  0.101338,  18.371586,  46.361],
    'Bi5+':[  16.734028, 20.580494,  9.452623, 61.155834, -34.041023,   4.113663,  0.105076,  4.773282, 11.762162,   1.211775,   1.619],
    'Ra2+':[   4.986228, 32.474945, 21.947443, 11.800013,  10.807292,   3.956572,  0.082597,  0.791468,  4.608034,  24.792431,   0.082],
    'Ac3+':[  15.584983, 32.022125, 21.456327,  0.757593,  12.341252,   3.838984,  0.077438,  0.739963,  4.040735,  47.525002,  19.406],
    'Th4+':[  15.515445, 32.090691, 13.996399, 12.918157,   7.635514,   3.831122,  0.074499,  0.711663,  3.871044,  18.596891,   3.871],
    'U3+ ':[  15.360309, 32.395657, 21.961290,  1.325894,  14.251453,   3.706622,  0.067815,  0.654643,  3.643409,  39.604965,  16.330],
    'U4+ ':[  15.355091, 32.235306,  0.557745, 14.396367,  21.751173,   3.705863,  0.067789,  0.652613, 42.354237,  15.908239,   3.553],
    'U6+ ':[  15.333844, 31.770849, 21.274414, 13.872636,   0.048519,   3.700591,  0.067644,  0.646384,  3.317894,  14.650250,  75.339],
    'Np3+':[  15.378152, 32.572132, 22.206125,  1.413295,  14.828381,   3.603370,  0.064613,  0.631420,  3.561936,  37.875511,  15.546],
    'Np4+':[  15.373926, 32.423019, 21.969994,  0.662078,  14.969350,   3.603039,  0.064597,  0.629658,  3.476389,  39.438942,  15.135],
    'Np6+':[  15.359986, 31.992825, 21.412458,  0.066574,  14.568174,   3.600942,  0.064528,  0.624505,  3.253441,  67.658318,  13.980],
    'Pu3+':[  15.356004, 32.769127, 22.680210,  1.351055,  15.416232,   3.428895,  0.060590,  0.604663,  3.491509,  37.260635,  14.981],
    'Pu4+':[  15.416219, 32.610569, 22.256662,  0.719495,  15.518152,   3.480408,  0.061456,  0.607938,  3.411848,  37.628792,  14.464],
    'Pu6+':[  15.436506, 32.289719, 14.726737, 15.012391,   7.024677,   3.502325,  0.061815,  0.606541,  3.245363,  13.616438,   3.245],
    }  
#  Parametrization of incoherently scattered X-ray intensities vs sin(theta)/lambda.
#  
#  This file contains the tabulated coefficients for calculation
#  of Compton Cross Section as a function of sin(theta)/lambda, being
#  theta=scattering angle, and lambda=photon wavelength. 
#  REFERENCE: 
#  Analytic Approximations to Incoherently Scattered X-Ray Intensities
#  by H.H.M. Balyuzi (Acta Cryst. (1975). A31, 600
#  
#  This is the abstract from the mentioned paper:
#  The theoretically calculated incoherently scattered X-ray intensities of 
#  Cromer & Mann [j. Chem. Phys. (1967). 47, 1892-1893] and Cromer [J. Chem. 
#  Phys. (1969). 50, 4857-4859] for the elements 2 to 95 and 102 have been 
#  fitted to analytic function. The coefficients of the fit are tabulated 
#  and the accurance of the fitting is discussed. The accuracy is very good 
#  for sin(theta)/Lambda<=1.4 A^(-1)
#  
#  The analytical function is: 
#    Compton[s] = Z-[SUM a_i*EXP(-b_i*(s^2)) ]
#                i=1,5
# 
#  where s = sin(theta)/lambda, and a_i and b_i are the tabulated coefficients.
#  Coefficients for elements Cm-Cf simply copied from those for Am
#  Column description:  Z a1 a2 a3 a4 a5 b1 b2 b3 b4 b5
Compton = {
    'H' :[  1.0, 0.2623,  0.5094,  0.2034,  0.0249,  0.0000, 32.3717, 14.7084,  6.6884,  2.4843,  0.0000], 
    'He':[  2.0, 0.5246,  1.0188,  0.4068,  0.0498,  0.0000, 32.3717, 14.7084,  6.6884,  2.4843,  0.0000],
    'Li':[  3.0, 0.0518,  0.9578,  0.7348,  1.0817,  0.1740, 533.922, 245.845, 10.1831,  4.4364,  1.5031],
    'Be':[  4.0, 0.4634,  1.5592,  0.7685,  1.0623,  0.1470, 185.856, 104.601,  4.8589,  2.1932,  0.7641],
    'B' :[  5.0, 0.9046,  1.9822,  0.2279,  1.4873,  0.3979, 104.961, 46.0191,  8.9873,  1.9674,  0.6778],
    'C' :[  6.0, 0.7568,  2.5511,  0.7051,  1.4605,  0.5263, 82.2385, 31.7282, 11.9471,  1.4637,  0.5150],
    'N' :[  7.0, 0.9070,  2.8972,  1.1659,  1.5526,  0.4769, 64.1555, 20.8507,  7.7576,  1.0335,  0.3516],
    'O' :[  8.0, 0.8847,  3.2189,  1.7990,  1.5538,  0.5434, 52.0063, 16.4487,  6.5958,  0.8143,  0.2815],
    'F' :[  9.0, 0.9756,  3.5101,  2.3561,  1.5896,  0.5883, 41.7194, 12.7747,  5.2945,  0.6470,  0.2254],
    'Ne':[ 10.0, 1.1544,  3.8033,  2.8085,  1.6647,  0.5687, 34.2567, 9.76720,  4.0375,  0.5256,  0.1800],
    'Na':[ 11.0, 1.0243,  2.0704,  5.3197,  1.5214,  1.0637, 293.411, 15.2373,  4.4697,  0.6246,  0.1922],
    'Mg':[ 12.0, 2.0049,  1.9449,  5.4291,  1.6315,  0.9888, 178.983, 11.2433,  3.4272,  0.4907,  0.1542],
    'Al':[ 13.0, 2.6790,  1.2336,  5.9358,  1.6622,  1.4835, 121.363, 16.0728,  3.2632,  0.6616,  0.1586],
    'Si':[ 14.0, 2.1006,  2.2025,  5.3063,  2.5162,  1.8716, 116.957, 34.7760,  3.3215,  0.9933,  0.1553],
    'P' :[ 15.0, 1.8936,  3.3782,  5.3752,  2.4942,  1.8564, 108.033, 26.6586,  2.6865,  0.7973,  0.1333],
    'S' :[ 16.0, 2.0717,  4.2023,  5.6883,  2.2623,  1.7735, 86.7211, 21.4574,  2.1255,  0.5882,  0.1128],
    'Cl':[ 17.0, 2.1221,  5.0905,  5.6885,  2.2809,  1.8164, 73.8395, 17.9756,  1.8037,  0.5162,  0.1008],
    'Ar':[ 18.0, 2.1778,  5.9791,  5.6104,  2.3442,  1.8873, 65.6187, 14.4380,  1.5611,  0.4716,  0.0914],
    'K' :[ 19.0, 1.0863,  2.8562,  5.6616,  6.7809,  2.6145, 425.474, 35.7249,  9.3261,  1.0206,  0.1036],
    'Ca':[ 20.0, 2.0554,  2.7251,  5.9104,  6.7171,  2.5916, 289.862, 28.7190,  7.4788,  0.8644,  0.0920],
    'Sc':[ 21.0, 2.1608,  2.9545,  6.5125,  6.7745,  2.5969, 251.387, 26.8528,  6.4238,  0.7616,  0.0831],
    'Ti':[ 22.0, 2.1983,  3.2282,  7.0649,  6.8832,  2.6243, 226.968, 24.9306,  5.6213,  0.6865,  0.0758],
    'V' :[ 23.0, 2.2399,  3.4348,  7.6412,  7.0270,  2.6558, 206.399, 22.9026,  4.9366,  0.6256,  0.0697],
    'Cr':[ 24.0, 1.5893,  3.6692,  8.7249,  7.2913,  2.7226, 196.584, 25.2663,  4.5848,  0.5876,  0.0650],
    'Mn':[ 25.0, 2.3156,  3.7420,  8.8241,  7.3811,  2.7355, 174.656, 19.5879,  3.8896,  0.5326,  0.0597],
    'Fe':[ 26.0, 2.3141,  3.9729,  9.2624,  7.5083,  2.9404, 164.104, 18.2898,  3.5861,  0.5155,  0.0597],
    'Co':[ 27.0, 2.3636,  4.4188,  9.7350,  7.6872,  2.7927, 151.337, 16.0770,  3.1445,  0.4553,  0.0514],
    'Ni':[ 28.0, 2.4174,  4.7076, 10.2289,  7.8357,  2.8071, 139.963, 14.5797,  2.8142,  0.4203,  0.0476],
    'Cu':[ 29.0, 1.7532,  4.8167, 11.2836,  8.2460,  2.8943, 127.251, 16.9194,  2.7458,  0.4076,  0.0454],
    'Zn':[ 30.0, 2.4474,  5.1080, 11.4407,  8.1791,  2.8211, 125.083, 12.8443,  2.3122,  0.3605,  0.0410],
    'Ga':[ 31.0, 3.0360,  5.1760, 11.6692,  8.2940,  2.8148, 104.577, 11.2803,  2.0663,  0.3314,  0.0380],
    'Ge':[ 32.0, 3.4795,  5.2306, 11.9324,  8.5063,  2.8388, 87.5304, 10.8491,  1.9060,  0.3107,  0.0356],
    'As':[ 33.0, 3.3385,  5.6229, 12.2449,  8.8668,  2.9138, 82.4879, 12.0723,  1.8317,  0.2996,  0.0341],
    'Se':[ 34.0, 3.5333,  5.9644, 12.4403,  9.1212,  2.9289, 73.4400, 12.2369,  1.7151,  0.2824,  0.0318],
    'Br':[ 35.0, 3.4898,  6.5641, 12.5318,  9.3877,  3.0170, 68.5438, 12.5940,  1.6235,  0.2712,  0.0308],
    'Kr':[ 36.0, 3.0598,  7.6671, 12.5852,  9.6168,  3.0647, 70.8068, 12.6449,  1.5230,  0.2577,  0.0294],
    'Rb':[ 37.0, 2.3770,  8.6605, 12.5729, 10.1507,  3.2083, 178.867, 13.9200,  1.5224,  0.2567,  0.0289],
    'Sr':[ 38.0, 3.0740,  8.8344, 12.5145, 10.3129,  3.2433, 210.803, 12.3784,  1.4148,  0.2424,  0.0275],
    'Y' :[ 39.0, 3.5086,  9.2574, 12.4297, 10.4962,  3.2840, 177.423, 11.0509,  1.3242,  0.2300,  0.0263],
    'Zr':[ 40.0, 3.8436,  9.6980, 12.3352, 10.7459,  3.3493, 152.274, 10.0161,  1.2610,  0.2206,  0.0254],
    'Nb':[ 41.0, 3.7519, 10.5738, 12.2251, 10.9745,  3.4425, 117.448,  8.9975,  1.2044,  0.2127,  0.0248],
    'Mo':[ 42.0, 3.9567, 11.1228, 12.1674, 11.2328,  3.4857, 105.180,  8.0754,  1.1551,  0.2038,  0.0237],
    'Tc':[ 43.0, 4.3317, 10.7065, 12.1316, 12.0687,  3.7296, 115.939,  7.9151,  1.2769,  0.2105,  0.0240],
    'Ru':[ 44.0, 4.7308, 12.0616, 11.9367, 11.6021,  3.6225, 77.6118,  6.4667,  1.0431,  0.1877,  0.0223],
    'Rh':[ 45.0, 4.8745, 12.2804, 12.0133, 12.0245,  3.7592, 71.5772,  6.1701,  1.0589,  0.1853,  0.0220],
    'Pd':[ 46.0, 5.4847, 13.7150, 11.6608, 11.4709,  3.6545, 53.2508,  4.9329,  0.8575,  0.1655,  0.0206],
    'Ag':[ 47.0, 5.3490, 13.4127, 11.9402, 12.3201,  3.9205, 58.9663,  5.0548,  0.9684,  0.1715,  0.0210],
    'Cd':[ 48.0, 4.5224,  9.6282, 14.9509, 14.2066,  4.6563, 87.3897,  7.7119,  1.5820,  0.2036,  0.0234],
    'In':[ 49.0, 4.7031,  8.2361, 16.4206, 14.6608,  4.9493, 89.2096,  8.9965,  1.7229,  0.2062,  0.0238],
    'Sn':[ 50.0, 4.9677,  7.4873, 17.4268, 14.9229,  5.1678, 83.2133, 10.1130,  1.7605,  0.2042,  0.0237],
    'Sb':[ 51.0, 4.6001,  7.4399, 18.3475, 15.1777,  5.4096, 62.5070, 12.5902,  1.7906,  0.2029,  0.0238],
    'Te':[ 52.0, 4.5671,  7.7178, 18.8338, 15.2844,  5.5769, 79.7245, 13.8028,  1.7411,  0.1976,  0.0235],
    'I' :[ 53.0, 4.3147,  8.3728, 19.1961, 15.3645,  5.7365, 78.6996, 14.9421,  1.6795,  0.1921,  0.0232],
    'Xe':[ 54.0, 3.4467, 19.4724,  9.7473, 15.4210,  5.9034, 88.3050,  1.6131, 16.1669,  0.1869,  0.0229],
    'Cs':[ 55.0, 2.5751, 19.7181, 11.0138, 15.5312,  6.1374, 224.598,  1.5722, 17.7908,  0.1845,  0.0229],
    'Ba':[ 56.0, 3.2477, 19.7824, 11.2142, 15.4790,  6.2602, 266.593,  1.4758, 16.2710,  0.1775,  0.0225],
    'La':[ 57.0, 3.6683, 11.7149, 19.8077, 15.4015,  6.3860, 224.726, 14.7472,  1.3823,  0.1708,  0.0222],
    'Ce':[ 58.0, 3.7218, 12.0076, 20.1925, 15.4540,  6.5996, 212.566, 14.0417,  1.3272,  0.1681,  0.0221],
    'Pr':[ 59.0, 3.6287, 11.9193, 20.8823, 15.6425,  6.8971, 208.103, 13.8486,  1.3025,  0.1690,  0.0222],
    'Nd':[ 60.0, 3.7229, 12.0836, 21.2664, 15.7405,  7.1521, 194.998, 13.2282,  1.2573,  0.1677,  0.0222],
    'Pm':[ 61.0, 3.8009, 12.2546, 21.6281, 15.8659,  7.4126, 184.340, 12.6793,  1.2139,  0.1669,  0.0222],
    'Sm':[ 62.0, 3.8533, 12.3520, 22.0136, 16.0323,  7.7082, 176.392, 12.2878,  1.1810,  0.1672,  0.0223],
    'Eu':[ 63.0, 3.9234, 12.4961, 22.3133, 16.2152,  8.0085, 167.816, 11.8294,  1.1433,  0.1676,  0.0224],
    'Gd':[ 64.0, 4.1100, 13.0008, 22.2962, 16.2937,  8.2594, 160.859, 11.0492,  1.0914,  0.1659,  0.0223],
    'Tb':[ 65.0, 4.1589, 13.2197, 22.4627, 16.5329,  8.5842, 155.186, 10.6407,  1.0667,  0.1674,  0.0224],
    'Dy':[ 66.0, 4.0415, 13.1038, 22.8132, 17.0066,  8.9849, 150.058, 20.6511,  1.0708,  0.1723,  0.0227],
    'Ho':[ 67.0, 4.0620, 13.2132, 22.9351, 17.3999,  9.3388, 145.976, 10.3972,  1.0569,  0.1752,  0.0228],
    'Er':[ 68.0, 4.0815, 13.3282, 22.9865, 17.8548,  9.6967, 142.072, 10.1525,  1.0438,  0.1785,  0.0229],
    'Tm':[ 69.0, 4.7416, 13.2875, 23.1429, 17.9215,  9.8935, 100.170,  9.0676,  0.9833,  0.1753,  0.0227],
    'Yb':[ 70.0, 4.1014, 13.4586, 22.9675, 18.9998, 10.4188, 135.833,  9.8125,  1.0290,  0.1860,  0.0231],
    'Lu':[ 71.0, 4.3261, 13.8638, 22.6992, 19.3649, 10.6986, 133.360,  9.1664,  0.9951,  0.1861,  0.0230],
    'Hf':[ 72.0, 4.5759, 14.3292, 22.3741, 19.7027, 10.9720, 124.501,  8.5195,  0.9599,  0.1857,  0.0229],
    'Ta':[ 73.0, 4.7416, 14.9284, 22.0214, 20.0290, 11.2347, 117.648,  7.9070,  0.9250,  0.1850,  0.0227],
    'W' :[ 74.0, 4.8423, 15.6421, 21.6701, 20.3207, 11.4817, 112.694,  7.3551,  0.8892,  0.1837,  0.0226],
    'Re':[ 75.0, 5.0015, 16.3874, 21.3263, 20.5247, 11.7163, 106.001,  6.7592,  0.8491,  0.1817,  0.0224],
    'Os':[ 76.0, 5.3630, 17.5066, 23.3197, 20.9425,  8.8158, 93.6155,  6.0569,  0.7156,  0.1373,  0.0127],
    'Ir':[ 77.0, 5.0953, 15.4321, 15.5924, 26.5617, 14.2784, 98.2594,  6.9463,  1.2550,  0.2600,  0.0242],
    'Pt':[ 78.0, 5.6451, 18.8015, 20.6881, 20.5266, 12.2868, 69.2677,  5.3127,  0.7107,  0.1702,  0.0216],
    'Au':[ 79.0, 5.8170, 19.5635, 20.3716, 20.6837, 12.5114, 65.4079,  4.9468,  0.6809,  0.1680,  0.0214],
    'Hg':[ 80.0, 5.9389, 19.4946, 19.0547, 22.4442, 13.0069, 74.0107,  4.9118,  0.7375,  0.1779,  0.0216],
    'Tl':[ 81.0, 6.5803, 19.6010, 18.3955, 23.0433, 13.3134, 69.9997,  4.6032,  0.7312,  0.1787,  0.0216],
    'Pb':[ 82.0, 7.3754, 19.7974, 18.0002, 23.2605, 13.4943, 62.4634,  4.2561,  0.7021,  0.1757,  0.0213],
    'Bi':[ 83.0, 8.2324, 20.0183, 17.6603, 23.3246, 13.6777, 53.0479,  3.9249,  0.6675,  0.1722,  0.0210],
    'Po':[ 84.0, 9.0662, 20.1395, 23.6273, 17.1872, 13.8913, 48.3272,  3.6624,  0.1701,  0.6471,  0.0208],
    'At':[ 85.0, 9.9223, 20.2544, 23.9015, 16.7271, 14.1028, 43.7514,  3.4229,  0.1680,  0.6271,  0.0206],
    'Rn':[ 86.0,10.7789, 20.3926, 16.3720, 24.0637, 14.2879, 38.6121,  3.2050,  0.6037,  0.1650,  0.0203],
    'Fr':[ 87.0, 2.5332, 11.5319, 24.7459, 31.9247, 16.2475, 269.878, 21.9610,  2.0716,  0.2243,  0.0223],
    'Ra':[ 88.0, 3.2249, 11.6694, 24.8687, 31.8957, 16.3306, 321.663, 20.3846,  1.9730,  0.2166,  0.0219],
    'Ac':[ 89.0, 3.6207, 12.1269, 24.9363, 31.8798, 16.4227, 270.774, 18.9025,  1.8811,  0.2095,  0.0215],
    'Th':[ 90.0, 4.0181, 12.6119, 24.9621, 31.8646, 16.5249, 232.371, 17.3825,  1.7950,  0.2030,  0.0211],
    'Pa':[ 91.0, 3.7926, 12.6626, 25.8795, 31.9403, 16.7045, 236.803, 17.5908,  1.7607,  0.1984,  0.0209],
    'U' :[ 92.0, 3.9051, 12.8415, 26.4298, 31.9514, 16.8475, 221.178, 16.8737,  1.7008,  0.1932,  0.0205],
    'Np':[ 93.0, 4.0333, 12.9014, 26.7571, 31.4732, 17.8065, 207.727, 16.3175,  1.6677,  0.1954,  0.0220],
    'Pu':[ 94.0, 4.2695, 12.5662, 28.0362, 31.9382, 17.1487, 185.955, 15.5936,  1.5914,  0.1834,  0.0200],
    'Am':[ 95.0, 4.3981, 12.6808, 28.6781, 31.9085, 17.2883, 174.359, 14.9676,  1.5304,  0.1786,  0.0197],
    'Cm':[ 96.0, 4.3981, 12.6808, 28.6781, 31.9085, 17.2883, 174.359, 14.9676,  1.5304,  0.1786,  0.0197],
    'Bk':[ 97.0, 4.3981, 12.6808, 28.6781, 31.9085, 17.2883, 174.359, 14.9676,  1.5304,  0.1786,  0.0197],
    'Cf':[ 98.0, 4.3981, 12.6808, 28.6781, 31.9085, 17.2883, 174.359, 14.9676,  1.5304,  0.1786,  0.0197]
    }
    
