# This module contains utility functions for interacting with GitHub issues and pull requests using AI.
# It includes functions for setting up AI remarks, commenting on issues, reviewing pull requests, and solving issues.

from ._utilities import remove_indentation, catch_error

SYSTEM_PROMPT = """You are an extremely skilled python developer. Your name is git-bob."""

def setup_ai_remark():
    """
    Set up the AI remark for comments.

    Returns
    -------
    str
        The AI remark string.
    """
    from git_bob import __version__
    return f"<sup>This message was generated by [git-bob](https://github.com/haesleinhuepf/git-bob) {__version__}, an experimental AI-based assistant. It can make mistakes and has [limitations](https://github.com/haesleinhuepf/git-bob?tab=readme-ov-file#limitations). Check its messages carefully.</sup>"

@catch_error
def comment_on_issue(repository, issue, prompt_function):
    """
    Comment on a GitHub issue using a prompt function.

    Parameters
    ----------
    repository : str
        The full name of the GitHub repository.
    issue : int
        The issue number to comment on.
    prompt_function : function
        The function to generate the comment.
    """
    print(f"-> comment_on_issue({repository}, {issue})")
    from ._github_utilities import get_conversation_on_issue, add_comment_to_issue

    ai_remark = setup_ai_remark()

    discussion = get_conversation_on_issue(repository, issue)
    print("Discussion:", discussion)

    comment = prompt_function(remove_indentation(f"""
    {SYSTEM_PROMPT}
    Respond to a github issue. Its entire discussion is given.

    ## Discussion

    {discussion}

    ## Your task

    Respond to the discussion above. 
    Do NOT explain your response or anything else. 
    Just respond to the discussion.
    """))

    print("comment:", comment)

    add_comment_to_issue(repository, issue, remove_indentation(f"""        
    {ai_remark}

    {comment}
    """))

@catch_error
def review_pull_request(repository, issue, prompt_function):
    """
    Review a GitHub pull request using a prompt function.

    Parameters
    ----------
    repository : str
        The full name of the GitHub repository.
    issue : int
        The pull request number to review.
    prompt_function : function
        The function to generate the review comment.
    """
    print(f"-> review_pull_request({repository}, {issue})")
    from ._github_utilities import get_conversation_on_issue, add_comment_to_issue, get_diff_of_pull_request

    ai_remark = setup_ai_remark()

    discussion = get_conversation_on_issue(repository, issue)
    print("Discussion:", discussion)

    file_changes = get_diff_of_pull_request(repository, issue)

    print("file_changes:", file_changes)

    comment = prompt_function(remove_indentation(f"""
    {SYSTEM_PROMPT}
    Generate a response to a github pull-request. 
    Given are the discussion on the pull-request and the changed files.

    ## Discussion

    {discussion}

    ## Changed files

    {file_changes}

    ## Your task

    Review this pull-request and contribute to the discussion. 
    
    Do NOT explain your response or anything else. 
    Just respond to the discussion.
    """))

    print("comment:", comment)

    add_comment_to_issue(repository, issue, remove_indentation(f"""        
    {ai_remark}

    {comment}
    """))

@catch_error
def solve_github_issue(repository, issue, llm_model):
    """
    Attempt to solve a GitHub issue by modifying a single file and sending a pull-request.

    Parameters
    ----------
    repository : str
        The full name of the GitHub repository.
    issue : int
        The issue number to solve.
    llm_model : str
        The language model to use for generating the solution.
    """
    # modified from: https://github.com/ScaDS/generative-ai-notebooks/blob/main/docs/64_github_interaction/solving_github_issues.ipynb

    print(f"-> solve_github_issue({repository}, {issue})")

    from ._github_utilities import get_github_issue_details, list_repository_files, get_repository_file_contents, write_file_in_new_branch, send_pull_request, add_comment_to_issue, create_branch, check_if_file_exists
    from ._utilities import remove_outer_markdown
    from blablado import Assistant

    ai_remark = setup_ai_remark()

    assistant = Assistant(model=llm_model)
    assistant.register_tool(get_github_issue_details)
    assistant.register_tool(list_repository_files)
    assistant.register_tool(create_branch)
    assistant.register_tool(send_pull_request)

    issue_summary = assistant.tell(f"Summarize the most important details of issue #{issue} in the repository {repository}")
    print("issue_summary", issue_summary)

    assistant.do(f"List all files in the repository {repository}")
    filenames_json = remove_outer_markdown(assistant.tell("Which of these files might be relevant for issue #{issue} ? You can also consider files which do not exist yet. Respond ONLY the filenames  as JSON list."))

    print("Related filenames", filenames_json)

    # parse the filenames_json into list:
    import json
    filenames = json.loads(filenames_json)

    branch_name = assistant.tell(f"Create a new branch on repository {repository}. Respond ONLY the branch name.")
    branch_name = branch_name.strip().strip('"')

    print("Created branch", branch_name)

    errors = []
    for filename in filenames:
        modifier = Assistant(model=llm_model)
        modifier.register_tool(get_repository_file_contents)
        modifier.register_tool(write_file_in_new_branch)

        try:
            if check_if_file_exists(repository, filename):
                print(filename, "will be overwritten")
                modifier.do(f"Load the entire content of {filename} from the repository {repository} branch {branch_name}.")
                modifier.do(f"Modify the file content of {filename} and write it to repository {repository} branch {branch_name}. Your task is defined like this: \n\n {issue_summary} \n\nDo not do any additional modifications you were not instructed to do.")
            else:
                print(filename, "will be created")
                modifier.do(f"Create the file {filename} and write it to repository {repository} branch {branch_name}. Your task is defined like this: \n\n {issue_summary}")
        except Exception as e:
            errors.append(f"Error processing {filename}" + str(e))

    error_messages = ""
    if len(errors) > 0:
        error_messages = "The following errors occurred:\n\n"
        for e in errors:
            error_messages += f"{str(e)}\n"

    add_comment_to_issue(repository, issue, remove_indentation(f"""
    {ai_remark}
    
    I created a branch with a potential solution [here](https://github.com/{repository}/tree/{branch_name}). 
    
    {error_messages}
    """))
    assistant.do(f"Send a pull-request of the branch {branch_name} in repository {repository} explaining in detail what we changed. Finish the message with 'closes #{issue}'.")
