"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline');
    const sourceStage = pipeline.addStage({
        stageName: 'Source',
        actions: [
            new cpactions.S3SourceAction({
                actionName: 'Source',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                bucketKey: options.bucketKey || 'some/path/to',
                output: sourceOutput,
                trigger: options.trigger,
            }),
        ],
    });
    pipeline.addStage({
        stageName: 'Build',
        actions: [
            new cpactions.CodeBuildAction({
                actionName: 'Build',
                project: new codebuild.PipelineProject(stack, 'MyProject'),
                input: sourceOutput,
            }),
        ],
    });
    return sourceStage;
}
module.exports = {
    'S3 Source Action': {
        'by default polls for source changes and does not use Events'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, undefined);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {},
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not poll for source changes and uses Events for S3Trigger.EVENTS'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.EVENTS });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': false,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.countResources('AWS::Events::Rule', 1));
            test.done();
        },
        'polls for source changes and does not use Events for S3Trigger.POLL'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.POLL });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': true,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not poll for source changes and does not use Events for S3Trigger.NONE'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.NONE });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': false,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not allow passing an empty string for the bucketKey property'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new cpactions.S3SourceAction({
                    actionName: 'Source',
                    bucket: new s3.Bucket(stack, 'MyBucket'),
                    bucketKey: '',
                    output: new codepipeline.Artifact(),
                });
            }, /Property bucketKey cannot be an empty string/);
            test.done();
        },
        'allows using the same bucket with events trigger mutliple times with different bucket paths'(test) {
            const stack = new core_1.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const sourceStage = minimalPipeline(stack, {
                bucket,
                bucketKey: 'my/path',
                trigger: cpactions.S3Trigger.EVENTS,
            });
            sourceStage.addAction(new cpactions.S3SourceAction({
                actionName: 'Source2',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            }));
            test.done();
        },
        'throws an error if the same bucket and path with trigger = Events are added to the same pipeline twice'(test) {
            const stack = new core_1.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const sourceStage = minimalPipeline(stack, {
                bucket,
                bucketKey: 'my/path',
                trigger: cpactions.S3Trigger.EVENTS,
            });
            sourceStage.addAction(new cpactions.S3SourceAction({
                actionName: 'Source2',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            }));
            const duplicateBucketAndPath = new cpactions.S3SourceAction({
                actionName: 'Source3',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            });
            test.throws(() => {
                sourceStage.addAction(duplicateBucketAndPath);
            }, /S3 source action with path 'my\/other\/path' is already present in the pipeline for this source bucket/);
            test.done();
        },
        'allows using a Token bucketKey with trigger = Events, multiple times'(test) {
            const stack = new core_1.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const sourceStage = minimalPipeline(stack, {
                bucket,
                bucketKey: core_1.Lazy.stringValue({ produce: () => 'my-bucket-key1' }),
                trigger: cpactions.S3Trigger.EVENTS,
            });
            sourceStage.addAction(new cpactions.S3SourceAction({
                actionName: 'Source2',
                bucket,
                bucketKey: core_1.Lazy.stringValue({ produce: () => 'my-bucket-key2' }),
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            }));
            assert_1.expect(stack, /* skipValidation = */ true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'S3ObjectKey': 'my-bucket-key1',
                                },
                            },
                            {
                                'Configuration': {
                                    'S3ObjectKey': 'my-bucket-key2',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const s3SourceAction = new cpactions.S3SourceAction({
                actionName: 'Source',
                output: sourceOutput,
                bucket: new s3.Bucket(stack, 'Bucket'),
                bucketKey: 'key.zip',
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [s3SourceAction],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                                environmentVariables: {
                                    VersionId: { value: s3SourceAction.variables.versionId },
                                },
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'Build',
                                'Configuration': {
                                    'EnvironmentVariables': '[{"name":"VersionId","type":"PLAINTEXT","value":"#{Source_Source_NS.VersionId}"}]',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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