"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const sourceAction = new cpactions.GitHubSourceAction({
        actionName: 'Source',
        owner: 'aws',
        repo: 'aws-cdk',
        output: sourceOutput,
        oauthToken: core_1.SecretValue.plainText('secret'),
    });
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [sourceAction],
            },
        ],
    });
    const deploymentStage = pipeline.addStage({
        stageName: 'Deploy',
        actions: [
            new cpactions.S3DeployAction({
                accessControl: options.accessControl,
                actionName: 'CopyFiles',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                cacheControl: options.cacheControl,
                extract: options.extract,
                input: sourceOutput,
                objectKey: options.objectKey,
            }),
        ],
    });
    return deploymentStage;
}
module.exports = {
    'S3 Deploy Action': {
        'by default extract artifacts'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                        'Actions': [
                            {
                                'Name': 'Source',
                                'ActionTypeId': {
                                    'Category': 'Source',
                                    'Owner': 'ThirdParty',
                                },
                            },
                        ],
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'ActionTypeId': {
                                    'Category': 'Deploy',
                                    'Provider': 'S3',
                                },
                                'Configuration': {
                                    'Extract': 'true',
                                },
                                'Name': 'CopyFiles',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'grant the pipeline correct access to the target bucket'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Effect': 'Allow',
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                        },
                        {},
                        {
                            'Effect': 'Allow',
                            'Action': 'sts:AssumeRole',
                        },
                    ],
                },
            }));
            test.done();
        },
        'kebab-case CannedACL value'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                accessControl: s3.BucketAccessControl.PUBLIC_READ_WRITE,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CannedACL': 'public-read-write',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing cache-control'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                cacheControl: [
                    cpactions.CacheControl.setPublic(),
                    cpactions.CacheControl.maxAge(core_1.Duration.hours(12)),
                    cpactions.CacheControl.sMaxAge(core_1.Duration.hours(12)),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CacheControl': 'public, max-age=43200, s-maxage=43200',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing objectKey (deployment path on S3)'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                objectKey: '/a/b/c',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'ObjectKey': '/a/b/c',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'correctly makes the action cross-region for a Bucket imported with a different region'(test) {
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: '123456789012', region: 'us-west-2' },
            });
            const deployBucket = s3.Bucket.fromBucketAttributes(stack, 'DeployBucket', {
                bucketName: 'my-deploy-bucket',
                region: 'ap-southeast-1',
            });
            minimalPipeline(stack, {
                bucket: deployBucket,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Name: 'Deploy',
                        Actions: [
                            {
                                Name: 'CopyFiles',
                                Region: 'ap-southeast-1',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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