"use strict";
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const cpactions = require("../../lib");
function _assertActionMatches(test, stack, actions, provider, category, configuration) {
    const configurationStr = configuration
        ? `, configuration including ${JSON.stringify(stack.resolve(configuration), null, 2)}`
        : '';
    const actionsStr = JSON.stringify(actions.map(a => ({
        owner: a.actionProperties.owner,
        provider: a.actionProperties.provider,
        category: a.actionProperties.category,
        configuration: stack.resolve(a.actionConfig.configuration),
    })), null, 2);
    test.ok(_hasAction(stack, actions, provider, category, configuration), `Expected to find an action with provider ${provider}, category ${category}${configurationStr}, but found ${actionsStr}`);
}
function _hasAction(stack, actions, provider, category, configuration) {
    for (const action of actions) {
        if (action.actionProperties.provider !== provider) {
            continue;
        }
        if (action.actionProperties.category !== category) {
            continue;
        }
        if (configuration && !action.actionConfig.configuration) {
            continue;
        }
        if (configuration) {
            for (const key of Object.keys(configuration)) {
                if (!_.isEqual(stack.resolve(action.actionConfig.configuration[key]), stack.resolve(configuration[key]))) {
                    continue;
                }
            }
        }
        return true;
    }
    return false;
}
function _assertPermissionGranted(test, stack, statements, action, resource, conditions) {
    const conditionStr = conditions
        ? ` with condition(s) ${JSON.stringify(stack.resolve(conditions))}`
        : '';
    const resolvedStatements = stack.resolve(statements.map(s => s.toStatementJson()));
    const statementsStr = JSON.stringify(resolvedStatements, null, 2);
    test.ok(_grantsPermission(stack, resolvedStatements, action, resource, conditions), `Expected to find a statement granting ${action} on ${JSON.stringify(stack.resolve(resource))}${conditionStr}, found:\n${statementsStr}`);
}
function _grantsPermission(stack, statements, action, resource, conditions) {
    for (const statement of statements.filter(s => s.Effect === 'Allow')) {
        if (!_isOrContains(stack, statement.Action, action)) {
            continue;
        }
        if (!_isOrContains(stack, statement.Resource, resource)) {
            continue;
        }
        if (conditions && !_isOrContains(stack, statement.Condition, conditions)) {
            continue;
        }
        return true;
    }
    return false;
}
function _isOrContains(stack, entity, value) {
    const resolvedValue = stack.resolve(value);
    const resolvedEntity = stack.resolve(entity);
    if (_.isEqual(resolvedEntity, resolvedValue)) {
        return true;
    }
    if (!Array.isArray(resolvedEntity)) {
        return false;
    }
    for (const tested of entity) {
        if (_.isEqual(tested, resolvedValue)) {
            return true;
        }
    }
    return false;
}
function _stackArn(stackName, scope) {
    return cdk.Stack.of(scope).formatArn({
        service: 'cloudformation',
        resource: 'stack',
        resourceName: `${stackName}/*`,
    });
}
class PipelineDouble extends cdk.Resource {
    constructor(scope, id, { pipelineName, role }) {
        super(scope, id);
        this.pipelineName = pipelineName || 'TestPipeline';
        this.pipelineArn = cdk.Stack.of(this).formatArn({ service: 'codepipeline', resource: 'pipeline', resourceName: this.pipelineName });
        this.role = role;
        this.artifactBucket = new BucketDouble(scope, 'BucketDouble');
    }
    onEvent(_id, _options) {
        throw new Error('Method not implemented.');
    }
    onStateChange(_id, _options) {
        throw new Error('Method not implemented.');
    }
}
class FullAction {
    constructor(actionProperties, actionConfig) {
        this.actionProperties = actionProperties;
        this.actionConfig = actionConfig;
        // empty
    }
}
class StageDouble {
    constructor({ name, pipeline, actions }) {
        this.actions = [];
        this.stageName = name || 'TestStage';
        this.pipeline = pipeline;
        const stageParent = new cdk.Construct(pipeline, this.stageName);
        const fullActions = new Array();
        for (const action of actions) {
            const actionParent = new cdk.Construct(stageParent, action.actionProperties.actionName);
            fullActions.push(new FullAction(action.actionProperties, action.bind(actionParent, this, {
                role: pipeline.role,
                bucket: pipeline.artifactBucket,
            })));
        }
        this.fullActions = fullActions;
    }
    get node() {
        throw new Error('StageDouble is not a real construct');
    }
    addAction(_action) {
        throw new Error('addAction() is not supported on StageDouble');
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not supported on StageDouble');
    }
}
class RoleDouble extends iam.Role {
    constructor(scope, id, props = { assumedBy: new iam.ServicePrincipal('test') }) {
        super(scope, id, props);
        this.statements = new Array();
    }
    addToPolicy(statement) {
        super.addToPolicy(statement);
        this.statements.push(statement);
        return true;
    }
}
class BucketDouble extends s3.Bucket {
    grantRead(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantReadWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
}
module.exports = nodeunit.testCase({
    CreateReplaceChangeSet: {
        'works'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Stack');
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            const action = new cpactions.CloudFormationCreateReplaceChangeSetAction({
                actionName: 'Action',
                changeSetName: 'MyChangeSet',
                stackName: 'MyStack',
                templatePath: artifact.atPath('path/to/file'),
                adminPermissions: false,
            });
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [action],
            });
            app.synth();
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
            const stackArn = _stackArn('MyStack', stack);
            const changeSetCondition = { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } };
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStacks', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteChangeSet', stackArn, changeSetCondition);
            // TODO: revert "as any" once we move all actions into a single package.
            test.deepEqual(stage.fullActions[0].actionProperties.inputs, [artifact], 'The input was correctly registered');
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_CREATE_REPLACE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/file'),
                    }),
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/other/file'),
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::GetAtt': ['PipelineTestStageActionARole9283FBE3', 'Arn'] },
                        { 'Fn::GetAtt': ['PipelineTestStageActionBRoleCABC8FA5', 'Arn'] },
                    ],
                },
                {
                    Action: [
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    ExecuteChangeSet: {
        'works'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action',
                        changeSetName: 'MyChangeSet',
                        stackName: 'MyStack',
                    }),
                ],
            });
            const stackArn = _stackArn('MyStack', stack);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:ExecuteChangeSet', stackArn, { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } });
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_EXECUTE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: [
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                        'cloudformation:ExecuteChangeSet',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    'the CreateUpdateStack Action sets the DescribeStack*, Create/Update/DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'Action',
            templatePath: new codepipeline.Artifact('TestArtifact').atPath('some/file'),
            stackName: 'MyStack',
            adminPermissions: false,
            replaceOnFailure: true,
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:UpdateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
    'the DeleteStack Action sets the DescribeStack*, DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationDeleteStackAction({
            actionName: 'Action',
            adminPermissions: false,
            stackName: 'MyStack',
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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