"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = stage.pipeline.node.uniqueId + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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