"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const sqs = require("@aws-cdk/aws-sqs");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test getter methods', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const filter = { prefix: 'the/place', suffix: '*.mp3' };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    const app = new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    expect(app.sqsQueue !== null);
    // Assertion 2
    expect(app.deadLetterQueue !== null);
    // Assertion 3
    expect(app.s3Bucket !== null);
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const kmsKey = new kms.Key(stack, 'ExistingQueueEncryptionKey', {});
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj',
        encryptionMasterKey: kmsKey
    });
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingQueueObj: queue
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "existingqueueobjF8AF0ED1",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing s3 Bucket
// --------------------------------------------------------------
test('Test deployment w/ existing Bucket', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const [myBucket] = defaults.buildS3Bucket(stack, {});
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingBucketObj: myBucket
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Pattern deployment w/ bucket block public access override
// --------------------------------------------------------------
test('Pattern deployment w/ bucket versioning turned off', () => {
    const stack = new core_1.Stack();
    const props = {
        bucketProps: {
            blockPublicAccess: {
                blockPublicAcls: false,
                blockPublicPolicy: true,
                ignorePublicAcls: false,
                restrictPublicBuckets: true
            }
        }
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: false,
            BlockPublicPolicy: true,
            IgnorePublicAcls: false,
            RestrictPublicBuckets: true
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ specific s3 event types
// --------------------------------------------------------------
test('Test deployment w/ s3 event types and filters', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const filter = {
        prefix: 'the/place',
        suffix: '*.mp3'
    };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectRemoved:*'],
                    Filter: {
                        Key: {
                            FilterRules: [
                                {
                                    Name: 'suffix',
                                    Value: '*.mp3'
                                },
                                {
                                    Name: 'prefix',
                                    Value: 'the/place'
                                }
                            ]
                        }
                    },
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ SSE encryption enabled using customer managed KMS CMK
// --------------------------------------------------------------
test('Test deployment w/ SSE encryption enabled using customer managed KMS CMK', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        enableEncryptionWithCustomerManagedKey: true
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications");
    // Assertion 2
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        "kms:Create*",
                        "kms:Describe*",
                        "kms:Enable*",
                        "kms:List*",
                        "kms:Put*",
                        "kms:Update*",
                        "kms:Revoke*",
                        "kms:Disable*",
                        "kms:Get*",
                        "kms:Delete*",
                        "kms:ScheduleKeyDeletion",
                        "kms:CancelKeyDeletion",
                        "kms:GenerateDataKey",
                        "kms:TagResource",
                        "kms:UntagResource"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:Decrypt",
                        "kms:Encrypt",
                        "kms:ReEncrypt*",
                        "kms:GenerateDataKey*"
                    ],
                    Condition: {
                        ArnLike: {
                            "aws:SourceArn": {
                                "Fn::GetAtt": [
                                    "tests3sqsS3BucketFF76CDA6",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Service: "s3.amazonaws.com"
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:GenerateDataKey*",
                        "kms:Decrypt"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "s3.amazonaws.com"
                    },
                    Resource: "*"
                }
            ],
            Version: "2012-10-17"
        },
        EnableKeyRotation: true
    });
});
//# sourceMappingURL=data:application/json;base64,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