/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import * as React from 'react';
import { Global } from '../../Global';
import { ServerConnection } from '@jupyterlab/services';
import { Machine } from './Machine';
import { Tag } from '../../components/Tag';
import { InfoSkirt } from '../../components/InfoSkirt';
import { CircularProgress, IconButton } from '@material-ui/core';
import DeleteIcon from '@material-ui/icons/Delete';
import WarningPopup from '../../core/WarningPopup';
import FormatUtils from '../../utils/FormatUtils';
export class MachineComponent extends React.Component {
    constructor(props) {
        super(props);
        this._isMounted = false;
        this.stopApp = (uuid) => {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/stop-notebook";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    uuid: uuid,
                }),
            };
            return ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    throw new ServerConnection.ResponseError(response);
                }
                return response.text();
            });
        };
        this.deleteApp = (uuid) => {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/teardown-notebook";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    uuid: uuid,
                }),
            };
            return ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    throw new ServerConnection.ResponseError(response);
                }
                Global.user.appTracker.removeApp(uuid);
                return response.text();
            });
        };
        this.handleDeleteClicked = () => {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/delete-machine";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    uuid: this.props.machine.uuid,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    this.checkAndSetState({ waiting: false });
                    throw new ServerConnection.ResponseError(response);
                }
                this.checkAndSetState({ waiting: false, deleting: true });
            }, () => this.checkAndSetState({ waiting: false }));
        };
        this.showLoading = (machineState) => {
            switch (machineState) {
                case 'Acquiring...':
                case 'Releasing...':
                    return true;
                default:
                    return false;
            }
        };
        this.render = () => {
            if (Global.shouldLogOnRender)
                console.log('MachineComponentRender (' + new Date().getSeconds() + ')');
            var machine = this.props.machine;
            var tags = [];
            var specialTag = [];
            if (machine.rate != undefined)
                specialTag.push(React.createElement(Tag, { key: 'machineRate' + machine.rate, label: FormatUtils.styleRateUnitValue()(machine.rate) }));
            var machineState = Machine.getStateMessage(this.props.machine.state);
            if (machineState != '')
                tags.push(React.createElement(Tag, { key: 'machineState' + machineState, label: machineState, showLoading: this.showLoading(machineState) }));
            const user = Global.user;
            return (React.createElement(InfoSkirt, { leftButton: this.props.machine.getPopupComponent(), rightButton: React.createElement(React.Fragment, null,
                    React.createElement(WarningPopup, { open: this.state.showDeleteMachinePopup, headerText: "Are you sure?", bodyText: (() => {
                            if (user.appTracker.activeSessions.length == 0 && user.appTracker.activeJobs.length == 0)
                                return "";
                            const app = user.appTracker.activeSessions.length == 0 ? user.appTracker.activeJobs[0] : user.appTracker.activeSessions[0];
                            const appWord = app.interactive ? "session" : "job";
                            const appAction = app.interactive ? "closed" : "terminated";
                            const appName = app.name.split('/').pop();
                            // if (app is running) {
                            if ((app.requisitioning.completed && !app.requisitioning.error) && !app.running.completed) {
                                return "Your " + appWord + " '" + appName + "' is active on this machine. It will be " + appAction + " if the machine is released.";
                            }
                            else {
                                return "Your " + appWord + " '" + appName + "' is waiting for this machine. It will be deleted if the machine is released.";
                            }
                        })(), cancel: {
                            text: `Cancel`,
                            onCancel: () => {
                                this.checkAndSetState({ showDeleteMachinePopup: false });
                            },
                        }, continue: {
                            text: `Release it`,
                            onContinue: () => {
                                if (user.appTracker.activeSessions.length == 0 && user.appTracker.activeJobs.length == 0)
                                    return "";
                                const app = user.appTracker.activeSessions.length == 0 ? user.appTracker.activeJobs[0] : user.appTracker.activeSessions[0];
                                // if (app is running) {
                                if ((app.requisitioning.completed && !app.requisitioning.error) && !app.running.completed) {
                                    // Only delete the machine after the job is stopped to avoid a race condition
                                    // We set these outside so they will span the job stop/delete as well
                                    this.setState({ waiting: true, spinning: false });
                                    setTimeout(() => this.checkAndSetState({ spinning: true }), 1000);
                                    this.stopApp(app.uuid).then(() => this.handleDeleteClicked());
                                }
                                else {
                                    // Only delete the machine after the job is deleted to avoid a race condition
                                    // We set these outside so they will span the job stop/delete as well
                                    this.setState({ waiting: true, spinning: false });
                                    setTimeout(() => this.checkAndSetState({ spinning: true }), 1000);
                                    this.deleteApp(app.uuid).then(() => this.handleDeleteClicked());
                                }
                                this.checkAndSetState({ showDeleteMachinePopup: false });
                            },
                            color: `error`,
                        } }),
                    React.createElement(IconButton, { onClick: () => {
                            if (user.appTracker.activeSessions.length == 0 && user.appTracker.activeJobs.length == 0) {
                                // We set these outside so they will span the job stop/delete as well
                                this.setState({ waiting: true, spinning: false });
                                setTimeout(() => this.checkAndSetState({ spinning: true }), 1000);
                                this.handleDeleteClicked();
                            }
                            else {
                                this.checkAndSetState({ showDeleteMachinePopup: true });
                            }
                        }, disabled: this.state.deleting || this.state.waiting || this.props.machine.state.includes('sequestration'), style: { position: 'relative', display: 'inline-block', width: '36px', height: '36px', padding: '3px' } },
                        React.createElement(DeleteIcon, { style: { position: 'relative', width: '30px', height: '30px', padding: '3px' } }),
                        (this.state.deleting || this.state.waiting) && this.state.spinning && React.createElement(CircularProgress, { size: '30px', style: { position: 'absolute' } }))), tags: tags, specialTag: specialTag }, this.props.machine.getIdentityComponent()));
        };
        this.shouldComponentUpdate = (nextProps, nextState) => {
            try {
                if (JSON.stringify(this.props) != JSON.stringify(nextProps))
                    return true;
                if (JSON.stringify(this.state) != JSON.stringify(nextState))
                    return true;
                if (Global.shouldLogOnRender)
                    console.log('SuppressedRender');
                return false;
            }
            catch (error) {
                return true;
            }
        };
        this.componentDidMount = () => {
            this._isMounted = true;
        };
        this.componentWillUnmount = () => {
            this._isMounted = false;
        };
        // This is only ever needed to set state after a promise
        this.checkAndSetState = (map) => {
            if (this._isMounted)
                this.setState(map);
        };
        this.state = {
            showDeleteMachinePopup: false,
            waiting: false,
            spinning: false,
            deleting: false,
        };
    }
}
