/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import { ServerConnection } from '@jupyterlab/services';
import { Global } from '../Global';
import FormatUtils from '../utils/FormatUtils';
export class ProgressMessage {
    constructor(statusType, updates = []) {
        this._appUUID = "";
        this.loaded = 0;
        this.total = 0;
        this.sending = false;
        this.sendQueue = [];
        this.sendNextMessage = () => {
            const nextUpdate = this.sendQueue.shift();
            if (nextUpdate) {
                this.sendMessage(nextUpdate);
            }
            else {
                this.sending = false;
            }
        };
        this.phase = statusType;
        this._updates = updates;
    }
    set appUUID(appUUID) {
        this._appUUID = appUUID;
    }
    get message() {
        for (let i = this._updates.length - 1; i >= 0; i--) {
            const update = this._updates[i];
            if (update.line != 'error' && update.line != 'stop' && update.line != '') {
                return update.line;
            }
        }
        return "";
    }
    get messages() {
        const messages = [];
        for (let update of this._updates) {
            if (update.line != 'error' && update.line != 'stop' && update.line != '') {
                messages.push(update.line);
            }
        }
        return messages;
    }
    // There are cases where we do not want to send an update to the controller
    addUpdate(update, send = true) {
        if (this._updates.length == 0 || this._updates[this._updates.length - 1].line != update.line) {
            // We do not want to record the same update twice
            this._updates.push(update);
            if (send)
                this.pushStatusUpdate(update);
        }
    }
    get length() {
        return this._updates.length;
    }
    get started() {
        return this._updates.length > 0;
    }
    get elapsed() {
        try {
            const start = Date.parse(this._updates[0].modifier);
            var end = new Date().getTime();
            if (this.completed) {
                for (let update of this._updates) {
                    var date = Date.parse(update.modifier);
                    if (!isNaN(date)) {
                        end = date;
                    }
                }
            }
            return FormatUtils.msToTime(end - start);
        }
        catch (err) {
            return undefined;
        }
    }
    get endTime() {
        var end = new Date().getTime();
        if (this.completed) {
            for (let update of this._updates) {
                var date = Date.parse(update.modifier);
                if (!isNaN(date)) {
                    end = date;
                }
            }
            return new Date(end);
        }
        return undefined;
    }
    get completed() {
        for (let update of this._updates) {
            if (update.line == "stop")
                return true;
        }
        return false;
    }
    get error() {
        for (let update of this._updates) {
            if (update.line == "error")
                return true;
        }
        return false;
    }
    pushStatusUpdate(update) {
        if (this._appUUID == "") {
            console.log("Status update not pushed because app UUID is empty");
            return;
        }
        // If we are already sending a message, queue this new message
        if (this.sending) {
            this.sendQueue.push(update);
        }
        else {
            this.sending = true;
            this.sendMessage(update);
        }
    }
    sendMessage(update) {
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/push-workload-status-update";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                uuid: this._appUUID,
                phase: this.phase,
                update: update.line,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            // Send the next message (in order to try to be robust, we will send the next message even if we encountered an error sending the previous one)
            this.sendNextMessage();
            if (response.status !== 200 && response.status !== 201) {
                console.log("unable to push status update to the controller");
                if (response.status == 401)
                    Global.user = null;
                throw new ServerConnection.ResponseError(response);
            }
        }, () => {
            // If we encounter an error, we will want to send the next message...
            this.sendNextMessage();
        });
    }
}
