/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import { Signal } from '@lumino/signaling';
import { ServerConnection } from '@jupyterlab/services';
import { Global } from '../Global';
import { Update } from './Update';
import { OutputFile } from './OutputFile';
import { Machine } from './machine/Machine';
export var Status;
(function (Status) {
    Status[Status["Initializing"] = 0] = "Initializing";
    Status[Status["Running"] = 1] = "Running";
    Status[Status["Completed"] = 2] = "Completed";
})(Status || (Status = {}));
export class Module {
    constructor(uuid, workloadUUID, addOutput, machine = null, sessionToken = null, output = [], updates = [], files = []) {
        this._changed = new Signal(this);
        this._sessionReady = false;
        this.pushModuleInput = (line) => {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/push-module-input";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    workloadUUID: this._workloadUUID,
                    moduleUUID: this._uuid,
                    line: line,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401) {
                        this.stopPolling(true);
                        Global.user = null;
                    }
                    throw new ServerConnection.ResponseError(response);
                }
            });
        };
        this._uuid = uuid;
        this._workloadUUID = workloadUUID;
        this.addOutput = addOutput;
        this._sessionToken = sessionToken;
        this._machine = machine;
        this._output = output;
        for (var l of this._output) {
            this.addOutput(l.line, l.modifier);
        }
        this._sessionReady = output.map(x => x.line.includes('Jupyter Notebook ') && x.line.includes(' is running at:')).includes(true);
        this._files = files;
        this._updates = updates;
        this._polling = false;
    }
    get changed() {
        return this._changed;
    }
    get sessionReady() {
        return this._sessionReady;
    }
    get uuid() {
        return this._uuid;
    }
    ////
    /// We are not using the module stdin yet
    //
    get sessionToken() {
        return this._sessionToken;
    }
    get sessionPort() {
        return this._sessionPort;
    }
    get machine() {
        return this._machine;
    }
    get output() {
        return this._output;
    }
    get files() {
        return this._files;
    }
    get modStatus() {
        return (this._updates.includes("closed") || this._updates.includes("stop")) ? Status.Completed : Status.Running;
    }
    get error() {
        return this._updates.includes("error");
    }
    async receiveUpdate() {
        // If there is an unsigned agreement, do not poll
        if (Global.user != null && Global.user.unsignedAgreement) {
            if (this._polling)
                setTimeout(() => this.receiveUpdate(), 500);
            return;
        }
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/pull-module-status-update";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                workloadUUID: this._workloadUUID,
                moduleUUID: this._uuid,
                lastUpdateLine: this._updates.length.toString(),
                lastOutputLine: this._output.length.toString(),
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (this._polling) {
                // If we are polling, send a new request in half a second
                setTimeout(() => this.receiveUpdate(), 500);
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401) {
                    this.stopPolling(true);
                    Global.user = null;
                }
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body.output != null) {
                for (let i = 0; i < body.output.length; i++) {
                    this.addOutput(body.output[i], body.outputmod[i]);
                    this._output.push(new Update(body.output[i], body.outputmod[i]));
                    if (body.output[i].includes('Jupyter Notebook ') && body.output[i].includes(' is running at:')) {
                        this._sessionReady = true;
                        window.open('http://localhost:' + this._sessionPort + '?token=' + this._sessionToken, '_blank');
                    }
                }
            }
            if (body.files != null) {
                this._files = [];
                for (let i = 0; i < body.files.length; i++) {
                    if (body.files[i] != '') {
                        this._files.push(new OutputFile(body.files[i], body.filesmod[i], body.filessize[i]));
                    }
                }
            }
            if (body.token != null) {
                this._sessionToken = body.token;
            }
            if (body.updates != null) {
                for (var update of body.updates) {
                    this._updates.push(update);
                    if (update == "stop") {
                        this._updates.push("stop");
                    }
                    else if (update == "launched") {
                    }
                    else if (update == "closed") {
                    }
                }
            }
            if (body.machine != null) {
                this._machine = Object.setPrototypeOf(body.machine, Machine.prototype);
            }
            this._changed.emit(this);
        });
    }
    startPolling(interactive = false) {
        this._polling = true;
        this.receiveUpdate();
        // Start a session handler if necessary
        if (interactive) {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/connect-session";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    module: this._uuid,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    throw new ServerConnection.ResponseError(response);
                }
                return response.json();
            }).then((body) => {
                if (body.port) {
                    this._sessionPort = body.port;
                }
            });
        }
    }
    stopPolling(interactive = false) {
        this._polling = false;
        // Stop a session handler if necessary
        if (interactive) {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/disconnect-session";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    module: this._uuid,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    throw new ServerConnection.ResponseError(response);
                }
                return response.text();
            });
        }
    }
}
