///<reference path="../../node_modules/@types/node/index.d.ts"/>
/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import * as React from 'react';
import { ProgressMessage } from './ProgressMessage';
import { Module, Status } from './Module';
import { Signal } from '@lumino/signaling';
import { ServerConnection } from '@jupyterlab/services';
import { OptumiMetadata } from './OptumiMetadata';
import { Machine, NoMachine } from './machine/Machine';
import { Global } from '../Global';
import { InfoSkirt } from '../components/InfoSkirt';
import { StatusColor, StatusWrapper } from '../components/StatusWrapper';
import { IconButton, CircularProgress } from '@material-ui/core';
import ClearIcon from '@material-ui/icons/Clear';
import CheckIcon from '@material-ui/icons/Check';
import DeleteIcon from '@material-ui/icons/Delete';
import StopIcon from '@material-ui/icons/Stop';
import { DetailsDialog } from '../components/monitor/DetailsDialog';
import { Tag } from '../components/Tag';
import { Update } from './Update';
import { OutputFile } from './OutputFile';
import { Snackbar } from './Snackbar';
import WarningPopup from '../core/WarningPopup';
import FileServerUtils from '../utils/FileServerUtils';
import NotebookUtils from '../utils/NotebookUtils';
import FormatUtils from '../utils/FormatUtils';
export var Phase;
(function (Phase) {
    Phase["Initializing"] = "initializing";
    Phase["Uploading"] = "uploading";
    Phase["Requisitioning"] = "requisitioning";
    Phase["Running"] = "running";
})(Phase || (Phase = {}));
export class App {
    constructor(name, notebook, script, uuid = "", initializing = [], uploading = [], requisitioning = [], running = [], timestamp = new Date()) {
        this._changed = new Signal(this);
        // Callback for module to add output to the notebook
        this.formatForNotebook = (text) => {
            // We will not fix overwritten characters in the last line, in case the last line we have wipes out all the characters without writing new ones
            // We will leave the task of removing backspace characters from the last line to the component that renders the lines
            var lastLine = text.pop();
            // Remove overwritten characters and split into lines
            const formatted = NotebookUtils.fixOverwrittenChars(text).split('\n').map(x => x + '\n');
            // Remove the extra '\n'
            formatted[formatted.length - 1] = formatted[formatted.length - 1].substr(0, formatted[formatted.length - 1].length - 1);
            // Add the last line back
            formatted.push(lastLine);
            return formatted;
        };
        this.addOutput = (line, modifier) => {
            // We do not want to add output to the notebook for an interactive session
            if (!this._interactive) {
                // TODO:JJ If modifier is input, we can not parse it
                if (modifier == 'input')
                    return;
                // TODO:JJ We could get the info that this line is error from either the modifier or the metadata
                try {
                    const parsed = JSON.parse(line);
                    const metadata = parsed.metadata;
                    const outputs = this._notebook.cells[metadata.cell].outputs;
                    if (metadata.status) {
                        if (metadata.status == 'started') {
                            this._notebook.cells[metadata.cell].execution_count = '*';
                        }
                        else if (metadata.status == 'ended') {
                            this._notebook.cells[metadata.cell].execution_count = +metadata.cell + 1;
                        }
                    }
                    else {
                        if (modifier == 'output') {
                            // We will write our other status updates to output
                            if (outputs.length == 0 || outputs[outputs.length - 1].name != 'stdout') {
                                outputs.push({
                                    name: 'stdout',
                                    output_type: 'stream',
                                    text: this.formatForNotebook([parsed.text]),
                                });
                            }
                            else {
                                const text = outputs[outputs.length - 1].text;
                                text.push(parsed.text);
                                outputs[outputs.length - 1].text = this.formatForNotebook(text);
                            }
                        }
                        else if (modifier == 'error') {
                            // This could be a print to stderr (like a log message) or it could be a caught and formatted exception
                            if (parsed.text) {
                                if (outputs.length == 0 || outputs[outputs.length - 1].name != 'stderr') {
                                    outputs.push({
                                        name: 'stderr',
                                        output_type: 'stream',
                                        text: this.formatForNotebook([parsed.text]),
                                    });
                                }
                                else {
                                    const text = outputs[outputs.length - 1].text;
                                    text.push(parsed.text);
                                    outputs[outputs.length - 1].text = this.formatForNotebook(text);
                                }
                            }
                            else {
                                outputs.push({
                                    "ename": parsed.ename,
                                    "evalue": parsed.evalue,
                                    "output_type": "error",
                                    "traceback": parsed.traceback,
                                });
                            }
                        }
                        else if (modifier == 'input') {
                            // TODO:JJ
                        }
                        else {
                            console.error('Unknown modifier' + modifier);
                        }
                    }
                }
                catch (err) {
                    // Add this to the top cell as raw output
                    if (this._notebook.cells[0].execution_count == null) {
                        this._notebook.cells[0].execution_count = '*';
                    }
                    this._notebook.cells[0].outputs.push({
                        name: 'stdout',
                        output_type: 'stream',
                        text: [line],
                    });
                }
            }
        };
        this._uuid = "";
        this._modules = [];
        this.formatTime = () => {
            var app = this;
            var yesterday = new Date();
            yesterday.setDate(yesterday.getDate() - 1);
            if (app.timestamp == undefined)
                return undefined;
            var startTime = app.timestamp < yesterday ? app.timestamp.toLocaleDateString() : app.timestamp.toLocaleTimeString();
            if (app.getEndTime() == undefined)
                return startTime;
            var endTime = app.getEndTime() < yesterday ? app.getEndTime().toLocaleDateString() : app.getEndTime().toLocaleTimeString();
            return startTime == endTime ? startTime : startTime + " - " + endTime;
        };
        this._notebook = notebook;
        const optumi = new OptumiMetadata(this._notebook["metadata"]["optumi"] || {});
        this._interactive = optumi.interactive;
        this._script = script;
        this._name = name;
        this._uuid = uuid;
        this._initializing = new ProgressMessage(Phase.Initializing, initializing);
        this._uploading = new ProgressMessage(Phase.Uploading, uploading);
        this._requisitioning = new ProgressMessage(Phase.Requisitioning, requisitioning);
        this._running = new ProgressMessage(Phase.Running, running);
        if (this._uuid != "") {
            this._initializing.appUUID = this._uuid;
            this._uploading.appUUID = this._uuid;
            this._requisitioning.appUUID = this._uuid;
            this._running.appUUID = this._uuid;
        }
        this._timestamp = timestamp;
        this._polling = false;
        // Handle errors where we were unable to load some of the updates
        if (this._running.started) {
            if (!this._requisitioning.completed) {
                this._requisitioning.addUpdate(new Update("Unable to retrieve requisitioning updates", ""));
                this._requisitioning.addUpdate(new Update("stop", ""));
            }
            if (!this._uploading.completed) {
                this._uploading.addUpdate(new Update("Unable to retrieve uploading updates", ""));
                this._uploading.addUpdate(new Update("stop", ""));
            }
            if (!this._initializing.completed) {
                this._initializing.addUpdate(new Update("Unable to retrieve initializing updates", ""));
                this._initializing.addUpdate(new Update("stop", ""));
            }
        }
        else if (this._requisitioning.started) {
            if (!this._uploading.completed) {
                this._uploading.addUpdate(new Update("Unable to retrieve uploading updates", ""));
                this._uploading.addUpdate(new Update("stop", ""));
            }
            if (!this._initializing.completed) {
                this._initializing.addUpdate(new Update("Unable to retrieve initializing updates", ""));
                this._initializing.addUpdate(new Update("stop", ""));
            }
        }
        else if (this._uploading.started) {
            if (!this._initializing.completed) {
                this._initializing.addUpdate(new Update("Unable to retrieve initializing updates", ""));
                this._initializing.addUpdate(new Update("stop", ""));
            }
        }
        // Handle some errors with requests failing while we get an application up and running
        if (initializing != null) {
            if (this._initializing.started && !this._initializing.completed) {
                if (this._initializing.message == "Compressing files...")
                    this._initializing.total = -1;
                this.getCompressionProgress();
                return;
            }
        }
        if (uploading != null) {
            if (this._uploading.started && !this._uploading.completed) {
                this.getUploadProgress();
                return;
            }
        }
        if (requisitioning != null) {
            if (this._requisitioning.started && !this._requisitioning.completed) {
                if (this._requisitioning.message == "Waiting for cloud provider...")
                    this._requisitioning.total = -1;
                return;
            }
        }
    }
    get changed() {
        return this._changed;
    }
    // Static function for generating an app from controller synchronization structure
    static reconstruct(appMap) {
        // Reconstruct the app
        const initializing = [];
        for (let i = 0; i < appMap.initializing.length; i++) {
            initializing.push(new Update(appMap.initializing[i], appMap.initializingmod[i]));
        }
        const uploading = [];
        for (let i = 0; i < appMap.uploading.length; i++) {
            uploading.push(new Update(appMap.uploading[i], appMap.uploadingmod[i]));
        }
        const requisitioning = [];
        for (let i = 0; i < appMap.requisitioning.length; i++) {
            requisitioning.push(new Update(appMap.requisitioning[i], appMap.requisitioningmod[i]));
        }
        const running = [];
        for (let i = 0; i < appMap.running.length; i++) {
            running.push(new Update(appMap.running[i], appMap.runningmod[i]));
        }
        var app = new App(appMap.name, JSON.parse(appMap.notebook), appMap.script, appMap.uuid, initializing, uploading, requisitioning, running, new Date(appMap.timestamp));
        // Add modules
        for (let module of appMap.modules) {
            const output = [];
            for (let i = 0; i < module.output.length; i++) {
                output.push(new Update(module.output[i], module.outputmod[i]));
            }
            const files = [];
            for (let i = 0; i < module.files.length; i++) {
                if (module.files[i] != '') {
                    files.push(new OutputFile(module.files[i], module.filesmod[i], module.filessize[i]));
                }
            }
            var mod = new Module(module.uuid, appMap.uuid, app.addOutput, module.machine ? Object.setPrototypeOf(module.machine, Machine.prototype) : null, module.token, output, module.updates, files);
            app.addModule(mod);
            if (mod.modStatus == Status.Running) {
                // The module is still running
                if (!app._initializing.completed || !app._uploading.completed || !app._requisitioning.completed || !app._running.completed) {
                    // Don't poll the module if the app is not running
                    mod.startPolling(app.interactive);
                }
            }
        }
        if (!app._initializing.completed || !app._uploading.completed || !app._requisitioning.completed || !app._running.completed) {
            app.startPolling();
        }
        return app;
    }
    get script() {
        return this._script;
    }
    get notebook() {
        return this._notebook;
    }
    get name() {
        return this._name;
    }
    get uuid() {
        return this._uuid;
    }
    get modules() {
        return this._modules;
    }
    get initializing() {
        return this._initializing;
    }
    get uploading() {
        return this._uploading;
    }
    get requisitioning() {
        return this._requisitioning;
    }
    get running() {
        return this._running;
    }
    get timestamp() {
        return this._timestamp;
    }
    get failed() {
        for (let mod of this.modules) {
            if (mod.error)
                return true;
        }
        return this._initializing.error || this._uploading.error || this._requisitioning.error || this._running.error;
    }
    get interactive() {
        return this._interactive;
    }
    get sessionToken() {
        for (let mod of this.modules) {
            if (mod.sessionToken)
                return mod.sessionToken;
        }
        return undefined;
    }
    get sessionPort() {
        for (let mod of this.modules) {
            if (mod.sessionPort)
                return mod.sessionPort;
        }
        return undefined;
    }
    get machine() {
        for (let mod of this.modules) {
            if (mod.machine)
                return mod.machine;
        }
        return undefined;
    }
    addModule(mod) {
        mod.changed.connect(() => this._changed.emit(this), this);
        this._modules.push(mod);
    }
    async previewNotebook(printRecommendations) {
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/preview-notebook";
        const notebook = {
            content: JSON.stringify(this._notebook),
            path: this.name,
        };
        const init = {
            method: 'POST',
            body: JSON.stringify({
                notebook: notebook,
            }),
        };
        return ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (printRecommendations) {
                console.log("////");
                console.log("///  Start Recommendations: ");
                console.log("//");
                for (let machine of body.machines) {
                    console.log(Object.setPrototypeOf(machine, Machine.prototype));
                }
                console.log("//");
                console.log("///  End Recommendations: ");
                console.log("////");
            }
            if (body.machines.length == 0)
                return [new NoMachine()]; // we have no recommendations
            const machines = [];
            for (let machine of body.machines) {
                machines.push(Object.setPrototypeOf(machine, Machine.prototype));
            }
            return machines;
        });
    }
    // We only want to add this app to the app tracker if the initialization succeeds
    async setupNotebook(appTracker) {
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/setup-notebook";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                name: this._name,
                timestamp: this._timestamp.toISOString(),
                notebook: {
                    path: this._name,
                    content: JSON.stringify(this._notebook),
                },
                script: this._script,
            }),
        };
        return ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            Global.jobLaunched.emit(void 0);
            this._uuid = body.uuid;
            this._initializing.appUUID = this._uuid;
            this._uploading.appUUID = this._uuid;
            this._requisitioning.appUUID = this._uuid;
            this._running.appUUID = this._uuid;
            this._initializing.addUpdate(new Update("Initializing...", ""));
            appTracker.addApp(this);
            this._changed.emit(this);
            this.launchNotebook();
        });
    }
    getLaunchStatus() {
        // If there is an unsigned agreement, do not poll
        if (Global.user != null && Global.user.unsignedAgreement) {
            if (!this.failed)
                setTimeout(() => this.getLaunchStatus(), 500);
            return;
        }
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/get-launch-status";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                uuid: this._uuid,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status == 204) {
                if (!this.failed)
                    setTimeout(() => this.getLaunchStatus(), 200);
                return;
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                if (!this.failed)
                    setTimeout(() => this.getLaunchStatus(), 200);
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body) {
                if (body.status == "Finished") {
                    for (let i = 0; i < body.modules.length; i++) {
                        const mod = new Module(body.modules[i], this._uuid, this.addOutput);
                        this.addModule(mod);
                        mod.startPolling(this.interactive);
                    }
                }
                else if (body.status == "Failed") {
                    if (!this._initializing.completed) {
                        this._initializing.addUpdate(new Update(body.message || 'Initialization failed', ""));
                        this._initializing.addUpdate(new Update("error", ""));
                        this._initializing.addUpdate(new Update("stop", ""));
                    }
                    else if (!this._uploading.completed) {
                        this._uploading.addUpdate(new Update(body.message || 'File upload failed', ""));
                        this._uploading.addUpdate(new Update("error", ""));
                        this._uploading.addUpdate(new Update("stop", ""));
                    }
                    if (body.snackbar) {
                        Global.snackbarChange.emit(new Snackbar(body.snackbar, { variant: 'error', }));
                    }
                }
                else {
                    if (!this.failed)
                        setTimeout(() => this.getLaunchStatus(), 500);
                }
                this._changed.emit(this);
            }
        });
    }
    getCompressionProgress() {
        // If there is an unsigned agreement, do not poll
        if (Global.user != null && Global.user.unsignedAgreement) {
            if (!this.failed)
                setTimeout(() => this.getCompressionProgress(), 500);
            return;
        }
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/get-launch-compression-progress";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                uuid: this._uuid,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status == 204) {
                if (!this.failed)
                    setTimeout(() => this.getCompressionProgress(), 200);
                return;
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                if (!this.failed)
                    setTimeout(() => this.getCompressionProgress(), 200);
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body) {
                if (this._initializing.message != "Compressing files...") {
                    this._initializing.addUpdate(new Update("Compressing files...", ""));
                }
                this._initializing.loaded = body.read;
                this._initializing.total = body.total;
                if (body.read != 0 && body.read == body.total) {
                    // Do nothing
                }
                else {
                    if (!this.failed)
                        setTimeout(() => this.getCompressionProgress(), 200);
                }
                this._changed.emit(this);
            }
        });
    }
    getUploadProgress() {
        // If there is an unsigned agreement, do not poll
        if (Global.user != null && Global.user.unsignedAgreement) {
            if (!this.failed)
                setTimeout(() => this.getUploadProgress(), 500);
            return;
        }
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/get-launch-upload-progress";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                uuid: this._uuid,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status == 204) {
                if (!this.failed)
                    setTimeout(() => this.getUploadProgress(), 200);
                return;
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                if (!this.failed)
                    setTimeout(() => this.getUploadProgress(), 200);
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body) {
                if (!this._initializing.completed) {
                    this._initializing.total = 0;
                    this._initializing.addUpdate(new Update("stop", ""));
                    this._uploading.addUpdate(new Update("Uploading files...", ""));
                }
                this._uploading.loaded = body.read;
                this._uploading.total = body.total;
                if (body.read != 0 && body.read == body.total) {
                    this._uploading.addUpdate(new Update("stop", ""));
                    this.startPolling();
                }
                else {
                    if (!this.failed)
                        setTimeout(() => this.getUploadProgress(), 200);
                }
                this._changed.emit(this);
            }
        });
    }
    // Convert and send a python notebook to the REST interface for deployment
    async launchNotebook() {
        const optumi = new OptumiMetadata(this._notebook["metadata"]["optumi"] || {});
        const uploadFiles = optumi.upload.fileVars;
        const requirementsFileName = optumi.upload.requirements;
        const compressFiles = Global.user.compressFilesEnabled;
        var data = {};
        if (requirementsFileName != null) {
            data.requirementsFile = requirementsFileName;
        }
        data.dataFiles = [];
        for (var uploadEntry of uploadFiles) {
            if (uploadEntry.type == 'directory') {
                for (var file of (await FileServerUtils.getRecursiveTree(uploadEntry.path))) {
                    data.dataFiles.push(file);
                }
            }
            else {
                data.dataFiles.push(uploadEntry.path);
            }
        }
        data.compress = compressFiles;
        data.uuid = this._uuid;
        data.notebook = {
            path: this._name,
            content: JSON.stringify(this._notebook),
        };
        data.timestamp = this._timestamp.toISOString();
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/launch-notebook";
        const init = {
            method: 'POST',
            body: JSON.stringify(data),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                this._initializing.addUpdate(new Update('Initialization failed', ""));
                this._initializing.addUpdate(new Update('error', ""));
                this._initializing.addUpdate(new Update('stop', ""));
                throw new ServerConnection.ResponseError(response);
            }
        });
        this.getLaunchStatus();
        if (compressFiles && data.dataFiles.length != 0)
            this.getCompressionProgress();
        this.getUploadProgress();
    }
    getAppStatus() {
        if (this.initializing.error)
            return Status.Completed;
        if (this.uploading.error)
            return Status.Completed;
        if (this.requisitioning.error)
            return Status.Completed;
        if (this.running.completed)
            return Status.Completed;
        if (this.uploading.completed)
            return Status.Running;
        return Status.Initializing;
    }
    getAppMessage() {
        var message = "";
        if (this._initializing.message != "")
            message = this._initializing.message;
        if (this._uploading.message != "")
            message = this._uploading.message;
        if (this._requisitioning.message != "")
            message = this._requisitioning.message;
        if (this._running.message != "")
            message = this._running.message;
        // We will say a session is starting until we can connect to it
        if (this.interactive && message == 'Running...' && !(this.modules.length > 0 && this.modules[0].sessionReady))
            return 'Starting...';
        // We call a terminated app 'closed'
        if (this.interactive && message == 'Terminated')
            return 'Closed';
        return message;
    }
    getTimeElapsed() {
        if (!this._initializing.completed)
            return undefined;
        if (!this._uploading.completed)
            return undefined;
        if (!this._requisitioning.completed)
            return undefined;
        return this._running.elapsed;
    }
    getEndTime() {
        if (!this._initializing.completed)
            return undefined;
        if (!this._uploading.completed)
            return undefined;
        if (!this._requisitioning.completed)
            return undefined;
        return this._running.endTime;
    }
    getCost() {
        if (this.getTimeElapsed() == undefined)
            return undefined;
        if (this.machine == undefined)
            return undefined;
        var rate = this.machine.rate;
        const split = this.getTimeElapsed().split(':');
        if (split.length == 3) {
            const hours = +split[0];
            const minutes = +split[1];
            const seconds = +split[2];
            const cost = ((hours * rate) + (minutes * rate / 60) + (seconds * rate / 3600));
            return (cost.toFixed(2) == '0.00' ? '< $0.01' : '~ $' + cost.toFixed(2));
        }
        else {
            const minutes = +split[0];
            const seconds = +split[1];
            const cost = ((minutes * rate / 60) + (seconds * rate / 3600));
            return (cost.toFixed(2) == '0.00' ? '< $0.01' : '~ $' + cost.toFixed(2));
        }
    }
    getShowLoading() {
        if (!this._initializing.completed)
            return this._initializing.loaded != this._initializing.total;
        if (!this._uploading.completed)
            return this._uploading.loaded != this._uploading.total;
        if (!this._requisitioning.completed)
            return this._requisitioning.loaded != this._requisitioning.total;
        if (!this._running.completed)
            return this._running.loaded != this._running.total;
        return false;
    }
    getPercentLoaded() {
        if (!this._initializing.completed)
            return undefined;
        if (!this._uploading.completed)
            return this._uploading.total == -1 ? undefined : this._uploading.loaded / this._uploading.total;
        if (!this._requisitioning.completed)
            return this._requisitioning.total == -1 ? undefined : this._requisitioning.loaded / this._requisitioning.total;
        if (!this._running.completed)
            return this._running.total == -1 ? undefined : this._running.loaded / this._running.total;
        return undefined;
    }
    getLoadingTooltip() {
        if (!this.getShowLoading())
            return undefined;
        if (!this._initializing.completed)
            return this._initializing.loaded + '/' + this._initializing.total + ' files';
        if (!this._uploading.completed)
            return this._uploading.total == -1 ? '' : FormatUtils.styleCapacityUnitValue()(this._uploading.loaded) + '/' + FormatUtils.styleCapacityUnitValue()(this._uploading.total);
        if (!this._requisitioning.completed)
            return this._requisitioning.total == -1 ? '' : FormatUtils.styleCapacityUnitValue()(this._requisitioning.loaded / Math.pow(1024, 2)) + '/' + FormatUtils.styleCapacityUnitValue()(this._requisitioning.total / Math.pow(1024, 2));
        if (!this._running.completed)
            return this._running.total == -1 ? '' : FormatUtils.styleCapacityUnitValue()(this._running.loaded / Math.pow(1024, 2)) + '/' + FormatUtils.styleCapacityUnitValue()(this._running.total / Math.pow(1024, 2));
    }
    getError() {
        if (this.failed) {
            return true;
        }
        for (var mod of this._modules) {
            if (mod.error) {
                return true;
            }
        }
        return false;
    }
    receiveUpdate() {
        // If there is an unsigned agreement, do not poll
        if (Global.user != null && Global.user.unsignedAgreement) {
            setTimeout(() => this.receiveUpdate(), 500);
            return;
        }
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/pull-workload-status-update";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                uuid: this._uuid,
                lastInitializingLine: this._initializing.length.toString(),
                lastUploadingLine: this._uploading.length.toString(),
                lastRequisitioningLine: this._requisitioning.length.toString(),
                lastRunningLine: this._running.length.toString(),
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (this._polling) {
                // If we are polling, send a new request in 2 seconds
                setTimeout(() => this.receiveUpdate(), 2000);
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401) {
                    ;
                    this.stopPolling();
                    Global.user = null;
                }
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body.initializing != null) {
                for (let i = 0; i < body.initializing.length; i++) {
                    this._initializing.addUpdate(new Update(body.initializing[i], body.initializingmod[i]), false);
                }
            }
            if (body.uploading != null) {
                for (let i = 0; i < body.uploading.length; i++) {
                    this._uploading.addUpdate(new Update(body.uploading[i], body.uploadingmod[i]), false);
                }
            }
            if (body.requisitioning != null) {
                for (let i = 0; i < body.requisitioning.length; i++) {
                    this._requisitioning.addUpdate(new Update(body.requisitioning[i], body.requisitioningmod[i]), false);
                    // Special case a warning when we fail to get a machine and start trying a new one
                    if (body.requisitioning[i] == 'Machine unavailable, trying another') {
                        Global.snackbarChange.emit(new Snackbar(body.requisitioning[i], { variant: 'warning', }));
                    }
                    // Special case loading bar while waiting for a server
                    if (this._requisitioning.message == "Waiting for cloud provider...") {
                        this._requisitioning.total = -1;
                    }
                    else {
                        this._requisitioning.total = 0;
                    }
                }
            }
            if (body.running != null) {
                for (let i = 0; i < body.running.length; i++) {
                    if (body.running[i] == 'Completed') {
                        Global.snackbarChange.emit(new Snackbar(body.running[i], { variant: 'success', }));
                    }
                    else if (body.running[i] == 'Failed') {
                        Global.snackbarChange.emit(new Snackbar(body.running[i], { variant: 'error', }));
                    }
                    this._running.addUpdate(new Update(body.running[i], body.runningmod[i]), false);
                }
            }
            if (this.getAppStatus() == Status.Completed) {
                this.stopPolling();
            }
            this._changed.emit(this);
        });
    }
    startPolling() {
        this._polling = true;
        this.receiveUpdate();
    }
    stopPolling() {
        this._polling = false;
        for (let module of this._modules) {
            module.stopPolling(this.interactive);
        }
    }
    getComponent() {
        return React.createElement(AppComponent, { key: this._uuid, app: this });
    }
    getIdentityComponent() {
        var app = this;
        return (React.createElement("div", { style: { paddingLeft: '3px' }, title: app.name },
            " ",
            React.createElement("div", { style: { paddingBottom: '3px', fontSize: '13px', lineHeight: '1', fontWeight: 'normal', whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' } }, app.name.split('/').pop()),
            React.createElement("span", { style: { fontSize: '13px', lineHeight: '1', fontWeight: 'normal', color: 'gray' } }, this.formatTime())));
    }
}
class AppComponent extends React.Component {
    constructor(props) {
        super(props);
        this._isMounted = false;
        this.getDeleteJobPreventValue = () => {
            return Global.user.deleteJobPreventEnabled;
        };
        this.saveDeleteJobPreventValue = (prevent) => {
            Global.user.deleteJobPreventEnabled = prevent;
        };
        this.handleDeleteClicked = () => {
            this.checkAndSetState({ waiting: true, spinning: false });
            setTimeout(() => this.checkAndSetState({ spinning: true }), 1000);
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/teardown-notebook";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    uuid: this.props.app.uuid,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                this.checkAndSetState({ waiting: false });
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    throw new ServerConnection.ResponseError(response);
                }
                Global.user.appTracker.removeApp(this.props.app.uuid);
            }, () => this.checkAndSetState({ waiting: false }));
        };
        this.getStopJobPreventValue = () => {
            return Global.user.stopJobPreventEnabled;
        };
        this.saveStopJobPreventValue = (prevent) => {
            Global.user.stopJobPreventEnabled = prevent;
        };
        this.handleStopClicked = () => {
            this.checkAndSetState({ waiting: true, spinning: false });
            setTimeout(() => this.checkAndSetState({ spinning: true }), 1000);
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/stop-notebook";
            const init = {
                method: 'POST',
                body: JSON.stringify({
                    uuid: this.props.app.uuid,
                }),
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    throw new ServerConnection.ResponseError(response);
                }
            }, () => this.checkAndSetState({ waiting: false }));
        };
        this.getStatusColor = () => {
            const appStatus = this.props.app.getAppStatus();
            if (this.props.app.interactive && this.props.app.getAppMessage() == 'Closed') {
                return StatusColor.DARK_GRAY;
            }
            if (this.props.app.getError()) {
                return StatusColor.RED;
            }
            else {
                if (appStatus == Status.Initializing) {
                    return StatusColor.BLUE;
                }
                else {
                    return StatusColor.GREEN;
                }
            }
        };
        this.render = () => {
            var tags = [];
            var app = this.props.app;
            // Get the right progress message...
            var appMessage = app.getAppMessage();
            tags.push(React.createElement(Tag, { id: app.uuid + appMessage, key: 'appMessage' + appMessage, icon: app.getAppStatus() == Status.Completed ? (app.getError() ? ((app.getAppMessage() == 'Closed' ? (React.createElement(ClearIcon, { style: { height: '14px', width: '14px', fill: 'gray' } })) : (React.createElement(ClearIcon, { style: { height: '14px', width: '14px', fill: '#f48f8d' } })))) : (React.createElement(CheckIcon, { style: { height: '14px', width: '14px', fill: '#68da7c' } }))) : undefined, label: appMessage, color: (this.props.app.getShowLoading() || app.getAppStatus() == Status.Completed) ? this.getStatusColor() : undefined, showLoading: this.props.app.getShowLoading(), percentLoaded: this.props.app.getPercentLoaded(), tooltip: this.props.app.getLoadingTooltip() }));
            var appElapsed = app.getTimeElapsed();
            tags.push(React.createElement(Tag, { key: 'appElapsed' + appElapsed, label: appElapsed }));
            var appCost = app.getCost();
            if (appCost)
                tags.push(React.createElement(Tag, { key: 'appCost' + appCost, label: appCost }));
            return (React.createElement(React.Fragment, null,
                React.createElement(StatusWrapper, { key: this.props.app.uuid, statusColor: app.getAppStatus() == Status.Completed ? 'var(--jp-layout-color2)' : this.getStatusColor() },
                    React.createElement(InfoSkirt, { leftButton: React.createElement(DetailsDialog, { app: this.props.app }), rightButton: (this.props.app.requisitioning.completed && !this.props.app.requisitioning.error) && !this.props.app.running.completed ? (React.createElement(React.Fragment, null,
                            React.createElement(WarningPopup, { open: this.state.showStopJobPopup, headerText: "Are you sure?", bodyText: (() => {
                                    if (this.props.app.interactive) {
                                        return "This session is active. If you close it, the session cannot be resumed.";
                                    }
                                    else {
                                        return "This job is running. If you terminate it, the job cannot be resumed.";
                                    }
                                })(), preventText: "Don't ask me again", cancel: {
                                    text: `Cancel`,
                                    onCancel: (prevent) => {
                                        // this.saveStopJobPreventValue(prevent)
                                        this.checkAndSetState({ showStopJobPopup: false });
                                    },
                                }, continue: {
                                    text: (() => {
                                        if (this.props.app.interactive) {
                                            return "Close it";
                                        }
                                        else {
                                            return "Terminate it";
                                        }
                                    })(),
                                    onContinue: (prevent) => {
                                        this.checkAndSetState({ showStopJobPopup: false });
                                        this.saveStopJobPreventValue(prevent);
                                        this.handleStopClicked();
                                    },
                                    color: `error`,
                                } }),
                            React.createElement(IconButton, { disabled: this.state.waiting, onClick: () => {
                                    if (this.getStopJobPreventValue()) {
                                        this.handleStopClicked();
                                    }
                                    else {
                                        this.checkAndSetState({ showStopJobPopup: true });
                                    }
                                }, style: { position: 'relative', display: 'inline-block', width: '36px', height: '36px', padding: '3px' } },
                                React.createElement(StopIcon, { style: { position: 'relative', width: '30px', height: '30px', padding: '3px' } }),
                                this.state.waiting && this.state.spinning && React.createElement(CircularProgress, { size: '30px', style: { position: 'absolute' } })))) : (React.createElement(React.Fragment, null,
                            React.createElement(WarningPopup, { open: this.state.showDeleteJobPopup, headerText: "Are you sure?", bodyText: (() => {
                                    if (this.props.app.interactive) {
                                        return "You will lose all session information and any files that have not been downloaded. This cannot be undone.";
                                    }
                                    else {
                                        return "You will lose all job information and any files that have not been downloaded. This cannot be undone.";
                                    }
                                })(), preventText: "Don't ask me again", cancel: {
                                    text: `Cancel`,
                                    onCancel: (prevent) => {
                                        // this.saveDeleteJobPreventValue(prevent)
                                        this.checkAndSetState({ showDeleteJobPopup: false });
                                    },
                                }, continue: {
                                    text: `Delete it`,
                                    onContinue: (prevent) => {
                                        this.checkAndSetState({ showDeleteJobPopup: false });
                                        this.saveDeleteJobPreventValue(prevent);
                                        this.handleDeleteClicked();
                                    },
                                    color: `error`,
                                } }),
                            React.createElement(IconButton, { disabled: this.state.waiting || !this.props.app.initializing.completed, onClick: () => {
                                    if (this.getDeleteJobPreventValue()) {
                                        this.handleDeleteClicked();
                                    }
                                    else {
                                        this.checkAndSetState({ showDeleteJobPopup: true });
                                    }
                                }, style: { position: 'relative', display: 'inline-block', width: '36px', height: '36px', padding: '3px' } },
                                React.createElement(DeleteIcon, { style: { position: 'relative', width: '30px', height: '30px', padding: '3px' } }),
                                this.state.waiting && this.state.spinning && React.createElement(CircularProgress, { size: '30px', style: { position: 'absolute' } })))), tags: tags }, this.props.app.getIdentityComponent()))));
        };
        this.shouldComponentUpdate = (nextProps, nextState) => {
            try {
                if (JSON.stringify(this.props) != JSON.stringify(nextProps))
                    return true;
                if (JSON.stringify(this.state) != JSON.stringify(nextState))
                    return true;
                if (Global.shouldLogOnRender)
                    console.log('SuppressedRender');
                return false;
            }
            catch (error) {
                return true;
            }
        };
        this.checkAndSetState = (map) => {
            if (this._isMounted) {
                this.setState(map);
            }
        };
        // Will be called automatically when the component is mounted
        this.componentDidMount = () => {
            this._isMounted = true;
        };
        // Will be called automatically when the component is unmounted
        this.componentWillUnmount = () => {
            this._isMounted = false;
        };
        this.state = {
            waiting: false,
            spinning: false,
            showDeleteJobPopup: false,
            showStopJobPopup: false,
        };
    }
}
