/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import * as React from 'react';
import { Global } from '../Global';
import withStyles from '@material-ui/core/styles/withStyles';
import { OutlinedInput, FormControl, FormHelperText, MenuItem, Select } from '@material-ui/core';
const StyledMenuItem = withStyles({
    root: {
        fontSize: 'var(--jp-ui-font-size1)',
        padding: '3px 3px 3px 6px',
        justifyContent: 'center',
    }
})(MenuItem);
const StyledOutlinedInput = withStyles({
    input: {
        fontSize: '12px',
        padding: '3px 6px 3px 6px',
    },
    adornedEnd: {
        paddingRight: '0px',
    },
})(OutlinedInput);
export class TextBoxDropdown extends React.Component {
    constructor(props) {
        super(props);
        this.getStyledSelect = () => {
            return withStyles({
                root: {
                    fontSize: "var(--jp-ui-font-size1)",
                    width: '68px',
                },
                iconOutlined: {
                    right: '0px'
                }
            })(Select);
        };
        this.handleValueChange = (event) => {
            try {
                let unit = this.state.unit;
                const charTyped = event.currentTarget.value.replace(/[0-9\.]/g, '');
                if (charTyped.length > 0) {
                    for (let unitValue of this.props.unitValues) {
                        if (unitValue.unit.charAt(0) === charTyped[0]) {
                            unit = unitValue.unit;
                            break;
                        }
                    }
                }
                let value = event.currentTarget.value === '' ? 0 : Number.parseFloat(event.currentTarget.value.replace(/[^0-9\.]/g, ''));
                const denormalizedValue = this.denormalize(unit, value);
                let errorMessage = '';
                if (this.props.minValue && denormalizedValue < this.props.minValue) {
                    const [minUnit, minValue] = this.normalize(this.props.minValue);
                    errorMessage = `Minimum value is ${minValue}${minUnit}`;
                }
                else if (this.props.maxValue && denormalizedValue > this.props.maxValue) {
                    const [maxUnit, maxValue] = this.normalize(this.props.maxValue);
                    errorMessage = `Maximum value is ${maxValue}${maxUnit}`;
                }
                this.setState({ unit: unit, value: value === NaN ? this.state.value : value, errorMessage: errorMessage });
            }
            catch (error) {
                // If Number.parseFloat fails, we don't want to update the values
            }
        };
        this.handleValueFocus = (event) => {
            if (this.props.onFocus) {
                this.props.onFocus(event);
            }
        };
        this.handleValueBlur = (event) => {
            if (this.props.onBlur) {
                this.props.onBlur(event);
            }
            const [normalizedUnit, normalizedValue] = this.normalize(this.denormalize(this.state.unit, this.state.value));
            this.setState({ unit: normalizedUnit, value: normalizedValue });
            this.saveChanges();
        };
        this.handleUnitChange = (event) => {
            const unit = event.target.value;
            const value = this.state.value;
            const denormalizedValue = this.denormalize(unit, value);
            let errorMessage = '';
            if (this.props.minValue && denormalizedValue < this.props.minValue) {
                const [minUnit, minValue] = this.normalize(this.props.minValue);
                errorMessage = `Minimum value is ${minValue}${minUnit}`;
            }
            else if (this.props.maxValue && denormalizedValue > this.props.maxValue) {
                const [maxUnit, maxValue] = this.normalize(this.props.maxValue);
                errorMessage = `Maximum value is ${maxValue}${maxUnit}`;
            }
            this.setState({ unit: unit, errorMessage: errorMessage });
        };
        this.handleUnitBlur = () => {
            const [normalizedUnit, normalizedValue] = this.normalize(this.denormalize(this.state.unit, this.state.value));
            this.setState({ unit: normalizedUnit, value: normalizedValue });
            this.saveChanges();
        };
        this.saveChanges = () => {
            if (this.state.errorMessage === '') {
                var saveErrorMessage = this.props.saveValue(this.denormalize(this.state.unit, this.state.value));
                if (typeof saveErrorMessage === 'string' && saveErrorMessage !== '') {
                    this.setState({ errorMessage: saveErrorMessage });
                }
            }
        };
        this.render = () => {
            if (Global.shouldLogOnRender)
                console.log('ResourceTextBoxRender (' + new Date().getSeconds() + ')');
            return (React.createElement("div", { style: Object.assign({
                    display: 'inline-flex',
                    width: '100%',
                    padding: '6px 0px',
                    textAlign: 'center',
                    justifyContent: 'center'
                }, this.props.style) },
                React.createElement("div", { style: {
                        minWidth: this.props.labelWidth || '68px',
                        margin: '0px 12px',
                        lineHeight: '24px',
                    } }, this.props.label),
                React.createElement(FormControl, { title: this.state.errorMessage, error: this.state.errorMessage != '', variant: 'outlined', style: {
                        width: '100%',
                        margin: '2px 3px 2px 6px',
                        height: this.props.helperText ? '32px' : '20px',
                    } },
                    React.createElement(StyledOutlinedInput, { inputRef: this.textField, value: this.state.value, placeholder: this.props.placeholder, onChange: this.handleValueChange, onFocus: this.handleValueFocus, onBlur: this.handleValueBlur, onKeyDown: (event) => { if (event.key == 'Enter' || event.key == 'Escape')
                            this.textField.current.blur(); } }),
                    this.props.helperText &&
                        React.createElement(FormHelperText, { style: { fontSize: '10px', lineHeight: '10px', margin: '4px 6px', whiteSpace: 'nowrap' } }, this.props.helperText)),
                React.createElement(this.StyledSelect, { value: this.state.unit, variant: 'outlined', onChange: this.handleUnitChange, onBlur: this.handleUnitBlur, SelectDisplayProps: { style: { padding: '3px 20px 3px 6px' } }, MenuProps: { MenuListProps: { style: { paddingTop: '6px', paddingBottom: '6px' } } }, style: { margin: '2px 6px 2px 3px', height: '20px' } }, this.props.unitValues.map(unitValue => React.createElement(StyledMenuItem, { key: unitValue.unit, value: unitValue.unit }, unitValue.unit)))));
        };
        this.shouldComponentUpdate = (nextProps, nextState) => {
            try {
                if (JSON.stringify(this.props) != JSON.stringify(nextProps))
                    return true;
                if (JSON.stringify(this.state) != JSON.stringify(nextState))
                    return true;
                if (Global.shouldLogOnRender)
                    console.log('SuppressedRender');
                return false;
            }
            catch (error) {
                return true;
            }
        };
        this.textField = React.createRef();
        const [unit, value] = this.normalize(this.props.getValue());
        this.state = {
            unit: unit,
            value: value,
            errorMessage: '',
        };
        this.StyledSelect = this.getStyledSelect();
    }
    normalize(value) {
        let largestSmallerUnit = this.props.unitValues[0].unit;
        let largestSmallerValue = this.props.unitValues[0].value;
        for (let unitValue of this.props.unitValues) {
            if (unitValue.value <= value && unitValue.value > largestSmallerValue) {
                largestSmallerUnit = unitValue.unit;
                largestSmallerValue = unitValue.value;
            }
        }
        return [largestSmallerUnit, Number.parseFloat((value / largestSmallerValue).toPrecision(3))];
    }
    denormalize(unit, value) {
        for (let unitValue of this.props.unitValues) {
            if (unitValue.unit === unit) {
                return unitValue.value * value;
            }
        }
        return value;
    }
}
