/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import * as React from 'react';
import { Global } from '../../Global';
import { ServerConnection } from '@jupyterlab/services';
import { ListItem, IconButton, CircularProgress, } from '@material-ui/core';
import GetAppIcon from '@material-ui/icons/GetApp';
import DoneIcon from '@material-ui/icons/Done';
export class OutputFileEntry extends React.Component {
    constructor(props) {
        super(props);
        // We need to know if the component is mounted to change state
        this._isMounted = false;
        this.formatSize = (value) => {
            if (value < Math.pow(1024, 1)) {
                return value.toFixed() + ' B';
            }
            else if (value < Math.pow(1024, 2)) {
                return (value / Math.pow(1024, 1)).toFixed(1) + ' KiB';
            }
            else if (value < Math.pow(1024, 3)) {
                return (value / Math.pow(1024, 2)).toFixed(1) + ' MiB';
            }
            else if (value < Math.pow(1024, 4)) {
                return (value / Math.pow(1024, 3)).toFixed(1) + ' GiB';
            }
            else if (value < Math.pow(1024, 5)) {
                return (value / Math.pow(1024, 4)).toFixed(1) + ' TiB';
            }
            else {
                return (value / Math.pow(1024, 5)).toFixed(1) + ' PiB';
            }
        };
        this.formatExtraInfo = () => {
            var lastModified = this.props.lastModified == "" ? "" : new Date(this.props.lastModified).toLocaleTimeString();
            var size = this.props.size == 0 ? "" : this.formatSize(this.props.size);
            if (lastModified == "" && size == "")
                return "";
            if (lastModified == "")
                return " (" + size + ")";
            if (size == "")
                return " (" + lastModified + ")";
            return " (" + size + ", " + lastModified + ")";
        };
        // Will be called automatically when the component is mounted
        this.componentDidMount = () => {
            this._isMounted = true;
            this.update();
        };
        // Will be called automatically when the component is unmounted
        this.componentWillUnmount = () => {
            this._isMounted = false;
        };
        this.checkAndSetState = (map) => {
            if (this._isMounted) {
                this.setState(map);
            }
        };
        this.shouldComponentUpdate = (nextProps, nextState) => {
            try {
                if (JSON.stringify(this.props) != JSON.stringify(nextProps))
                    return true;
                if (JSON.stringify(this.state) != JSON.stringify(nextState))
                    return true;
                if (Global.shouldLogOnRender)
                    console.log('SuppressedRender');
                return false;
            }
            catch (error) {
                return true;
            }
        };
        this.state = {
            loaded: 0,
            total: 0,
            downloading: false,
            downloaded: false,
        };
    }
    getDownloadProgress(name) {
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/get-file-download-progress";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                // The key we use to watch download progress
                name: name,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status == 204) {
                setTimeout(() => this.getDownloadProgress(name), 200);
                return;
            }
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                setTimeout(() => this.getDownloadProgress(name), 200);
                throw new ServerConnection.ResponseError(response);
            }
            return response.json();
        }).then((body) => {
            if (body) {
                this.checkAndSetState({ loaded: body.read, total: body.total });
                if (!(body.read != 0 && body.read == body.total)) {
                    setTimeout(() => this.getDownloadProgress(name), 200);
                }
            }
        });
    }
    async saveOutputFile(name, files) {
        const settings = ServerConnection.makeSettings();
        const url = settings.baseUrl + "optumi/save-notebook-output-file";
        const init = {
            method: 'POST',
            body: JSON.stringify({
                workloadUUID: this.props.workloadUUID,
                moduleUUID: this.props.moduleUUID,
                name: name,
                files: files,
                overwrite: this.props.overwrite,
            }),
        };
        ServerConnection.makeRequest(url, init, settings).then((response) => {
            if (response.status !== 200 && response.status !== 201) {
                if (response.status == 401)
                    Global.user = null;
                throw new ServerConnection.ResponseError(response);
            }
            return response.text();
        }).then((body) => {
            this.checkAndSetState({ downloaded: true, downloading: false });
            setTimeout(() => this.checkAndSetState({ downloaded: false }), 5000);
        });
        this.getDownloadProgress(name);
    }
    // The contents of the component
    render() {
        if (Global.shouldLogOnRender)
            console.log('OutputFileEntryRender (' + new Date().getSeconds() + ')');
        return (React.createElement(ListItem, { style: { paddingTop: "2px", paddingBottom: "2px" } },
            this.state.downloaded ? (React.createElement(IconButton, { edge: "end", disabled: true },
                React.createElement(DoneIcon, null))) : (React.createElement("div", null, !this.state.downloading ? (React.createElement(IconButton, { edge: "end", disabled: this.props.disabled, onClick: () => {
                    this.saveOutputFile(this.props.name, this.props.files);
                    this.checkAndSetState({ downloading: true });
                } },
                React.createElement(GetAppIcon, null))) : (React.createElement("div", { style: { position: 'relative', height: '48px', width: '48px' } },
                React.createElement(CircularProgress, null),
                React.createElement("span", { style: { position: 'absolute', top: '12.5px', left: '9px', fontSize: '0.75rem' } }, this.state.loaded != this.state.total ? ((this.state.loaded / this.state.total) * 100).toFixed(0) + "%" : ''))))),
            React.createElement("div", { style: {
                    overflow: 'auto',
                    marginLeft: !this.state.downloading ? '24px' : '12px',
                    marginTop: '4px',
                    marginBottom: '4px',
                    opacity: this.props.disabled ? '0.5' : '1',
                } }, this.props.name + this.formatExtraInfo())));
    }
    update() {
        if (this._isMounted) {
            this.forceUpdate();
        }
    }
}
