/*
**  Copyright (C) Optumi Inc - All rights reserved.
**
**  You may only use this code under license with Optumi Inc and any distribution or modification is strictly prohibited.
**  To receive a copy of the licensing terms please write to contact@optumi.com or visit us at http://www.optumi.com.
**/
import * as React from 'react';
import { Global } from '../Global';
import { ServerConnection } from '@jupyterlab/services';
import { Divider, } from '@material-ui/core';
import { Machine } from '../models/machine/Machine';
import { Credits } from './machines/Credits';
import { Rate } from './machines/Rate';
import { Budget } from './machines/Budget';
export class MachinesPage extends React.Component {
    constructor(props) {
        super(props);
        // We need to know if the component is mounted to change state
        this._isMounted = false;
        this.polling = false;
        this.updateMachines = () => {
            const settings = ServerConnection.makeSettings();
            const url = settings.baseUrl + "optumi/get-machines";
            const init = {
                method: 'GET',
            };
            ServerConnection.makeRequest(url, init, settings).then((response) => {
                if (this.polling) {
                    // If we are polling, send a new request in 2 seconds
                    setTimeout(() => this.updateMachines(), 2000);
                }
                if (response.status !== 200 && response.status !== 201) {
                    if (response.status == 401)
                        Global.user = null;
                    throw new ServerConnection.ResponseError(response);
                }
                return response.json();
            }).then((body) => {
                var machines = [];
                for (var i = 0; i < body.machines.length; i++) {
                    machines.push(Object.setPrototypeOf(body.machines[i], Machine.prototype));
                }
                this.checkAndSetState({ machines: machines });
            });
        };
        // Will be called automatically when the component is mounted
        this.componentDidMount = () => {
            this._isMounted = true;
            this.startPolling();
        };
        // Will be called automatically when the component is unmounted
        this.componentWillUnmount = () => {
            this.stopPolling();
            Global.lastMachines = this.state.machines;
            this._isMounted = false;
        };
        this.shouldComponentUpdate = (nextProps, nextState) => {
            try {
                if (JSON.stringify(this.props) != JSON.stringify(nextProps))
                    return true;
                if (JSON.stringify(this.state) != JSON.stringify(nextState))
                    return true;
                if (Global.shouldLogOnRender)
                    console.log('SuppressedRender');
                return false;
            }
            catch (error) {
                return true;
            }
        };
        this.checkAndSetState = (map) => {
            if (this._isMounted) {
                this.setState(map);
            }
        };
        this.state = {
            machines: Global.lastMachines,
        };
    }
    // The contents of the component
    render() {
        if (Global.shouldLogOnRender)
            console.log('MachinesPageRender (' + new Date().getSeconds() + ')');
        var machines = new Array();
        for (var i = 0; i < this.state.machines.length; i++) {
            var machine = this.state.machines[i];
            if (Global.user.userExpertise >= 2 || Machine.getStateMessage(machine.state) != 'Releasing...') {
                machines.push(React.createElement("div", { key: machine.uuid, style: { display: 'inline-flex', width: '100%', padding: '6px 0px' } }, machine.getComponent()));
            }
        }
        return (React.createElement("div", { style: Object.assign({ display: 'flex', flexFlow: 'column', overflow: 'hidden' }, this.props.style) },
            React.createElement("div", { style: { margin: '0px 20px' } },
                React.createElement(Rate, null),
                React.createElement(Divider, { variant: 'middle' }),
                Global.user.userExpertise > 0 ? (React.createElement(Budget, null)) : (React.createElement(Credits, null)),
                React.createElement(Divider, { variant: 'middle' })),
            React.createElement("div", { style: { flexGrow: 1, overflowY: 'auto', padding: '6px' } }, machines.length == 0 ? (React.createElement("div", { style: { textAlign: 'center', margin: '12px' } }, "Running machines will appear here.")) : (machines))));
    }
    startPolling() {
        this.polling = true;
        this.updateMachines();
    }
    stopPolling() {
        this.polling = false;
    }
}
