"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateTransitionMetric = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Metrics on the rate limiting performed on state machine execution.
 *
 * These rate limits are shared across all state machines.
 */
class StateTransitionMetric {
    /**
     * Return the given named metric for the service's state transition metrics.
     *
     * @default average over 5 minutes
     */
    static metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ServiceMetric: 'StateTransition' },
            ...props,
        });
    }
    /**
     * Metric for the number of available state transitions.
     *
     * @default average over 5 minutes
     */
    static metricProvisionedBucketSize(props) {
        return StateTransitionMetric.metric('ProvisionedBucketSize', props);
    }
    /**
     * Metric for the provisioned steady-state execution rate.
     *
     * @default average over 5 minutes
     */
    static metricProvisionedRefillRate(props) {
        return StateTransitionMetric.metric('ProvisionedRefillRate', props);
    }
    /**
     * Metric for the number of available state transitions per second.
     *
     * @default average over 5 minutes
     */
    static metricConsumedCapacity(props) {
        return StateTransitionMetric.metric('ConsumedCapacity', props);
    }
    /**
     * Metric for the number of throttled state transitions.
     *
     * @default sum over 5 minutes
     */
    static metricThrottledEvents(props) {
        return StateTransitionMetric.metric('ThrottledEvents', { statistic: 'sum', ...props });
    }
}
exports.StateTransitionMetric = StateTransitionMetric;
//# sourceMappingURL=data:application/json;base64,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