import logging
import sys
from collections import deque
from os import chdir, path

import click
import docker
import requests
import yaml
from compose.progress_stream import stream_output

from nomnomdata.auth import NNDAuth

from .util import NomitallSession

_logger = logging.getLogger(__name__)


@click.command(name="deploy")
@click.option(
    "-p",
    "--path",
    "buildp",
    default=".",
    show_default=True,
    help="Docker build context path",
)
@click.option(
    "-f",
    "--model-file",
    "mpath",
    default="./",
    help="Path to the engine model.yaml to build, if this is a folder model.yaml will be assumed to be in there "
    "Examples: nomnom/test/waittask/model.yaml",
)
@click.option(
    "-d",
    "--docker-file",
    "dpath",
    default="./",
    help="Path to the engine dockerfile to build, if this is a folder Dockerfile will be assumed to be in there "
    "Examples: nomnom/test/waittask/engine.dockerfile",
)
@click.option(
    "-n",
    "--nomitall",
    default="nomitall-prod",
    help="Specify the nomitall to update [nomitall-prod,nomitall-stage,custom_url]",
)
@click.option(
    "-c",
    "--channel",
    default="dev",
    type=click.Choice(["stable", "beta", "dev"]),
    help="Channel to deploy to",
)
@click.option("--dry-run", is_flag=True, help="Build engine but do not deploy")
@click.option(
    "-y", "--yes", "skip_confirm", is_flag=True, help="Skip confirmation prompt"
)
def deploy(buildp, mpath, dpath, nomitall, channel, dry_run, skip_confirm):
    "Build engine docker images and optionally run tests."
    if nomitall == "nomitall-prod":
        nomitall_url = "https://user.api.nomnomdata.com/api/1/"
    elif nomitall == "nomitall-stage":
        nomitall_url = "https://staging.user.api.nomnomdata.com/api/1/"
    else:
        nomitall_url = nomitall
    chdir(buildp)
    if path.isdir(mpath):
        mpath = path.join(mpath, "model.yaml")
    if path.isdir(dpath):
        dpath = path.join(dpath, "Dockerfile")
    engine_model = yaml.full_load(open(mpath))
    engine_uuid = engine_model["uuid"]
    if channel == "beta":
        c_color = click.style(channel, fg="yellow", bold=True)
    elif channel == "stable":
        c_color = click.style(channel, fg="red", bold=True)
    else:
        c_color = click.style(channel, fg="green", bold=True)

    if not skip_confirm:
        click.confirm(
            text=f"Confirm deploy engine:\n\tUUID: {engine_uuid}\n\tChannel: {c_color}\n\tNomitall: {nomitall}\n",
            abort=True,
        )

    session = NomitallSession(prefix_url=nomitall_url)
    session.auth = NNDAuth()
    resp = session.request("get", f"engine/create-deploy-token/{engine_uuid}")
    data = resp.json()
    engine_name = data["repoUrl"]
    client = docker.from_env()
    try:
        client.ping()
    except (requests.ConnectionError, docker.errors.APIError) as e:
        raise Exception(
            "There was a problem connecting to the docker agent, ensure it is running and in a good state"
        ) from e
    image, tag = build_engine(
        engine_name=engine_name, client=client, tag=channel, docker_file=dpath,
    )
    if not dry_run:
        deploy_engine(image, tag, data)
        if channel == "stable" and nomitall == "nomitall-prod":
            deploy_engine(image, "prod", data)


def deploy_engine(image, tag, creds):
    _logger.info("----Pushing image----")
    client = docker.from_env()
    auth_config = {"username": creds["user"], "password": creds["token"]}
    _logger.info(f"Pushing {image}:{tag}")
    push_logs = client.api.push(image, tag=tag, auth_config=auth_config, stream=True)
    deque(stream_output(push_logs, sys.stdout), maxlen=0)
    _logger.info("----Successfully pushed image----")


def build_engine(engine_name, client, tag, docker_file, buildargs=None):
    _logger.info("----Building image----")
    build_logs = client.api.build(
        dockerfile=docker_file,
        tag=":".join([engine_name, tag]),
        rm=True,
        path=".",
        pull=True,
        buildargs=buildargs,
    )
    deque(stream_output(build_logs, sys.stdout), maxlen=0)
    _logger.info("----Finished building image----")
    return engine_name, tag
