"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        this.adminRole = this.createAdminRole(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: this.adminRole,
            subnets: props.subnets,
            vpc: props.vpc,
            environment: props.environment,
            onEventLayer: props.onEventLayer,
        });
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    kubernetesNetworkConfig: props.kubernetesNetworkConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.adminRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.adminRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    createAdminRole(props) {
        const stack = core_1.Stack.of(this);
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        const creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.list({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.string({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        // see https://github.com/aws/aws-cdk/issues/9027
        // these actions are the combined 'ec2:Describe*' actions taken from the EKS SLR policies.
        // (AWSServiceRoleForAmazonEKS, AWSServiceRoleForAmazonEKSForFargate, AWSServiceRoleForAmazonEKSNodegroup)
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeVpcs',
            ],
            resources: ['*'],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        return creationRole;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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