# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['configuror']

package_data = \
{'': ['*']}

install_requires = \
['pyyaml>=5.2,<6.0', 'toml>=0.10.0,<0.11.0']

setup_kwargs = {
    'name': 'configuror',
    'version': '0.1.3a0',
    'description': 'A configuration management toolkit',
    'long_description': "# configuror\n\n[![Pypi version](https://img.shields.io/pypi/v/configuror.svg)](https://pypi.org/project/configuror/)\n[![Build Status](https://travis-ci.com/lewoudar/configuror.svg?branch=master)](https://travis-ci.com/lewoudar/configuror)\n[![Windows Build Status](https://img.shields.io/appveyor/ci/lewoudar/configuror/master.svg?label=Windows)](https://ci.appveyor.com/project/lewoudar/configuror)\n[![Coverage Status](https://codecov.io/gh/lewoudar/configuror/branch/master/graphs/badge.svg?branch=master)](https://codecov.io/gh/lewoudar/configuror)\n[![Documentation Status](https://readthedocs.org/projects/configuror/badge/?version=latest)](https://configuror.readthedocs.io/en/latest/?badge=latest)\n[![License Apache 2](https://img.shields.io/hexpm/l/plug.svg)](http://www.apache.org/licenses/LICENSE-2.0)\n\nYour configuration management toolkit!\n\n## Why?\n\nWhile using [Flask](http://flask.pocoo.org/docs/1.0/), I realized that their Config class could be useful for any type\nof project. And the utility became more and more obvious to me when I looked at a project like\n[Ansible](https://docs.ansible.com/ansible/latest/index.html). If you look the \n[section](https://docs.ansible.com/ansible/latest/user_guide/playbooks_variables.html#variable-precedence-where-should-i-put-a-variable)\nwhere they define variable precedence, you will notice that there may be several locations for the configuration files,\nand these configuration files can be written in different formats (json, yaml..).\n\nWhat if there was a simple tool that would aggregate information from different sources in the order you wanted? This\nis **why** the configuror project exists!\n\n## Installation\n\n```bash\npip install configuror\n```\n\nconfiguror works starting from **python 3.6**. It has a few dependencies:\n- [pyyaml](https://pypi.org/project/PyYAML/) >= 5.1\n- [toml](https://pypi.org/project/toml/)\n\n## Documentation\n\nThe documentation is available at https://configuror.readthedocs.io/en/latest/.\n\n## Usage\n\nThe main class provided by configuror is `Config`. It is an extension of a regular dict object. There are two main ways \nto initialize it.\n\n### Using mapping files\n\n```python\nfrom configuror import Config\n\nmapping_files = {\n    'ini': ['foo.ini', 'bar.ini'],\n    'toml': ['foo.toml', 'bar.toml'],\n    'python': ['/path/to/python/file']\n}\nconfig = Config(mapping_files=mapping_files, ignore_file_absence=True)\n```\n\nYou can define a mapping of `file_type: <files>` where the `file_type` is the type of configuration file and `<files>` \nis the list of files from the lowest to the highest priority where values will be loaded.\n\nSince dictionaries are sorted starting from python3.6, the order of the keys is important as it will become the order of\nimportance of your files. For example in the example above, configuror will load values from files in the following order:\n- foo.ini\n- bar.ini\n- foo.toml\n- bar.toml\n- /path/to/python/file\n\nFor python files, only **uppercase** variables will be loaded.\n\nYou will notice the keyword argument `ignore_file_absence` in `Config` class initialization. If it is set to `True`, all\nfiles that does not exist will not raised `FileNotFoundError`. It comes in handy when you want to retrieve variables \nfrom files *that may or may not potentially exist*. By default this parameter is set to `False`.\n\nFile extension is not necessary when you use mapping files since the key is already telling which files we work with.\nThis is not the case with the second way to initialize `Config` class.\n\n### Using a list of files\n\n```python\nfrom configuror import Config\n\nfiles = [\n    'foo.yml',\n    'bar.toml',\n    'foobar.json',\n    '/path/to/python/file'\n    '.env'\n]\nconfig = Config(files=files)\n```\n\nIn this second form of initialization, you pass a list of files you want to retrieve values from the lowest to the\nhighest priority. File extension is **mandatory** here to help configuror to load the files properly.\n\nTo know file extensions supported by configuror, you can use the variable `EXTENSIONS`. it is a mapping\n`file_type: <extensions>` where `file_type` is a type of file supported like *yaml* and `extensions` is a list of\nrecognized extensions for this type of file, e.g: `[yml, yaml]`\n\nToday the file types supported are *toml*, *yaml*, *dotenv*, *ini*, *python* and *json*.\n\n### Other usages\n\nSince `Config` object is a dict-like object, you can pass arbitrary keyword arguments to initialize default values.\n\n```python\nfrom configuror import Config\n\nconfig = Config(FOO=2, BAR='a')\nprint(config)  # will print {'FOO': 2, 'BAR': 'a'}\n```\n\nYou can combine keyword arguments, mapping files and list of files at initialization. The order in which values will be\ninitialized is the following:\n- values from keyword arguments\n- values from mapping files\n- values from list of files\n\nYou can also add values from files after initialization. There are several practical methods for this:\n- `load_from_mapping_files(self, mapping_files: Dict[str, List[str]], ignore_file_absence: bool)`: It is in fact the \nmethod used under the hood when you initialized `Config` object by passing the parameter `mapping_files`.\n\n- `load_from_files(self, files: List[str], ignore_file_absence: bool)`: It is the method used under the hood when you\ninitialized `Config` objects by passing the parameter `files`. \n\n- `load_from_object(self, obj: Union[Object, str])`: `obj` can be an object or a path to a project module\n(with dotted notation). Only uppercase attributes of the corresponding object will be retrieved.\n\n- `load_from_python_file(self, filename: str, ignore_file_absence: bool)`: Loads values from an arbitrary python\nfile. It would be preferable if it were not a file related to your project (i.e a module). Only uppercase variables\nare considered.\n\n- `load_from_json(self, filename: str, ignore_file_absence: bool)`: Loads values from a json file.\n\n- `load_from_yaml(self, filename: str, ignore_file_absence: bool)`: Loads values from a yaml file.\n\n- `load_from_toml(self, filenames: Union[str, List[str]], ignore_file_absence: bool)`: Loads values from a toml file\nor a list of toml files.\n\n- `load_from_ini(self, filenames: Union[str, List], ignore_file_absence: bool, interpolation_method: str = 'basic')`:\nLoads values from an ini file or a list of ini files. There are two interpolation methods that can be used: **basic** \nor **extended** like explained in the\n[documentation](https://docs.python.org/3/library/configparser.html#interpolation-of-values).\n\n- `load_from_dotenv(self, filename: str, ignore_file_absence: bool)`: Loads values from a dotenv file.\n\nBonus: You also have the `update` method of a dict to add/update values.\n",
    'author': 'lewoudar',
    'author_email': 'lewoudar@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://configuror.readthedocs.io/en/stable',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
