#
# Copyright 2016 University of Southern California
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import os
import logging
from bdbag import urlsplit, urlunquote

Kilobyte = 1024
Megabyte = Kilobyte ** 2

SCHEME_HTTP = "http"
SCHEME_HTTPS = "https"
SCHEME_S3 = "s3"
SCHEME_GS = "gs"
SCHEME_GLOBUS = "globus"
SCHEME_FTP = "ftp"
SCHEME_SFTP = "sftp"
SCHEME_TAG = "tag"


def get_transfer_summary(total_bytes, elapsed_time):
    total_secs = elapsed_time.total_seconds()
    transferred = \
        float(total_bytes) / float(Kilobyte) if total_bytes < Megabyte else float(total_bytes) / float(Megabyte)
    throughput = str(" at %.2f MB/second" % (transferred / total_secs)) if (total_secs >= 1) else ""
    elapsed = str("Elapsed time: %s." % elapsed_time) if (total_secs > 0) else ""
    summary = "%.3f %s transferred%s. %s" % \
              (transferred, "KB" if total_bytes < Megabyte else "MB", throughput, elapsed)
    return summary


def check_transfer_size_mismatch(path, expected, total):
    if isinstance(expected, int) and isinstance(total, int):
        if expected != total:
            logging.warning("File [%s] transfer size mismatch. Expected %s bytes but received %s bytes." %
                            (path, expected, total))
            return True
    return False


def ensure_valid_output_path(url, output_path=None):
    if not output_path:
        upr = urlsplit(url, allow_fragments=False)
        output_path = os.path.join(os.curdir, urlunquote(os.path.basename(upr.path)))
    output_path = os.path.abspath(output_path)
    output_dir = os.path.dirname(output_path)
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    return output_path
