"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => (Object.assign(Object.assign({}, acc), { [k]: v.conditions })), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
        });
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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