"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToKinesisStreamsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.getPutRecordTemplate(props.putRecordRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordModel(props.putRecordRequestModel) }
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.getPutRecordsTemplate(props.putRecordsRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordsModel(props.putRecordsRequestModel) }
        });
    }
    getPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    getPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    getPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    getPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
//# sourceMappingURL=data:application/json;base64,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