# Copyright 2015 Oliver Cope
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import io
import json
import logging
import os
import pickle
import re
import tempfile
import time
from datetime import date, datetime
from collections import OrderedDict
from collections.abc import MutableMapping

__all__ = [
    "SessionMiddleware",
    "CookieIdPersistence",
    "Session",
    "FileBackend",
    "MemoryBackend",
]

logger = logging.getLogger(__name__)

VALID_ID_CHARS = (
    b"ABCDEFGHIJKLMNOP"
    b"QRSTUVWXYZabcdef"
    b"ghijklmnopqrstuv"
    b"wxyz0123456789.-"
)

assert 256 % len(VALID_ID_CHARS) == 0

COOKIE_NAME = "_obs"
SESSION_KEY = "ob.session"
ID_LEN = 40
ID_TRANS_TABLE = VALID_ID_CHARS * (256 // len(VALID_ID_CHARS))

ID_PATTERN = "[%s]{%d}" % (re.escape(VALID_ID_CHARS.decode("ASCII")), ID_LEN)
EPOCH = time.localtime(0)


def _generate_id(count=ID_LEN, _random_bytes=os.urandom, _trans=ID_TRANS_TABLE):
    """
    Return a randomly generated byte string of length ```count`` characters,
    containing only characters from ``valid_id_chars``.
    """
    return _random_bytes(count).translate(_trans).decode("ASCII")


class IdPersistence(object):
    def get_id(self, environ):
        raise NotImplementedError()

    def set_id_middleware(self, session, app, start_response, environ):
        return app(environ, start_response)


class CookieIdPersistence(IdPersistence):
    def __init__(
        self,
        cookie_name=COOKIE_NAME,
        max_age=None,
        path="/",
        domain="",
        secure=True,
        samesite="Lax",
    ):
        self.set_cookie_template = (
            cookie_name + f"=%s;HttpOnly;SameSite={samesite}"
        )
        if path:
            self.set_cookie_template += ";Path={}".format(path)
        if domain:
            self.set_cookie_template += ";Domain={}".format(domain)
        if max_age:
            self.set_cookie_template += ";Max-Age={}".format(max_age)
        if secure:
            self.set_cookie_template += ";Secure"
        self._extract_cookie = re.compile(
            r"\b{}=({})".format(re.escape(cookie_name), ID_PATTERN)
        ).search

    def get_id(self, environ):
        try:
            return self._extract_cookie(environ["HTTP_COOKIE"]).group(1)
        except (KeyError, AttributeError):
            return None

    def set_id_middleware(self, session, app, environ, start_response):
        def start_response(
            status, headers, exc_info=None, _start_response=start_response
        ):
            if session.id and session.orig_id != session.id:
                id = session.id
                if str is bytes:
                    id = id.encode("ASCII")
                headers.append(
                    ("Set-Cookie", self.set_cookie_template % (id,))
                )
            return _start_response(status, headers, exc_info)

        return app(environ, start_response)


class Serializer(object):
    def load(self, file):
        raise NotImplementedError()

    def dump(self, data, file):
        raise NotImplementedError()


class PickleSerializer(Serializer):

    # The pickle protocol version to use
    PROTOCOL = -1

    def load(self, file, _load=pickle.load):
        return _load(file)

    def dump(self, data, file, _dump=pickle.dump):
        return _dump(data, file, self.PROTOCOL)


class JSONSerializer(Serializer):
    def __init__(self, separators=(",", ":"), indent=None, sort_keys=False):
        self._iterencode = json.JSONEncoder(
            separators=separators, indent=indent, sort_keys=sort_keys
        ).iterencode

    def load(self, fp, _load=json.load):
        fp = io.TextIOWrapper(fp, encoding="utf8")
        return _load(fp)

    def dump(self, data, fp):
        write = fp.write
        for chunk in self._iterencode(data):
            write(chunk.encode("UTF-8"))


class Session(MutableMapping):

    #: The original id read from the client (will be ``None`` in the case of a
    #: new session)
    orig_id = None

    #: The session id
    id = None

    #: Set when the session is loaded from the backend
    is_loaded = False

    #: The session data dict
    data = None

    #: Set when the session is modified and cleared when it is persisted
    dirty = False

    #: Set when the id is changed
    id_needs_persisting = False

    def __init__(self, environ, backend, get_id):
        self.environ = environ
        self.backend = backend
        self.get_id = get_id
        self.data = None

    def __getitem__(self, name):
        if not self.is_loaded:
            self.load()
        return self.data[name]

    def __setitem__(self, name, value):
        if not self.is_loaded:
            self.load()
        if self.id is None:
            self.id = _generate_id()
            self.id_needs_persisting = True
        self.data[name] = value
        self.dirty = True

    def __delitem__(self, name):
        if not self.is_loaded:
            self.load()
        del self.data[name]
        self.dirty = True

    def __iter__(self):
        if not self.is_loaded:
            self.load()
        return iter(self.data)

    def __len__(self):
        if not self.is_loaded:
            self.load()
        return len(self.data)

    def load(self):
        id = self.orig_id = self.id = self.get_id(self.environ)
        if id is not None:
            try:
                self.data = self.backend.load(id)
            except BackendFailure:
                logger.warning(
                    "Could not load session %r from %r", id, self.backend
                )
                self.orig_id = None
                self.id = _generate_id()
                self.id_needs_persisting = True
                self.data = {}
        else:
            self.id = None
            self.data = {}
        self.is_loaded = True

    def save(self):
        if not self.is_loaded:
            self.load()
        if self.id is None:
            self.id = _generate_id()
            self.id_needs_persisting = True
        self.backend.save(self.id, self.data)
        self.dirty = False

    def set_id(self, id):
        self.orig_id = self.id = id
        self.data = self.backend.load(id)
        self.is_loaded = True
        self.id_needs_persisting = True

    def cycle(self):
        if not self.is_loaded:
            self.load()
        id = self.id = _generate_id()
        self.id_needs_persisting = True
        if self.data:
            self.backend.save(id, self.data)
        if self.orig_id is not None:
            self.backend.delete(self.orig_id)

    def delete(self):
        if not self.is_loaded:
            self.load()
        if self.id is not None:
            self.backend.delete(self.id)
        self.id = None
        self.id_needs_persisting = False


class BackendFailure(Exception):
    """
    Backend could not load or persist the session data
    """


class Backend(object):
    def load(self, id):
        """
        Load the session data dict associated with ``id``
        """
        raise NotImplementedError()

    def save(self, id, data):
        """
        Save ``data`` to the backend
        """
        raise NotImplementedError()

    def delete(self, id):
        """
        Delete data associated with ``id``
        """
        raise NotImplementedError()

    def modified_at(self, session_id):
        """
        Return a ``struct_time`` representing the last modified time of the
        session.
        """

    def keys(self):
        """
        Return all session ids managed by the backend
        """
        raise NotImplementedError()

    def keys_olderthan(self, when):
        """
        Return session ids managed by the backend that have not
        been accessed since ``when``
        """
        if isinstance(when, (date, datetime)):
            when = when.timetuple()

        return (k for k in self.keys() if self.modified_at(k) >= when)


class MemoryBackend(Backend):
    def __init__(self, size=1000):
        self.store = OrderedDict()
        self.size = size

    def load(self, id):
        value = self.store.get(id)
        if value:
            del self.store[id]
            self.store[id] = value
            return value[1]
        raise BackendFailure()

    def save(self, id, data, timestamp=time.localtime):
        self.store[id] = (timestamp(), data)
        if len(self.store) > self.size:
            self.store.popitem(last=False)

    def delete(self, id):
        try:
            del self.store[id]
        except KeyError:
            pass

    def modified_at(self, id):
        try:
            return self.store[id][1]
        except KeyError:
            return EPOCH

    def keys(self):
        return self.store.keys()


class FileBackend(Backend):
    def __init__(
        self, directory=None, prefix=None, serializer=PickleSerializer()
    ):
        self.serializer = serializer
        self.directory = directory or tempfile.gettempdir()
        self.prefix = prefix or "s_"

        if not os.path.exists(self.directory):
            try:
                os.makedirs(self.directory)
            except OSError:
                if not os.path.isdir(self.directory):
                    raise

        def getpath(id, d=self.directory, p=self.prefix, join=os.path.join):
            return join(d, p + id)

        self.getpath = getpath

    def __repr__(self):
        return "<%s %r>" % (self.__class__.__name__, self.directory)

    def load(self, id):
        path = self.getpath(id)
        try:
            with open(path, "rb") as f:
                return self.serializer.load(f)
        except (EOFError, IOError) as e:
            raise BackendFailure(e)

    def save(
        self, id, data, rename=os.rename, mktmp=tempfile.NamedTemporaryFile
    ):
        path = self.getpath(id)
        tmpfile = mktmp(mode="wb", delete=False, dir=self.directory)
        try:
            self.serializer.dump(data, tmpfile)
        finally:
            tmpfile.close()
            rename(tmpfile.name, path)

    def delete(self, id, unlink=os.unlink):
        try:
            unlink(self.getpath(id))
        except OSError:
            if os.path.exists(self.getpath(id)):
                raise

    def modified_at(self, id, t=time.localtime):
        try:
            return t(os.stat(self.getpath(id)).mtime)
        except OSError:
            return EPOCH

    def keys(self):
        p = re.compile(
            "^{}{}$".format(re.escape(self.prefix), ID_PATTERN)
        ).match
        prefix_len = len(self.prefix)
        return (s[prefix_len:] for s in os.listdir(self.directory) if p(s))


def SessionMiddleware(
    app, id_persister=CookieIdPersistence(), backend=FileBackend()
):
    def wsgi_app_with_sessions(
        environ,
        start_response,
        app=app,
        set_id_middleware=id_persister.set_id_middleware,
        get_id=id_persister.get_id,
        SESSION_KEY=SESSION_KEY,
        backend=backend,
    ):

        session = environ[SESSION_KEY] = Session(environ, backend, get_id)
        content = None
        try:
            content = id_persister.set_id_middleware(
                session, app, environ, start_response
            )
            for item in content:
                yield item
        finally:
            try:
                if session.dirty and session.data:
                    backend.save(session.id, session.data)
            finally:
                close = getattr(content, "close", None)
                if close is not None:
                    close()

    return wsgi_app_with_sessions
