"""
This module contains the default values for Configuration Data.
These values are recommended in the original Excel CBA

DANGEROUS: Do not change these values if you're not sure how to change them
"""
import numpy as np
import pandas as pd
from intervaltree import IntervalTree

dDiscount_Rate = 0.12
dEconomic_Factor = 0.91

# 1. Annual Traffic Growth
dGrowth = np.array(
    [
        [3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4, 3.4],
        [4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6, 4.6],
        [5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4, 5.4],
        [5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7, 5.7],
        [8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6, 8.6],
    ],
    dtype=np.float64,
)
dGrowth = dGrowth / 100

# 2. Traffic Levels
dTrafficLevels = np.array(
    #   <-------------- vehicle type ------------------------------->
    #              s.car       4wd        s.trk     l.trk s.bus  l.bus  <- condition class + 12 ->
    [  # vol mbike      m.car        delv       m.trk   a.trk m.bus   13   14   15   16   17
        [25, 0.75, 0.02, 0.05, 0.01, 0.01, 0.12, 0.03, 0, 0, 0.01, 0, 0, 1.5, 1.5, 1.5, 1.5, 1.5],
        [75, 0.75, 0.02, 0.05, 0.01, 0.01, 0.12, 0.03, 0, 0, 0.01, 0, 0, 1.5, 1.5, 1.5, 1.5, 1.5],
        [175, 0.65, 0.03, 0.07, 0.02, 0.02, 0.13, 0.04, 0.01, 0, 0.02, 0.01, 0, 1.5, 1.5, 1.5, 1.5, 1.5],
        [375, 0.65, 0.03, 0.07, 0.02, 0.02, 0.13, 0.04, 0.01, 0, 0.02, 0.01, 0, 2.0, 2.0, 2.0, 2.0, 2.0],
        [750, 0.5, 0.05, 0.09, 0.04, 0.04, 0.15, 0.06, 0.02, 0, 0.04, 0.01, 0, 3.0, 3.0, 3.0, 3.0, 3.0],
        [2000, 0.5, 0.05, 0.09, 0.04, 0.04, 0.15, 0.06, 0.02, 0, 0.04, 0.01, 0, 4.0, 4.0, 4.0, 4.0, 4.0],
        [4000, 0.5, 0.05, 0.09, 0.04, 0.04, 0.15, 0.06, 0.02, 0, 0.04, 0.01, 0, 5.0, 5.0, 5.0, 5.0, 5.0],
        [6000, 0.5, 0.05, 0.09, 0.04, 0.04, 0.15, 0.06, 0.02, 0, 0.04, 0.01, 0, 6.0, 6.0, 6.0, 6.0, 6.0],
        [8000, 0.5, 0.05, 0.09, 0.04, 0.04, 0.15, 0.06, 0.02, 0, 0.04, 0.01, 0, 7.0, 7.0, 7.0, 7.0, 7.0],
        [10500, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
        [13500, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
        [17500, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
        [25000, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
        [35000, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
        [500000, 0.25, 0.08, 0.12, 0.07, 0.07, 0.18, 0.08, 0.05, 0.01, 0.06, 0.02, 0.01, 8.0, 8.0, 8.0, 8.0, 8.0],
    ],
    dtype=np.float64,
)

traffic_ranges_data = [
    (0, 50, 1),
    (50, 100, 2),
    (100, 250, 3),
    (250, 500, 4),
    (500, 1000, 5),
    (1000, 3000, 6),
    (3000, 5000, 7),
    (5000, 7000, 8),
    (7000, 9000, 9),
    (9000, 12000, 10),
    (12000, 15000, 11),
    (15000, 20000, 12),
    (20000, 30000, 13),
    (30000, 40000, 14),
    (40000, 100000, 15),
]

traffic_ranges = pd.DataFrame(data=traffic_ranges_data, columns=["lower", "upper", "traffic_class"])

# 3. Vehicle Fleet
dVehicleFleet = np.array(
    [
        [0.00, 1.3, 2.52],
        [0.00, 1.9, 5.60],
        [0.00, 1.9, 5.60],
        [0.01, 0, 0.00],
        [0.02, 0, 0.00],
        [0.10, 0, 0.00],
        [1.25, 0, 0.00],
        [2.50, 0, 0.00],
        [5.00, 0, 0.00],
        [0.04, 15, 2.80],
        [0.70, 22, 2.80],
        [1.00, 30, 2.80],
    ],
    dtype=np.float64,
)

iSurfaceDefaults = np.array([3, 4], dtype=np.uint8)

# 4. Default Width (width, XXXX)
#                           1 lane    2 lane    3 lane    4 lane    5 lane     6 lane     7 lane
dWidthDefaults = np.array([[3.5, 1], [5.0, 2], [7.0, 2], [9.0, 2], [14.0, 4], [21.0, 6], [28.0, 8]], dtype=np.float64)

# columns = ["lower_width", "upper_width", "lanes"],
default_lanes = [
    (0.00, 4.25, 1),
    (4.25, 6.00, 2),
    (6.00, 8.00, 3),
    (8.00, 11.5, 4),
    (11.5, 17.50, 5),
    (17.5, 24.50, 6),
    (24.5, 100.0, 7),
]

# 5. Surface type, condition levels, variables [initialRoughness, pavementAge]
dConditionData = np.array(
    [  #   VGood     Good      Fair        Poor       VPoor
        [[3.0, 1], [4.0, 5], [7.0, 10], [10.0, 15], [14.0, 20]],  # 1: Cement Concrete
        [[2.0, 1], [3.0, 5], [6.0, 10], [12.0, 15], [16.0, 20]],  # 2: Asphalt Concrete
        [[3.0, 1], [4.0, 5], [7.0, 10], [12.0, 15], [16.0, 20]],  # 3: Surface Treatment
        [[5.0, 1], [7.0, 4], [11.0, 6], [16.0, 8], [20.0, 10]],  # 4: Gravel
        [[7.0, 1], [9.0, 4], [13.0, 6], [18.0, 8], [22.0, 10]],  # 5: Earth
        [[3.0, 1], [4.0, 5], [6.0, 10], [10.0, 15], [14.0, 20]],  # 6: Macadam
        [[5.0, 1], [7.0, 5], [11.0, 10], [16.0, 15], [20.0, 20]],  # 7: Cobblestone
    ],
    dtype=np.float64,
)

# 6. Road works, variables
class MaintenanceAlternative(object):
    def __init__(self, i, arr):
        self.ia = i
        (
            self.name,
            self.code,
            self.work_class,
            self.cost_flat,
            self.cost_hilly,
            self.cost_mountainous,
            self.iri,
            self.lanes_class,
            self.width,
            self.surface,
            self.thickness,
            self.strength,
            self.snc,
            self.repair,
            self.repair_period,
        ) = arr
        self.unit_costs = {1: self.cost_flat, 2: self.cost_hilly, 3: self.cost_mountainous}

    def get_unit_cost(self, terrain):
        return self.unit_costs[terrain]


dRoadWorks = [
    [
        "Periodic Maintenance (Concrete)",
        "C-P",
        "Periodic",
        132.0,
        132.0,
        132.0,
        3.0,
        0,
        0,
        0,
        None,
        None,
        None,
        1,
        8,
    ],
    [
        "Reconstruction (Concrete)",
        "C-R",
        "Rehabilitation",
        549.0,
        549.0,
        549.0,
        1.8,
        0,
        0,
        0,
        None,
        None,
        None,
        1,
        8,
    ],
    ["Reseal", "B-P1", "Periodic", 127.0, 127.0, 127.0, 4.0, 0, 0, 0, 25, 0.20, None, 4, 8],
    ["Functional Overlay (<=50mm)", "B-P2", "Periodic", 176.25, 176.25, 176.25, 3.5, 0, 0, 0, 50, 0.40, None, 4, 8],
    ["Structural Overlay (51-100mm)", "B-P3", "Periodic", 282.0, 282.0, 282.0, 3.0, 0, 0, 0, 80, 0.40, None, 4, 8],
    ["Thick Overlay (>100mm)", "B-R1", "Rehabilitation", 352.5, 352.5, 352.5, 2.5, 0, 0, 0, 100, 0.40, None, 4, 8],
    ["Reconstruction Type V", "B-R2", "Rehabilitation", 397.0, 397.0, 397.0, 2.0, 0, 0, 0, None, None, 2.0, 4, 8],
    ["Reconstruction Type IV", "B-R3", "Rehabilitation", 397.0, 397.0, 397.0, 2.0, 0, 0, 0, None, None, 3.0, 4, 8],
    ["Reconstruction Type III", "B-R4", "Rehabilitation", 397.0, 397.0, 397.0, 2.0, 0, 0, 0, None, None, 4.0, 4, 8],
    ["Reconstruction Type II", "B-R5", "Rehabilitation", 397.0, 397.0, 397.0, 2.0, 0, 0, 0, None, None, 5.0, 4, 8],
    ["Reconstruction Type I", "B-R6", "Rehabilitation", 397.0, 397.0, 397.0, 2.0, 0, 0, 0, None, None, 6.0, 4, 8],
    ["Regravelling (Gravel)", "G-P", "Periodic", 37.0, 37.0, 37.0, 12.0, 0, 0, 0, None, None, None, 12, 4],
    [
        "Reconstruction (Gravel)",
        "G-R",
        "Rehabilitation",
        114.0,
        114.0,
        114.0,
        7.0,
        0,
        0,
        0,
        None,
        None,
        None,
        12,
        4,
    ],
    ["Heavy Grading (Earth)", "E-P", "Periodic", 16.0, 16.0, 16.0, 16.0, 0, 0, 0, None, None, None, 14, 4],
    ["Reconstruction (Earth)", "E-R", "Rehabilitation", 24.0, 24.0, 24.0, 10.0, 0, 0, 0, None, None, None, 14, 4],
    ["Periodic Maintenance (Macadam)", "M-P", "Periodic", 37.0, 37.0, 37.0, 8.0, 0, 0, 0, None, None, None, 16, 8],
    [
        "Reconstruction (Macadam)",
        "M-R",
        "Rehabilitation",
        114.0,
        114.0,
        114.0,
        4.0,
        0,
        0,
        0,
        None,
        None,
        None,
        16,
        8,
    ],
    [
        "Periodic Maintenance (Cobblestone)",
        "O-P",
        "Periodic",
        37.0,
        37.0,
        37.0,
        8.0,
        0,
        0,
        0,
        None,
        None,
        None,
        18,
        8,
    ],
    [
        "Reconstruction (Cobblestone)",
        "O-R",
        "Rehabilitation",
        114.0,
        114.0,
        114.0,
        4.0,
        0,
        0,
        0,
        None,
        None,
        None,
        18,
        8,
    ],
    ["Upgrade to Cobblestone", "U-C", "Upgrade", 500.0, 500.0, 500.0, 4.0, 3, 7.0, 7, None, None, None, 18, 8],
    ["Upgrade to Macadam", "U-M", "Upgrade", 500.0, 500.0, 500.0, 4.0, 3, 7.0, 6, None, None, None, 16, 8],
    ["Upgrade to Gravel", "U-G", "Upgrade", 500.0, 500.0, 500.0, 8.0, 3, 7.0, 4, None, None, None, 12, 4],
    ["Upgrade to Surface Treatment", "U-S", "Upgrade", 500.0, 500.0, 500.0, 2.6, 3, 7.0, 3, None, None, 2.0, 4, 8],
    ["Upgrade to Asphalt Concrete", "U-A", "Upgrade", 500.0, 500.0, 500.0, 2.2, 3, 7.0, 2, None, None, 4.0, 4, 8],
    ["Upgrade to Cement Concrete", "U-C", "Upgrade", 500.0, 500.0, 500.0, 1.8, 3, 7.0, 1, None, None, None, 1, 8],
]
alternatives = [MaintenanceAlternative(i, arr) for i, arr in enumerate(dRoadWorks)]

# 7. Recurrent: surface type X lanes
dRecurrent = np.array(
    [
        [
            10167.7938484056,
            19115.4524350025,
            20282.3695008311,
            19928.875942875,
            20335.5876968112,
            20335.5876968112,
            20335.5876968112,
        ],
        [
            14758.9085060652,
            26861.2134810386,
            27746.7479914025,
            28632.2825017664,
            29517.8170121303,
            29517.8170121303,
            29517.8170121303,
        ],
        [
            12117.2647173132,
            18418.242370316,
            20357.0047250861,
            22295.7670798562,
            24234.5294346263,
            24234.5294346263,
            24234.5294346263,
        ],
        [
            8845.72489898381,
            12737.8438545367,
            14330.0743363538,
            16099.2193161505,
            17691.4497979676,
            17691.4497979676,
            17691.4497979676,
        ],
        [
            5662.55318106041,
            7361.31913537853,
            8607.08083521182,
            9966.09359866632,
            11325.1063621208,
            11325.1063621208,
            11325.1063621208,
        ],
        [
            12117.2647173132,
            18418.242370316,
            20357.0047250861,
            22295.7670798562,
            24234.5294346263,
            24234.5294346263,
            24234.5294346263,
        ],
        [
            8845.72489898381,
            12737.8438545367,
            14330.0743363538,
            16099.2193161505,
            17691.4497979676,
            17691.4497979676,
            17691.4497979676,
        ],
    ],
    dtype=np.float64,
)

# 8. Recurrent Maintenance Condition Multipliers (#) (RecMult)
dRecMult = np.array([0.80, 0.90, 1.00, 1.10, 1.20], dtype=np.float64)

# 9. (surface type X road class X condition class), variables
dWorkEvaluated = np.array(
    [
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 0, 0, 1.00],
        [10, 2, 1, 0, 1.00],
        [10, 2, 1, 2, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 2, 2, 0, 1.00],
        [10, 11, 0, 0, 1.00],
        [10, 11, 5, 4, 1.00],
        [10, 11, 6, 5, 1.00],
        [10, 11, 11, 6, 1.00],
        [10, 11, 11, 0, 1.00],
        [10, 10, 0, 0, 1.00],
        [10, 10, 5, 4, 1.00],
        [10, 10, 5, 4, 1.00],
        [10, 10, 10, 6, 1.00],
        [10, 10, 10, 0, 1.00],
        [10, 9, 0, 0, 1.00],
        [10, 9, 5, 4, 1.00],
        [10, 9, 5, 4, 1.00],
        [10, 9, 9, 6, 1.00],
        [10, 9, 9, 0, 1.00],
        [10, 8, 0, 0, 1.00],
        [10, 8, 5, 4, 1.00],
        [10, 8, 5, 4, 1.00],
        [10, 8, 8, 6, 1.00],
        [10, 8, 8, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 11, 0, 0, 1.00],
        [10, 11, 4, 3, 1.00],
        [10, 11, 5, 4, 1.00],
        [10, 11, 11, 6, 1.00],
        [10, 11, 11, 0, 1.00],
        [10, 10, 0, 0, 1.00],
        [10, 10, 4, 3, 1.00],
        [10, 10, 5, 4, 1.00],
        [10, 10, 10, 6, 1.00],
        [10, 10, 10, 0, 1.00],
        [10, 9, 0, 0, 1.00],
        [10, 9, 4, 3, 1.00],
        [10, 9, 5, 4, 1.00],
        [10, 9, 9, 6, 1.00],
        [10, 9, 9, 0, 1.00],
        [10, 8, 0, 0, 1.00],
        [10, 8, 4, 3, 1.00],
        [10, 8, 5, 4, 1.00],
        [10, 8, 8, 6, 1.00],
        [10, 8, 8, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 7, 0, 0, 1.00],
        [10, 7, 4, 3, 1.00],
        [10, 7, 5, 4, 1.00],
        [10, 7, 7, 6, 1.00],
        [10, 7, 7, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 0, 0, 1.00],
        [10, 13, 12, 0, 1.00],
        [10, 13, 12, 13, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 13, 13, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 0, 0, 1.00],
        [10, 15, 14, 0, 1.00],
        [10, 15, 14, 15, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 15, 15, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 0, 0, 1.00],
        [10, 17, 16, 0, 1.00],
        [10, 17, 16, 17, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 17, 17, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 0, 0, 1.00],
        [10, 19, 18, 0, 1.00],
        [10, 19, 18, 19, 1.00],
        [10, 19, 19, 0, 1.00],
        [10, 19, 19, 0, 1.00],
    ],
    dtype=np.float64,
)

# 10. dm_coeff
dm_coeff = np.array(
    [
        [0.005, 0.010, 0.020, 0.025, 0.030],
        [0.100, 0.015, 0.025, 0.030, 0.040],
        [0.015, 0.025, 0.040, 0.060, 0.070],
        [0.025, 0.035, 0.060, 0.100, 0.120],
        [0.040, 0.060, 0.100, 0.200, 0.220],
    ],
    dtype=np.float64,
)

# 11. VOC
dVOC = np.array(
    [
        # 1
        [
            [
                [
                    1.47873191833496,
                    5.4439235051473,
                    7.50857830047608,
                    6.66936928431193,
                    5.84006366729736,
                    7.80676358540853,
                    9.32577244440715,
                    16.8067712783813,
                    24.1951700846354,
                    8.39294929504395,
                    10.1914302825928,
                    16.8410250345866,
                ],
                [
                    0.00186451401188108,
                    -0.0524617971703352,
                    -0.0652549546544835,
                    -0.0596933556603387,
                    -0.0150653453603122,
                    -0.00868533837674842,
                    0.123112712521468,
                    0.208345351889788,
                    -0.030461432865265,
                    0.033463025333905,
                    0.135965435284465,
                    0.000693846230554195,
                ],
                [
                    -0.00202818685547734,
                    0.0127078595556023,
                    0.0217959228512291,
                    0.028995215962261,
                    0.0223197599391004,
                    0.0216308019377968,
                    0.0113284526727138,
                    0.0304804779154993,
                    0.0656975354903784,
                    0.0250878135363261,
                    0.02258829373024,
                    0.0686603676189077,
                ],
                [
                    0.00011612885655501,
                    -0.0001709718132723,
                    -0.000388011220544462,
                    -0.000620958167356212,
                    -0.000469728459104051,
                    -0.000337418582704332,
                    -2.97325983840357e-05,
                    -0.000228632117021704,
                    -0.000796745058725118,
                    -0.000531680355168354,
                    -0.000422217467077591,
                    -0.00130972476920697,
                ],
            ],
            [
                [
                    1.19556461970011,
                    5.20847617785136,
                    7.39859746297201,
                    6.39333367347717,
                    5.43949100176493,
                    7.34984912872315,
                    9.26818577448527,
                    16.9815795898438,
                    23.6560812632243,
                    8.01432011922201,
                    9.73803803126017,
                    18.0261770884196,
                ],
                [
                    0.026315750590386,
                    -0.0126324053318257,
                    -0.0346204804744291,
                    0.000961503365656356,
                    0.0482657333168771,
                    0.0657583648322983,
                    0.13905459767455,
                    0.20258522774697,
                    0.0955914176520021,
                    0.0898847027906806,
                    0.207475709563183,
                    -0.102736611281056,
                ],
                [
                    -0.00158418956405912,
                    0.0109632666850146,
                    0.0197584725621141,
                    0.0255713095337083,
                    0.0198226641941737,
                    0.0188764201733339,
                    0.011065199102833,
                    0.0311757670693743,
                    0.0608306636899224,
                    0.0236365814586899,
                    0.0233015565327553,
                    0.0798978094176535,
                ],
                [
                    6.65929622907648e-05,
                    -0.000153963440318578,
                    -0.00034808891404944,
                    -0.000564141359640447,
                    -0.000452565899076809,
                    -0.000326751852183726,
                    -4.09014991946976e-05,
                    -0.000246301721767021,
                    -0.000765906717764286,
                    -0.000549214934784034,
                    -0.000551117142689905,
                    -0.00166557153807005,
                ],
            ],
            [
                [
                    1.05344875653585,
                    5.3635888894399,
                    7.70215841929118,
                    6.8025993347168,
                    5.57382804552714,
                    7.59177926381429,
                    9.68773371378581,
                    17.7301233927409,
                    24.8483093261719,
                    8.35374212265014,
                    11.3869802474976,
                    21.3128425598144,
                ],
                [
                    0.0132911097845924,
                    0.0151455516767019,
                    0.0151321789786718,
                    0.0466753788112941,
                    0.0842284451831468,
                    0.106500125431514,
                    0.131542458122982,
                    0.303176733534192,
                    0.265036876755531,
                    0.0870973749872134,
                    0.203321960607794,
                    0.139162431878461,
                ],
                [
                    0.0016874624264268,
                    0.00738871639425105,
                    0.0125608515072536,
                    0.0190056139355773,
                    0.0162676003846256,
                    0.0150672166775435,
                    0.0139894409057422,
                    0.0191627726966127,
                    0.0416828682928375,
                    0.0247401115777609,
                    0.023626542730487,
                    0.0427251962515022,
                ],
                [
                    -4.77517754557342e-05,
                    -5.96038921914922e-05,
                    -0.000143265506511709,
                    -0.000386982681497458,
                    -0.000372288805065734,
                    -0.000251142572967601,
                    -0.000180963383410084,
                    7.72708686822733e-05,
                    -0.000274653988655169,
                    -0.000614168830129094,
                    -0.00060375015126149,
                    -0.000530821830613161,
                ],
            ],
        ],
        # 2
        [
            [
                [
                    1.48654867808024,
                    5.45061941146851,
                    7.59041617711385,
                    6.6783325513204,
                    5.85077548027039,
                    7.89757486979167,
                    10.0190785725911,
                    17.6105839411418,
                    26.82186571757,
                    8.56465164820353,
                    10.2113579114278,
                    16.9151468276978,
                ],
                [
                    0.000863620474436781,
                    -0.0534385863254492,
                    -0.079976241442244,
                    -0.0610500991853895,
                    -0.0165877102786661,
                    -0.0231943845563649,
                    0.0347726076447698,
                    0.129365346555851,
                    -0.537399295697217,
                    0.00725542646836369,
                    0.133147109545935,
                    -0.0410889784495049,
                ],
                [
                    -0.00200160544165921,
                    0.0127458220873124,
                    0.022598172192807,
                    0.0290523589629949,
                    0.0223754223410066,
                    0.0222974045849069,
                    0.0136331876952609,
                    0.0300508673652346,
                    0.0966513707087591,
                    0.0262095990714493,
                    0.0226904396966343,
                    0.0725816741134182,
                ],
                [
                    0.000116366522550213,
                    -0.000171235854438598,
                    -0.000401329894629022,
                    -0.000621507819901166,
                    -0.000469996112673063,
                    -0.000345564157164366,
                    -7.62191500511772e-06,
                    -0.000104483793017098,
                    -0.00140178972376162,
                    -0.000543194448846018,
                    -0.000422680479848488,
                    -0.00141059810465033,
                ],
            ],
            [
                [
                    1.19661999146144,
                    5.20861760775248,
                    7.39828958511353,
                    6.39336121877035,
                    5.44034949938456,
                    7.35072345733643,
                    9.2704296429952,
                    17.5415482203166,
                    23.6610069274902,
                    8.01555131276449,
                    9.73916047414143,
                    17.6779888153076,
                ],
                [
                    0.0261639299301001,
                    -0.0126560916711679,
                    -0.0345806447498171,
                    0.000949758101361464,
                    0.0481343037320737,
                    0.065607883713462,
                    0.138710866626511,
                    0.155198290075733,
                    0.0948243619122974,
                    0.0896880107974482,
                    0.207287697062164,
                    -0.113427301108979,
                ],
                [
                    -0.00157848712586579,
                    0.0109644562075466,
                    0.0197573657774981,
                    0.025572460431319,
                    0.0198283188548677,
                    0.0188843300848296,
                    0.011080008981389,
                    0.0298760932086509,
                    0.0608646939128708,
                    0.0236457594624767,
                    0.0233110407571414,
                    0.0852330314529528,
                ],
                [
                    6.65602306802791e-05,
                    -0.00015398102762538,
                    -0.000348096370419145,
                    -0.000564173361823461,
                    -0.000452624091179499,
                    -0.000326878193652992,
                    -4.10545863379038e-05,
                    -0.000128448287368099,
                    -0.000766289400887659,
                    -0.000549331921056528,
                    -0.000551258651619282,
                    -0.00186524719069333,
                ],
            ],
            [
                [
                    1.05344887574514,
                    5.36358860333761,
                    7.70215771993001,
                    6.80259949366252,
                    5.57382709185283,
                    7.5917771021525,
                    9.68773066202799,
                    17.7301130930583,
                    24.8482981363932,
                    8.35373922983805,
                    11.3869764328003,
                    21.3128335952759,
                ],
                [
                    0.0132910460581035,
                    0.0151454502515487,
                    0.0151321327584419,
                    0.0466750912084629,
                    0.0842284451831468,
                    0.106500243621921,
                    0.131542100402488,
                    0.303177375245113,
                    0.265035272107721,
                    0.0870975468079064,
                    0.203321960607794,
                    0.13916238797387,
                ],
                [
                    0.00168746859182567,
                    0.00738872973235339,
                    0.0125608538692092,
                    0.0190056402644356,
                    0.0162676003846256,
                    0.0150672041730725,
                    0.0139894706385953,
                    0.0191627337938144,
                    0.0416830122331917,
                    0.0247401004626756,
                    0.023626542730487,
                    0.0427252084780963,
                ],
                [
                    -4.77519520234952e-05,
                    -5.96043784764731e-05,
                    -0.000143265413885999,
                    -0.000386983410924929,
                    -0.000372288805065734,
                    -0.000251142248777612,
                    -0.000180963846538635,
                    7.72716096879541e-05,
                    -0.000274657693683593,
                    -0.00061416857540839,
                    -0.00060375015126149,
                    -0.000530822803183132,
                ],
            ],
        ],
        # 3
        [
            [
                [
                    1.49910345872243,
                    5.46331548690796,
                    7.60276425679525,
                    6.69711820284526,
                    5.87212382952373,
                    7.9204594930013,
                    10.3851486523946,
                    19.1174034118652,
                    28.3614471435547,
                    8.58786948521932,
                    10.2356811523437,
                    17.389585240682,
                ],
                [
                    0.00135577021408384,
                    -0.0523986807143679,
                    -0.0787283601738635,
                    -0.0600451664954267,
                    -0.0159572733865759,
                    -0.021703539843141,
                    -0.0141464116134288,
                    -0.0799556156416315,
                    -0.863262532566906,
                    0.00810710709504424,
                    0.134720019805127,
                    -0.116303131945298,
                ],
                [
                    -0.00218138508585506,
                    0.0123610214475707,
                    0.0221256750585872,
                    0.0285414651140467,
                    0.0219457977579469,
                    0.02168791013442,
                    0.0151048153986186,
                    0.0370645600876886,
                    0.118607409350522,
                    0.0257436253807761,
                    0.0221419624793223,
                    0.0754722709700222,
                ],
                [
                    0.000122160993070684,
                    -0.000159588611301935,
                    -0.000387159087163308,
                    -0.000605747786805718,
                    -0.000456398079445314,
                    -0.000327636844535716,
                    -9.69756455410346e-06,
                    -0.000125225051577552,
                    -0.00189446254395291,
                    -0.000529298577645812,
                    -0.000407277912526696,
                    -0.00143849923801496,
                ],
            ],
            [
                [
                    1.20122200250626,
                    5.20607067743937,
                    7.3922856648763,
                    6.38989181518555,
                    5.44153858820597,
                    7.34853401184082,
                    9.27571207682292,
                    17.5664829889933,
                    23.7134711583455,
                    8.0163628578186,
                    9.73668553034464,
                    17.6533327738444,
                ],
                [
                    0.0273444117759677,
                    -0.0100330489498752,
                    -0.0313322261589482,
                    0.00471742910846015,
                    0.0512477878275636,
                    0.0697203100237067,
                    0.144374558049389,
                    0.165260906819696,
                    0.0843032377120032,
                    0.0928958841507617,
                    0.210686541103818,
                    -0.106783440433553,
                ],
                [
                    -0.00174999330844079,
                    0.0105688805346722,
                    0.0192617227286448,
                    0.0249903479239324,
                    0.0193354132431093,
                    0.0182463505884985,
                    0.0102023425357881,
                    0.0279951326219075,
                    0.0615592839278819,
                    0.0231501871333534,
                    0.0228215136450209,
                    0.0841040919710707,
                ],
                [
                    7.11788149100456e-05,
                    -0.000143919652436441,
                    -0.000335563393924805,
                    -0.000549097450648341,
                    -0.000439633520518335,
                    -0.000311042068536281,
                    -0.000018318540473827,
                    -7.38511274466726e-05,
                    -0.000798657594397723,
                    -0.000537123111392885,
                    -0.000540150142771935,
                    -0.00183617795449222,
                ],
            ],
            [
                [
                    1.05411531130473,
                    5.35330990155538,
                    7.68681446711222,
                    6.78851610819499,
                    5.56574603716533,
                    7.57657998402913,
                    9.67108605702718,
                    17.7063230514526,
                    24.8431182225545,
                    8.34142713546753,
                    11.3739653587341,
                    21.2698767344157,
                ],
                [
                    0.0137056572639211,
                    0.0167801677022873,
                    0.0173275219717969,
                    0.0487797495553724,
                    0.0865577036498957,
                    0.109466724277191,
                    0.135068896396639,
                    0.312439892397617,
                    0.254707301413263,
                    0.0893404342077828,
                    0.205523589552144,
                    0.146663176707731,
                ],
                [
                    0.00164514800885341,
                    0.00720290176240436,
                    0.0123087710989661,
                    0.0187642281149929,
                    0.0159928766024974,
                    0.0147225517770905,
                    0.0135678508620719,
                    0.0180095509215668,
                    0.0430536531346105,
                    0.0244756388219642,
                    0.0233780840337971,
                    0.0417793102753469,
                ],
                [
                    -0.000047216477001371,
                    -5.66895631458191e-05,
                    -0.000139230105334135,
                    -0.00038353488624976,
                    -0.000367236049395712,
                    -0.000245647228495873,
                    -0.000173046362890974,
                    0.000102914991356556,
                    -0.000334902104242143,
                    -0.000609591568544473,
                    -0.000600342358742559,
                    -0.000511064396037926,
                ],
            ],
        ],
        # 4
        [
            [
                [
                    1.49915534655253,
                    5.46337550481161,
                    7.60282573699951,
                    6.69721199671428,
                    5.87222623825073,
                    7.92055657704671,
                    10.3853942553202,
                    20.2893243789673,
                    29.1339371363322,
                    8.58796809514364,
                    10.2357676823934,
                    18.0058885574341,
                ],
                [
                    0.00136569172444978,
                    -0.0523859402331254,
                    -0.078714877760679,
                    -0.0600308715231959,
                    -0.0159432947311462,
                    -0.0216795959116975,
                    -0.0141235436222402,
                    -0.391658404490331,
                    -1.29310488812018,
                    0.00812621540863861,
                    0.134748170611093,
                    -0.35715776371604,
                ],
                [
                    -0.00218300950346572,
                    0.0123580598330998,
                    0.0221221744319498,
                    0.0285375348893635,
                    0.0219422416948216,
                    0.0216828679149245,
                    0.0150984406193375,
                    0.0662618878282311,
                    0.17860313895699,
                    0.0257396146411941,
                    0.0221371692377371,
                    0.103735124592459,
                ],
                [
                    0.000122208298757823,
                    -0.000159502423078216,
                    -0.000387057372554848,
                    -0.000605631425756769,
                    -0.000456291907224529,
                    -0.000327495845047745,
                    -9.50592195885866e-06,
                    -0.00102777428282446,
                    -0.00423281134425344,
                    -0.000529186755256684,
                    -0.000407152011029556,
                    -0.00242040267542711,
                ],
            ],
            [
                [
                    1.20125372807185,
                    5.20605640411377,
                    7.39224840799967,
                    6.3898709932963,
                    5.44155220985412,
                    7.34852803548177,
                    9.27575976053874,
                    17.8708564758301,
                    24.4904892603556,
                    8.016375096639,
                    9.73667971293131,
                    18.0202028910319,
                ],
                [
                    0.0273524043790667,
                    -0.010015428575695,
                    -0.0313104470756134,
                    0.00474307531521322,
                    0.0512686658942993,
                    0.0697482285243928,
                    0.144413503455671,
                    0.0035791389684686,
                    -0.348478477132719,
                    0.0929176044167708,
                    0.210710011995756,
                    -0.301899681002389,
                ],
                [
                    -0.00175115231659029,
                    0.0105662229177835,
                    0.0192584010548803,
                    0.0249864063062868,
                    0.0193321026149607,
                    0.0182420225132317,
                    0.0101963415012493,
                    0.0492536727087202,
                    0.12198766183742,
                    0.0231468520520173,
                    0.0228181549441287,
                    0.109761586833945,
                ],
                [
                    7.12099921453251e-05,
                    -0.000143852035667698,
                    -0.000335479417139927,
                    -0.000548995180285616,
                    -0.000439546348146439,
                    -0.000310934367991271,
                    -1.81627440285849e-05,
                    -0.000848997815598229,
                    -0.00315437093666569,
                    -0.000537041068169714,
                    -0.000540073240275706,
                    -0.00277169031418843,
                ],
            ],
            [
                [
                    1.05412019491196,
                    5.35324350992839,
                    7.68671417236328,
                    6.78842433293661,
                    5.5656969388326,
                    7.57648550669352,
                    9.67098890940348,
                    18.0107006708781,
                    25.6146814982096,
                    8.34135405222575,
                    11.3738956133525,
                    21.6345695495605,
                ],
                [
                    0.0137084876393382,
                    0.01679089334276,
                    0.0173421287684822,
                    0.0487940028923465,
                    0.086573282877604,
                    0.109486729207665,
                    0.135092630252972,
                    0.150578424184008,
                    -0.175378580100794,
                    0.08935492651909,
                    0.205537521200024,
                    -0.0473494033250312,
                ],
                [
                    0.00164486007604288,
                    0.00720169056545603,
                    0.0123070999458953,
                    0.0187626079523758,
                    0.0159910528948813,
                    0.0147202538721489,
                    0.0135650009542078,
                    0.0392958973393297,
                    0.103132556924175,
                    0.0244739197350882,
                    0.0233765143058795,
                    0.0672936197761055,
                ],
                [
                    -4.72128009184814e-05,
                    -5.66712232551187e-05,
                    -0.000139203683850185,
                    -0.00038351200769924,
                    -0.000367202993595239,
                    -0.000245611173938019,
                    -0.00017299213053741,
                    -0.000673260953691274,
                    -0.00267790451286927,
                    -0.000609562298819922,
                    -0.000600322212650505,
                    -0.00144137318225081,
                ],
            ],
        ],
        # 5
        [
            [
                [
                    1.51363519430161,
                    5.59740436871847,
                    7.79247236251831,
                    6.88707326253255,
                    6.00836242039998,
                    8.12783273061117,
                    10.5979465484619,
                    21.209879175822,
                    28.1808510462443,
                    8.74143927892049,
                    10.3998775482178,
                    18.287840906779,
                ],
                [
                    -0.00674155941515251,
                    -0.128511020955691,
                    -0.186490085482505,
                    -0.168519174395001,
                    -0.0944845222556136,
                    -0.14273941155636,
                    -0.137699304112373,
                    -0.608274771802262,
                    -0.841918586480499,
                    -0.0808852497884811,
                    0.0387114176198129,
                    -0.460928587924631,
                ],
                [
                    -0.00104342288140095,
                    0.0232463833335396,
                    0.0375477264533232,
                    0.0441779962508551,
                    0.0334085430993345,
                    0.0396326924815323,
                    0.0333193929601105,
                    0.0907694868274503,
                    0.132746515852032,
                    0.0388240112569226,
                    0.0364153510207063,
                    0.124369340620953,
                ],
                [
                    7.74361418955256e-05,
                    -0.000595140904072375,
                    -0.00100464367458963,
                    -0.00123706582728151,
                    -0.000925837337924421,
                    -0.00107506369099473,
                    -0.000763939625548131,
                    -0.00215558678768547,
                    -0.00335862720855559,
                    -0.00106913328356028,
                    -0.00100370340784484,
                    -0.00365241229673089,
                ],
            ],
            [
                [
                    1.21560388008753,
                    5.33825249671936,
                    7.57932939529419,
                    6.5769214630127,
                    5.57582122484843,
                    7.54720115661621,
                    9.48317403793335,
                    17.8628658930461,
                    23.5035137812297,
                    8.16305726369222,
                    9.89061266581218,
                    17.9882055918376,
                ],
                [
                    0.0193106178382431,
                    -0.0852156256462311,
                    -0.137793533544534,
                    -0.102376975935379,
                    -0.0263220482587633,
                    -0.0470982011937326,
                    0.0232587119082464,
                    -0.0365433787410367,
                    0.119940706436818,
                    0.00726586973935753,
                    0.119432058830823,
                    -0.342632219230102,
                ],
                [
                    -0.000619375570253898,
                    0.021344579599954,
                    0.0345308908493647,
                    0.0404738254480429,
                    0.0306836171444758,
                    0.035717764239767,
                    0.0281641650033165,
                    0.0632918803564042,
                    0.0740906464192142,
                    0.0358455224192782,
                    0.0366096277059099,
                    0.126537978510201,
                ],
                [
                    2.66786476578377e-05,
                    -0.000576135474485118,
                    -0.000948428181408911,
                    -0.0011762468825965,
                    -0.000905490676469367,
                    -0.00104551354210416,
                    -0.000766714344491494,
                    -0.0017864977841055,
                    -0.00222080593990928,
                    -0.00106596513691112,
                    -0.00112611601865838,
                    -0.00393886068762044,
                ],
            ],
            [
                [
                    1.06739687919617,
                    5.4703821182251,
                    7.85183795293172,
                    6.95179503758748,
                    5.68553797403972,
                    7.73024482727051,
                    9.8498187383016,
                    17.9933561960856,
                    24.6875475565593,
                    8.44793542226156,
                    11.4745155652364,
                    21.5822226842244,
                ],
                [
                    0.00619016079450632,
                    -0.0510408322572153,
                    -0.078350671150931,
                    -0.0464124682344203,
                    0.0161248197273954,
                    0.0142610199060562,
                    0.0274386241071053,
                    0.116566328642992,
                    0.263202300438513,
                    0.0230041076345962,
                    0.140854070329258,
                    -0.0749533754406568,
                ],
                [
                    0.00271810537382162,
                    0.017153166599207,
                    0.0263588549373867,
                    0.0328428914774826,
                    0.0265238505282325,
                    0.0298460762261789,
                    0.0301232954838893,
                    0.0523564548759195,
                    0.0586922524414419,
                    0.0350674424138103,
                    0.0340929984490633,
                    0.0819850112452653,
                ],
                [
                    -0.000090155877873429,
                    -0.00046647119494307,
                    -0.000718384170680428,
                    -0.000968431467776533,
                    -0.000809906952216262,
                    -0.000920679764684635,
                    -0.000889360256313631,
                    -0.00156970420546928,
                    -0.00184389566763854,
                    -0.00108435182675152,
                    -0.00109876774500393,
                    -0.00252096761356701,
                ],
            ],
        ],
        # 6
        [
            [
                [
                    1.51363519430161,
                    5.59740436871847,
                    7.79247236251831,
                    6.88707326253255,
                    6.00836242039998,
                    8.12783273061117,
                    10.5979465484619,
                    21.209879175822,
                    28.1808510462443,
                    8.74143927892049,
                    10.3998775482178,
                    18.287840906779,
                ],
                [
                    -0.00674155941515251,
                    -0.128511020955691,
                    -0.186490085482505,
                    -0.168519174395001,
                    -0.0944845222556136,
                    -0.14273941155636,
                    -0.137699304112373,
                    -0.608274771802262,
                    -0.841918586480499,
                    -0.0808852497884811,
                    0.0387114176198129,
                    -0.460928587924631,
                ],
                [
                    -0.00104342288140095,
                    0.0232463833335396,
                    0.0375477264533232,
                    0.0441779962508551,
                    0.0334085430993345,
                    0.0396326924815323,
                    0.0333193929601105,
                    0.0907694868274503,
                    0.132746515852032,
                    0.0388240112569226,
                    0.0364153510207063,
                    0.124369340620953,
                ],
                [
                    7.74361418955256e-05,
                    -0.000595140904072375,
                    -0.00100464367458963,
                    -0.00123706582728151,
                    -0.000925837337924421,
                    -0.00107506369099473,
                    -0.000763939625548131,
                    -0.00215558678768547,
                    -0.00335862720855559,
                    -0.00106913328356028,
                    -0.00100370340784484,
                    -0.00365241229673089,
                ],
            ],
            [
                [
                    1.21560388008753,
                    5.33825249671936,
                    7.57932939529419,
                    6.5769214630127,
                    5.57582122484843,
                    7.54720115661621,
                    9.48317403793335,
                    17.8628658930461,
                    23.5035137812297,
                    8.16305726369222,
                    9.89061266581218,
                    17.9882055918376,
                ],
                [
                    0.0193106178382431,
                    -0.0852156256462311,
                    -0.137793533544534,
                    -0.102376975935379,
                    -0.0263220482587633,
                    -0.0470982011937326,
                    0.0232587119082464,
                    -0.0365433787410367,
                    0.119940706436818,
                    0.00726586973935753,
                    0.119432058830823,
                    -0.342632219230102,
                ],
                [
                    -0.000619375570253898,
                    0.021344579599954,
                    0.0345308908493647,
                    0.0404738254480429,
                    0.0306836171444758,
                    0.035717764239767,
                    0.0281641650033165,
                    0.0632918803564042,
                    0.0740906464192142,
                    0.0358455224192782,
                    0.0366096277059099,
                    0.126537978510201,
                ],
                [
                    2.66786476578377e-05,
                    -0.000576135474485118,
                    -0.000948428181408911,
                    -0.0011762468825965,
                    -0.000905490676469367,
                    -0.00104551354210416,
                    -0.000766714344491494,
                    -0.0017864977841055,
                    -0.00222080593990928,
                    -0.00106596513691112,
                    -0.00112611601865838,
                    -0.00393886068762044,
                ],
            ],
            [
                [
                    1.06739687919617,
                    5.4703821182251,
                    7.85183795293172,
                    6.95179503758748,
                    5.68553797403972,
                    7.73024482727051,
                    9.8498187383016,
                    17.9933561960856,
                    24.6875475565593,
                    8.44793542226156,
                    11.4745155652364,
                    21.5822226842244,
                ],
                [
                    0.00619016079450632,
                    -0.0510408322572153,
                    -0.078350671150931,
                    -0.0464124682344203,
                    0.0161248197273954,
                    0.0142610199060562,
                    0.0274386241071053,
                    0.116566328642992,
                    0.263202300438513,
                    0.0230041076345962,
                    0.140854070329258,
                    -0.0749533754406568,
                ],
                [
                    0.00271810537382162,
                    0.017153166599207,
                    0.0263588549373867,
                    0.0328428914774826,
                    0.0265238505282325,
                    0.0298460762261789,
                    0.0301232954838893,
                    0.0523564548759195,
                    0.0586922524414419,
                    0.0350674424138103,
                    0.0340929984490633,
                    0.0819850112452653,
                ],
                [
                    -0.000090155877873429,
                    -0.00046647119494307,
                    -0.000718384170680428,
                    -0.000968431467776533,
                    -0.000809906952216262,
                    -0.000920679764684635,
                    -0.000889360256313631,
                    -0.00156970420546928,
                    -0.00184389566763854,
                    -0.00108435182675152,
                    -0.00109876774500393,
                    -0.00252096761356701,
                ],
            ],
        ],
        # 7
        [
            [
                [
                    1.51363519430161,
                    5.59740436871847,
                    7.79247236251831,
                    6.88707326253255,
                    6.00836242039998,
                    8.12783273061117,
                    10.5979465484619,
                    21.209879175822,
                    28.1808510462443,
                    8.74143927892049,
                    10.3998775482178,
                    18.287840906779,
                ],
                [
                    -0.00674155941515251,
                    -0.128511020955691,
                    -0.186490085482505,
                    -0.168519174395001,
                    -0.0944845222556136,
                    -0.14273941155636,
                    -0.137699304112373,
                    -0.608274771802262,
                    -0.841918586480499,
                    -0.0808852497884811,
                    0.0387114176198129,
                    -0.460928587924631,
                ],
                [
                    -0.00104342288140095,
                    0.0232463833335396,
                    0.0375477264533232,
                    0.0441779962508551,
                    0.0334085430993345,
                    0.0396326924815323,
                    0.0333193929601105,
                    0.0907694868274503,
                    0.132746515852032,
                    0.0388240112569226,
                    0.0364153510207063,
                    0.124369340620953,
                ],
                [
                    7.74361418955256e-05,
                    -0.000595140904072375,
                    -0.00100464367458963,
                    -0.00123706582728151,
                    -0.000925837337924421,
                    -0.00107506369099473,
                    -0.000763939625548131,
                    -0.00215558678768547,
                    -0.00335862720855559,
                    -0.00106913328356028,
                    -0.00100370340784484,
                    -0.00365241229673089,
                ],
            ],
            [
                [
                    1.21560388008753,
                    5.33825249671936,
                    7.57932939529419,
                    6.5769214630127,
                    5.57582122484843,
                    7.54720115661621,
                    9.48317403793335,
                    17.8628658930461,
                    23.5035137812297,
                    8.16305726369222,
                    9.89061266581218,
                    17.9882055918376,
                ],
                [
                    0.0193106178382431,
                    -0.0852156256462311,
                    -0.137793533544534,
                    -0.102376975935379,
                    -0.0263220482587633,
                    -0.0470982011937326,
                    0.0232587119082464,
                    -0.0365433787410367,
                    0.119940706436818,
                    0.00726586973935753,
                    0.119432058830823,
                    -0.342632219230102,
                ],
                [
                    -0.000619375570253898,
                    0.021344579599954,
                    0.0345308908493647,
                    0.0404738254480429,
                    0.0306836171444758,
                    0.035717764239767,
                    0.0281641650033165,
                    0.0632918803564042,
                    0.0740906464192142,
                    0.0358455224192782,
                    0.0366096277059099,
                    0.126537978510201,
                ],
                [
                    2.66786476578377e-05,
                    -0.000576135474485118,
                    -0.000948428181408911,
                    -0.0011762468825965,
                    -0.000905490676469367,
                    -0.00104551354210416,
                    -0.000766714344491494,
                    -0.0017864977841055,
                    -0.00222080593990928,
                    -0.00106596513691112,
                    -0.00112611601865838,
                    -0.00393886068762044,
                ],
            ],
            [
                [
                    1.06739687919617,
                    5.4703821182251,
                    7.85183795293172,
                    6.95179503758748,
                    5.68553797403972,
                    7.73024482727051,
                    9.8498187383016,
                    17.9933561960856,
                    24.6875475565593,
                    8.44793542226156,
                    11.4745155652364,
                    21.5822226842244,
                ],
                [
                    0.00619016079450632,
                    -0.0510408322572153,
                    -0.078350671150931,
                    -0.0464124682344203,
                    0.0161248197273954,
                    0.0142610199060562,
                    0.0274386241071053,
                    0.116566328642992,
                    0.263202300438513,
                    0.0230041076345962,
                    0.140854070329258,
                    -0.0749533754406568,
                ],
                [
                    0.00271810537382162,
                    0.017153166599207,
                    0.0263588549373867,
                    0.0328428914774826,
                    0.0265238505282325,
                    0.0298460762261789,
                    0.0301232954838893,
                    0.0523564548759195,
                    0.0586922524414419,
                    0.0350674424138103,
                    0.0340929984490633,
                    0.0819850112452653,
                ],
                [
                    -0.000090155877873429,
                    -0.00046647119494307,
                    -0.000718384170680428,
                    -0.000968431467776533,
                    -0.000809906952216262,
                    -0.000920679764684635,
                    -0.000889360256313631,
                    -0.00156970420546928,
                    -0.00184389566763854,
                    -0.00108435182675152,
                    -0.00109876774500393,
                    -0.00252096761356701,
                ],
            ],
        ],
    ],
    dtype=np.float64,
)

# 12. Speed
dSPEED = np.array(
    [
        # 1
        [
            [
                [
                    84.7220124562581,
                    91.2204128265381,
                    87.6185859680175,
                    93.1015405019124,
                    91.6138749440511,
                    84.3861798604329,
                    71.2916229248046,
                    58.3106264750162,
                    76.5592967351277,
                    82.1702290852864,
                    82.0395257314046,
                    58.7144254048665,
                ],
                [
                    2.79539402196273,
                    2.08322861255744,
                    2.57038104784241,
                    1.87232003586184,
                    1.84426259661055,
                    1.8261000267183,
                    3.49801236784728,
                    4.32446300436938,
                    1.78787253545809,
                    1.99421377078268,
                    1.75114165152705,
                    4.3086143043862,
                ],
                [
                    -0.724917920319351,
                    -0.715493747404406,
                    -0.732255376580158,
                    -0.712265061331796,
                    -0.700082900640848,
                    -0.614177696632617,
                    -0.662353357790791,
                    -0.662866676881876,
                    -0.636474387629051,
                    -0.608995158077678,
                    -0.57846957820279,
                    -0.665759243887344,
                ],
                [
                    0.0227123831442927,
                    0.0232415771706677,
                    0.0232985433720765,
                    0.0233817576621055,
                    0.0229680190831076,
                    0.0195580227206452,
                    0.0193722246224879,
                    0.0185476593389563,
                    0.0210662697208141,
                    0.0191430126685178,
                    0.0182301675800955,
                    0.0186843935055796,
                ],
            ],
            [
                [
                    66.3080510457357,
                    67.1266235351562,
                    67.4254456837971,
                    67.2535373687744,
                    67.1989744822184,
                    64.9907185872395,
                    65.9936719258626,
                    57.6497238159179,
                    65.2838109970092,
                    64.6301639556884,
                    60.8276870727539,
                    57.2139395395914,
                ],
                [
                    3.90469887058404,
                    3.93570060033437,
                    3.95194316706922,
                    3.94638058040712,
                    3.87864639671997,
                    3.02828333224081,
                    3.59452670845027,
                    4.29500728035122,
                    3.18584211889682,
                    2.97061443106557,
                    2.57614502021298,
                    4.13813079524245,
                ],
                [
                    -0.643353088196619,
                    -0.656505504688184,
                    -0.661904283892582,
                    -0.659255032217031,
                    -0.659054009230821,
                    -0.539806556590511,
                    -0.613146147250019,
                    -0.65128240551982,
                    -0.670799788339434,
                    -0.529295048791491,
                    -0.445500812886201,
                    -0.626945311095054,
                ],
                [
                    0.0180457767443594,
                    0.0185310781326235,
                    0.0187259985665157,
                    0.018627382751204,
                    0.0187434889868238,
                    0.0151494014327992,
                    0.0172940996513871,
                    0.0181175733409191,
                    0.0206414445852622,
                    0.0148087159181253,
                    0.0120448840155376,
                    0.0173379056844586,
                ],
            ],
            [
                [
                    48.5813054402669,
                    48.6066556294759,
                    48.6273543039958,
                    48.6000146230062,
                    48.4340633392334,
                    47.6174352010091,
                    48.0646458943685,
                    46.7616485595703,
                    44.0339042027791,
                    47.4886646270752,
                    45.6920996348063,
                    46.3646500905354,
                ],
                [
                    1.8739499741964,
                    1.8796854137726,
                    1.8885319079182,
                    1.8786941089645,
                    1.97409459835073,
                    1.62958391896495,
                    1.82760802775089,
                    2.8975109635348,
                    3.09689597650008,
                    1.58464594450494,
                    1.14871004855994,
                    2.58808280927041,
                ],
                [
                    -0.225373160866869,
                    -0.226200932945127,
                    -0.227308149382229,
                    -0.226033544206953,
                    -0.240724530809131,
                    -0.212600136136675,
                    -0.22680915763606,
                    -0.360345976613897,
                    -0.406995151982164,
                    -0.20709981784954,
                    -0.152201414108276,
                    -0.321084756117601,
                ],
                [
                    0.00429873375718503,
                    0.00432273984048009,
                    0.00435355113473343,
                    0.004317649501928,
                    0.00478258342268677,
                    0.00432483494142831,
                    0.00449647038318246,
                    0.00831024231795851,
                    0.010142414858847,
                    0.00417627997424312,
                    0.00271934090238629,
                    0.00716447885775623,
                ],
            ],
        ],
        # 2
        [
            [
                [
                    85.1564764658609,
                    91.6547089894612,
                    93.565439860026,
                    93.535823949178,
                    92.0481086730957,
                    86.8249804178873,
                    85.9976952870686,
                    72.6421396891276,
                    96.5823135375976,
                    87.0619514465331,
                    82.4737564086914,
                    72.3959491729736,
                ],
                [
                    2.74407176297122,
                    2.0319228101907,
                    1.81394758795063,
                    1.82102007299039,
                    1.79296200795978,
                    1.54658276440104,
                    2.26639224099114,
                    3.65000097457068,
                    -1.73607784810691,
                    1.44263681887471,
                    1.69962971582836,
                    3.61353762499938,
                ],
                [
                    -0.723986582322556,
                    -0.714562323821455,
                    -0.710803460407925,
                    -0.711334225181102,
                    -0.69915133811933,
                    -0.608742697255596,
                    -0.685327583815391,
                    -0.754802394858052,
                    -0.44335885481401,
                    -0.598831320142414,
                    -0.577510324669329,
                    -0.747078508763882,
                ],
                [
                    0.0227423496920654,
                    0.0232715182463699,
                    0.0233990817266374,
                    0.0234117152251842,
                    0.0229979471912103,
                    0.0196910004515748,
                    0.0218432847158615,
                    0.0231516033341557,
                    0.0177618638433592,
                    0.0194232008118174,
                    0.0182592260254013,
                    0.0228826741238574,
                ],
            ],
            [
                [
                    66.4012466430664,
                    67.2195220947265,
                    67.5188101450602,
                    67.346419652303,
                    67.2915932973225,
                    65.0825124104817,
                    66.0859336853027,
                    67.690056737264,
                    65.3774796168009,
                    64.720394261678,
                    60.907541402181,
                    64.498957824707,
                ],
                [
                    3.89120894656593,
                    3.92224557323797,
                    3.93842336421989,
                    3.93293342723714,
                    3.8652354935481,
                    3.01500611835056,
                    3.58117056411649,
                    4.04100157664373,
                    3.17228063299571,
                    2.95755401765458,
                    2.56456704499098,
                    3.97156542145939,
                ],
                [
                    -0.642838276349581,
                    -0.655991266379546,
                    -0.66138782256689,
                    -0.658741607532636,
                    -0.658541790533178,
                    -0.539300752408577,
                    -0.612636294954029,
                    -0.742084025225162,
                    -0.67028194763166,
                    -0.528796615022602,
                    -0.445057067559872,
                    -0.694277856177662,
                ],
                [
                    0.0180425257671685,
                    0.0185278038137535,
                    0.0187227190606059,
                    0.0186241317740132,
                    0.0187402393063929,
                    0.0151462386352848,
                    0.0172908746093952,
                    0.0221230130651336,
                    0.0206381637825926,
                    0.0148055673849703,
                    0.0120420129890116,
                    0.0202703802920907,
                ],
            ],
            [
                [
                    48.5813054402669,
                    48.6066556294759,
                    48.6273543039958,
                    48.6000146230062,
                    48.4340633392334,
                    47.6174352010091,
                    48.0646458943685,
                    46.7616485595703,
                    44.0339042027791,
                    47.4886646270752,
                    45.6920996348063,
                    46.3646500905354,
                ],
                [
                    1.8739499741964,
                    1.8796854137726,
                    1.8885319079182,
                    1.8786941089645,
                    1.97409459835073,
                    1.62958391896495,
                    1.82760802775089,
                    2.8975109635348,
                    3.09689597650008,
                    1.58464594450494,
                    1.14871004855994,
                    2.58808280927041,
                ],
                [
                    -0.225373160866869,
                    -0.226200932945127,
                    -0.227308149382229,
                    -0.226033544206953,
                    -0.240724530809131,
                    -0.212600136136675,
                    -0.22680915763606,
                    -0.360345976613897,
                    -0.406995151982164,
                    -0.20709981784954,
                    -0.152201414108276,
                    -0.321084756117601,
                ],
                [
                    0.00429873375718503,
                    0.00432273984048009,
                    0.00435355113473343,
                    0.004317649501928,
                    0.00478258342268677,
                    0.00432483494142831,
                    0.00449647038318246,
                    0.00831024231795851,
                    0.010142414858847,
                    0.00417627997424312,
                    0.00271934090238629,
                    0.00716447885775623,
                ],
            ],
        ],
        # 3
        [
            [
                [
                    85.8301226298014,
                    92.5364913940429,
                    94.5085728963216,
                    94.4778764088948,
                    92.9423764546712,
                    87.5516625722249,
                    92.2907770792643,
                    88.1383926391601,
                    105.723029136658,
                    87.7952003479003,
                    83.0485412597656,
                    85.2334046681722,
                ],
                [
                    2.76353973071523,
                    2.02859954641324,
                    1.80361287947673,
                    1.81094375468081,
                    1.7820103792785,
                    1.52776973971862,
                    1.57166789721035,
                    1.78358304676569,
                    -3.47607245904674,
                    1.42067881786463,
                    1.68753954006335,
                    2.09480160081118,
                ],
                [
                    -0.727794349332513,
                    -0.7180705576232,
                    -0.714189379921049,
                    -0.714739563303952,
                    -0.702167957257003,
                    -0.608864017855594,
                    -0.668705626245423,
                    -0.709791267946328,
                    -0.337870164906785,
                    -0.598645759351327,
                    -0.576705314062692,
                    -0.71218387897198,
                ],
                [
                    0.0228391936701588,
                    0.0233853814577816,
                    0.0235169905512113,
                    0.0235300867151826,
                    0.0231030977504052,
                    0.0196902138740404,
                    0.0220226946040097,
                    0.0237293150782493,
                    0.0157560982163229,
                    0.0194140311443445,
                    0.0182132547737187,
                    0.0234161255983947,
                ],
            ],
            [
                [
                    66.8897253672281,
                    67.7347124735514,
                    68.0428328196207,
                    67.8656948089599,
                    67.809558614095,
                    65.530953725179,
                    66.5657899220784,
                    68.2183176676432,
                    65.6708489100138,
                    65.1593985239664,
                    61.2320920308431,
                    64.934306971232,
                ],
                [
                    3.92262668728181,
                    3.95464375598911,
                    3.97138366165696,
                    3.96566395696598,
                    3.89576021830242,
                    3.01814951855913,
                    3.60255118053648,
                    4.07732136155065,
                    3.26660427587758,
                    2.95871860404857,
                    2.55172267993848,
                    4.00502188778147,
                ],
                [
                    -0.646183859774006,
                    -0.659762866013535,
                    -0.665328171425489,
                    -0.662600290525211,
                    -0.662395282225177,
                    -0.539332279236445,
                    -0.615020300680664,
                    -0.748603791345806,
                    -0.685800740768861,
                    -0.528502413165042,
                    -0.442012744072157,
                    -0.69930575277422,
                ],
                [
                    0.0181091902509807,
                    0.0186102246719455,
                    0.0188111558660761,
                    0.0187096094659066,
                    0.0188295371604689,
                    0.0151205114736609,
                    0.017333846455317,
                    0.0223200490702561,
                    0.0211997073383969,
                    0.0147695614556148,
                    0.011916750516647,
                    0.0204105627286685,
                ],
            ],
            [
                [
                    48.9660090128581,
                    48.9921900431315,
                    49.0135107676188,
                    48.9853623708089,
                    48.8140972137451,
                    47.9714805603027,
                    48.4329212188721,
                    47.0879604339599,
                    44.1095331192016,
                    47.8388140360514,
                    45.9853543599447,
                    46.6790084838867,
                ],
                [
                    1.86652427619987,
                    1.87246617218646,
                    1.88157763143983,
                    1.87145376020193,
                    1.96991651819581,
                    1.614431174485,
                    1.81877838111054,
                    2.92285519565088,
                    3.2154786646042,
                    1.56821759023126,
                    1.11906351002039,
                    2.60415657009371,
                ],
                [
                    -0.223919490000585,
                    -0.22477679319315,
                    -0.225916761340517,
                    -0.224605659226993,
                    -0.239768110510908,
                    -0.210752671137279,
                    -0.225413036235285,
                    -0.363213118973312,
                    -0.422778833162535,
                    -0.205099066376408,
                    -0.148559492506903,
                    -0.322781935716287,
                ],
                [
                    0.00423546076376677,
                    0.0042603284514219,
                    0.00429204136434228,
                    0.00425512659151423,
                    0.0047349804764861,
                    0.00426289112695726,
                    0.0044398953401496,
                    0.00837537995955699,
                    0.0106826684876941,
                    0.00411029230649841,
                    0.00261066750953989,
                    0.00719562851373041,
                ],
            ],
        ],
        # 4
        [
            [
                [
                    85.8328000386555,
                    92.5405784606934,
                    94.5130666097005,
                    94.4823734283447,
                    92.9465479532877,
                    87.5547232309977,
                    92.2948244730631,
                    94.6482789357503,
                    104.442230288188,
                    87.798326365153,
                    83.0506945292154,
                    88.8581707000732,
                ],
                [
                    2.76402749807779,
                    2.02893554247343,
                    1.80390197728713,
                    1.81123249421494,
                    1.78229529485282,
                    1.52799467940433,
                    1.57190576818625,
                    1.07046203531772,
                    -2.7444531154336,
                    1.42087922915064,
                    1.687791064069,
                    1.87585661479829,
                ],
                [
                    -0.727841540650056,
                    -0.718116088902756,
                    -0.714234062008092,
                    -0.71478423594317,
                    -0.702210241820153,
                    -0.608886112977855,
                    -0.668740658215432,
                    -0.716098058473816,
                    -0.442663164405558,
                    -0.598665627566251,
                    -0.576720309424235,
                    -0.744470640773818,
                ],
                [
                    0.0228402619224792,
                    0.0233865770704541,
                    0.0235182120990828,
                    0.0235313086335568,
                    0.0231042372318971,
                    0.0196906205935357,
                    0.0220236015023422,
                    0.0252758946266857,
                    0.0199462469455894,
                    0.0194143791391392,
                    0.0182133468436751,
                    0.0253793143818521,
                ],
            ],
            [
                [
                    66.8930197397867,
                    67.7381940205892,
                    68.0463653564453,
                    67.8692123413085,
                    67.8130583445231,
                    65.5339705149332,
                    66.5690293629964,
                    67.2903713226318,
                    64.3863873799641,
                    65.162361907959,
                    61.2342848459879,
                    64.0055136362711,
                ],
                [
                    3.92283923438843,
                    3.95486082730594,
                    3.97160693000432,
                    3.96588031210082,
                    3.89596547835914,
                    3.01817267239326,
                    3.6026960256587,
                    4.57290480131315,
                    3.99904281553968,
                    2.9587263771982,
                    2.55163431501056,
                    4.50070412323841,
                ],
                [
                    -0.646206471191976,
                    -0.659788171172421,
                    -0.665354796500751,
                    -0.662625912464027,
                    -0.662421249287391,
                    -0.539332618246546,
                    -0.615036475352753,
                    -0.813788039978847,
                    -0.790650869027163,
                    -0.528500441348914,
                    -0.441992002211529,
                    -0.764495568953473,
                ],
                [
                    0.0181096409676885,
                    0.0186107767211808,
                    0.0188117528387809,
                    0.0187101749458699,
                    0.0188301390423364,
                    0.0151203405792249,
                    0.0173341393378139,
                    0.0246962811841276,
                    0.0253911507435334,
                    0.0147693193320072,
                    0.0119158991011151,
                    0.0227869831506097,
                ],
            ],
            [
                [
                    48.9685961405436,
                    48.9947900136312,
                    49.0161178588867,
                    48.9879688262939,
                    48.8166620890299,
                    47.9738717397054,
                    48.4354076385498,
                    46.1551774342855,
                    42.8269112269083,
                    47.8411763509114,
                    45.9873320261637,
                    45.7445655822754,
                ],
                [
                    1.86647776697807,
                    1.87241788047548,
                    1.88153011778481,
                    1.87140265859739,
                    1.96988760822952,
                    1.61432864893845,
                    1.81871881351605,
                    3.4202135011218,
                    3.94654485154912,
                    1.56810707034487,
                    1.1188644833035,
                    3.1022933050006,
                ],
                [
                    -0.22391004051084,
                    -0.224767203097577,
                    -0.225907295733899,
                    -0.224595785140991,
                    -0.239761554833615,
                    -0.21074019445406,
                    -0.225403616200516,
                    -0.428615996887634,
                    -0.527468157814934,
                    -0.205085609898423,
                    -0.148535042098074,
                    -0.38828754313898,
                ],
                [
                    0.00423504404069476,
                    0.00425990746756719,
                    0.00429162371501315,
                    0.0042546975492227,
                    0.00473465582337043,
                    0.004262473199751,
                    0.00443951344434477,
                    0.0107595943719172,
                    0.0148697452852684,
                    0.0041098486293446,
                    0.00260993808206886,
                    0.00958364678614809,
                ],
            ],
        ],
        # 5
        [
            [
                [
                    84.179987335205,
                    90.8830510457356,
                    92.8542772928873,
                    92.8235649108886,
                    91.4898681640625,
                    86.1385505676269,
                    90.84542872111,
                    102.794658660889,
                    108.415410995483,
                    86.3850741068521,
                    81.6628318786621,
                    94.514970143636,
                ],
                [
                    3.74622243601128,
                    3.0134840152388,
                    2.78907976395046,
                    2.79641968979797,
                    2.67361289100678,
                    2.39924534439765,
                    2.45960421884533,
                    -0.525191991757107,
                    -4.68356715641747,
                    2.29069394183513,
                    2.54505876098017,
                    0.76706480739094,
                ],
                [
                    -0.876253411486434,
                    -0.866801894350209,
                    -0.862992993601554,
                    -0.863544227915769,
                    -0.841521878898284,
                    -0.745893149387033,
                    -0.807634459231006,
                    -0.644413941390033,
                    -0.233547946512006,
                    -0.735509992479446,
                    -0.712130134755916,
                    -0.70293578234586,
                ],
                [
                    0.0291295301664126,
                    0.0296838554005894,
                    0.0298176223025734,
                    0.0298307480141463,
                    0.0292096508503331,
                    0.0257303131210221,
                    0.0281169792433163,
                    0.0260760795264256,
                    0.0152210023112323,
                    0.025449556996865,
                    0.0242079737210515,
                    0.0267778839727845,
                ],
            ],
            [
                [
                    65.2721605936686,
                    66.1134562174478,
                    66.4200131734211,
                    66.2436546325683,
                    66.3860654195149,
                    64.2185136159261,
                    65.1705098470051,
                    68.2339937845866,
                    68.3625942230224,
                    63.8567989349365,
                    60.0371148427327,
                    64.9555394490559,
                ],
                [
                    4.88906835157371,
                    4.9230259389218,
                    4.94057682158881,
                    4.93445376490289,
                    4.77244971422791,
                    3.83945365237376,
                    4.46503525254519,
                    4.27069697050319,
                    2.05821019026691,
                    3.77512224867258,
                    3.31449387475514,
                    4.19491060519276,
                ],
                [
                    -0.792757006514046,
                    -0.806564007049953,
                    -0.812224085792241,
                    -0.80944908526672,
                    -0.800003558049947,
                    -0.670585250521041,
                    -0.750986130365404,
                    -0.814766332541869,
                    -0.581303975520991,
                    -0.659195404230577,
                    -0.566571591895223,
                    -0.76499707993372,
                ],
                [
                    0.0243443188596902,
                    0.0248520408569609,
                    0.025055738225313,
                    0.0249528129988155,
                    0.024884833137287,
                    0.0209950748671834,
                    0.0233418239394917,
                    0.0267817231232378,
                    0.0206581396749063,
                    0.02062833624101,
                    0.0176023373607526,
                    0.0248566151776792,
                ],
            ],
            [
                [
                    47.6635588328043,
                    47.6881294250488,
                    47.7072605133056,
                    47.6815816243489,
                    47.6765651702881,
                    47.2017490386963,
                    47.4113597869873,
                    47.2224323272705,
                    46.8611689249674,
                    47.0904954274495,
                    45.6525849660237,
                    46.8545650482177,
                ],
                [
                    2.67509635141309,
                    2.68184523741775,
                    2.69205027990038,
                    2.68069341151217,
                    2.70317845133357,
                    2.16664229313718,
                    2.49467595010742,
                    3.04853771486854,
                    1.97275847395725,
                    2.10986953422648,
                    1.44456846545441,
                    2.70673382402671,
                ],
                [
                    -0.352122929268506,
                    -0.353073932074167,
                    -0.354340476589603,
                    -0.352886585088877,
                    -0.360680108859545,
                    -0.311140108775426,
                    -0.339768033483368,
                    -0.420326584702606,
                    -0.313680664920585,
                    -0.30428820945722,
                    -0.221447879617865,
                    -0.376829031741981,
                ],
                [
                    0.00993380022308304,
                    0.00996139268126708,
                    0.00999676773690771,
                    0.00995571528096115,
                    0.0103021682281317,
                    0.009259238758102,
                    0.00982170523648682,
                    0.0125369933009055,
                    0.00998790790243677,
                    0.00907325698316053,
                    0.00678179721639615,
                    0.0112561083803273,
                ],
            ],
        ],
        # 6
        [
            [
                [
                    84.179987335205,
                    90.8830510457356,
                    92.8542772928873,
                    92.8235649108886,
                    91.4898681640625,
                    86.1385505676269,
                    90.84542872111,
                    102.794658660889,
                    108.415410995483,
                    86.3850741068521,
                    81.6628318786621,
                    94.514970143636,
                ],
                [
                    3.74622243601128,
                    3.0134840152388,
                    2.78907976395046,
                    2.79641968979797,
                    2.67361289100678,
                    2.39924534439765,
                    2.45960421884533,
                    -0.525191991757107,
                    -4.68356715641747,
                    2.29069394183513,
                    2.54505876098017,
                    0.76706480739094,
                ],
                [
                    -0.876253411486434,
                    -0.866801894350209,
                    -0.862992993601554,
                    -0.863544227915769,
                    -0.841521878898284,
                    -0.745893149387033,
                    -0.807634459231006,
                    -0.644413941390033,
                    -0.233547946512006,
                    -0.735509992479446,
                    -0.712130134755916,
                    -0.70293578234586,
                ],
                [
                    0.0291295301664126,
                    0.0296838554005894,
                    0.0298176223025734,
                    0.0298307480141463,
                    0.0292096508503331,
                    0.0257303131210221,
                    0.0281169792433163,
                    0.0260760795264256,
                    0.0152210023112323,
                    0.025449556996865,
                    0.0242079737210515,
                    0.0267778839727845,
                ],
            ],
            [
                [
                    65.2721605936686,
                    66.1134562174478,
                    66.4200131734211,
                    66.2436546325683,
                    66.3860654195149,
                    64.2185136159261,
                    65.1705098470051,
                    68.2339937845866,
                    68.3625942230224,
                    63.8567989349365,
                    60.0371148427327,
                    64.9555394490559,
                ],
                [
                    4.88906835157371,
                    4.9230259389218,
                    4.94057682158881,
                    4.93445376490289,
                    4.77244971422791,
                    3.83945365237376,
                    4.46503525254519,
                    4.27069697050319,
                    2.05821019026691,
                    3.77512224867258,
                    3.31449387475514,
                    4.19491060519276,
                ],
                [
                    -0.792757006514046,
                    -0.806564007049953,
                    -0.812224085792241,
                    -0.80944908526672,
                    -0.800003558049947,
                    -0.670585250521041,
                    -0.750986130365404,
                    -0.814766332541869,
                    -0.581303975520991,
                    -0.659195404230577,
                    -0.566571591895223,
                    -0.76499707993372,
                ],
                [
                    0.0243443188596902,
                    0.0248520408569609,
                    0.025055738225313,
                    0.0249528129988155,
                    0.024884833137287,
                    0.0209950748671834,
                    0.0233418239394917,
                    0.0267817231232378,
                    0.0206581396749063,
                    0.02062833624101,
                    0.0176023373607526,
                    0.0248566151776792,
                ],
            ],
            [
                [
                    47.6635588328043,
                    47.6881294250488,
                    47.7072605133056,
                    47.6815816243489,
                    47.6765651702881,
                    47.2017490386963,
                    47.4113597869873,
                    47.2224323272705,
                    46.8611689249674,
                    47.0904954274495,
                    45.6525849660237,
                    46.8545650482177,
                ],
                [
                    2.67509635141309,
                    2.68184523741775,
                    2.69205027990038,
                    2.68069341151217,
                    2.70317845133357,
                    2.16664229313718,
                    2.49467595010742,
                    3.04853771486854,
                    1.97275847395725,
                    2.10986953422648,
                    1.44456846545441,
                    2.70673382402671,
                ],
                [
                    -0.352122929268506,
                    -0.353073932074167,
                    -0.354340476589603,
                    -0.352886585088877,
                    -0.360680108859545,
                    -0.311140108775426,
                    -0.339768033483368,
                    -0.420326584702606,
                    -0.313680664920585,
                    -0.30428820945722,
                    -0.221447879617865,
                    -0.376829031741981,
                ],
                [
                    0.00993380022308304,
                    0.00996139268126708,
                    0.00999676773690771,
                    0.00995571528096115,
                    0.0103021682281317,
                    0.009259238758102,
                    0.00982170523648682,
                    0.0125369933009055,
                    0.00998790790243677,
                    0.00907325698316053,
                    0.00678179721639615,
                    0.0112561083803273,
                ],
            ],
        ],
        # 7
        [
            [
                [
                    84.179987335205,
                    90.8830510457356,
                    92.8542772928873,
                    92.8235649108886,
                    91.4898681640625,
                    86.1385505676269,
                    90.84542872111,
                    102.794658660889,
                    108.415410995483,
                    86.3850741068521,
                    81.6628318786621,
                    94.514970143636,
                ],
                [
                    3.74622243601128,
                    3.0134840152388,
                    2.78907976395046,
                    2.79641968979797,
                    2.67361289100678,
                    2.39924534439765,
                    2.45960421884533,
                    -0.525191991757107,
                    -4.68356715641747,
                    2.29069394183513,
                    2.54505876098017,
                    0.76706480739094,
                ],
                [
                    -0.876253411486434,
                    -0.866801894350209,
                    -0.862992993601554,
                    -0.863544227915769,
                    -0.841521878898284,
                    -0.745893149387033,
                    -0.807634459231006,
                    -0.644413941390033,
                    -0.233547946512006,
                    -0.735509992479446,
                    -0.712130134755916,
                    -0.70293578234586,
                ],
                [
                    0.0291295301664126,
                    0.0296838554005894,
                    0.0298176223025734,
                    0.0298307480141463,
                    0.0292096508503331,
                    0.0257303131210221,
                    0.0281169792433163,
                    0.0260760795264256,
                    0.0152210023112323,
                    0.025449556996865,
                    0.0242079737210515,
                    0.0267778839727845,
                ],
            ],
            [
                [
                    65.2721605936686,
                    66.1134562174478,
                    66.4200131734211,
                    66.2436546325683,
                    66.3860654195149,
                    64.2185136159261,
                    65.1705098470051,
                    68.2339937845866,
                    68.3625942230224,
                    63.8567989349365,
                    60.0371148427327,
                    64.9555394490559,
                ],
                [
                    4.88906835157371,
                    4.9230259389218,
                    4.94057682158881,
                    4.93445376490289,
                    4.77244971422791,
                    3.83945365237376,
                    4.46503525254519,
                    4.27069697050319,
                    2.05821019026691,
                    3.77512224867258,
                    3.31449387475514,
                    4.19491060519276,
                ],
                [
                    -0.792757006514046,
                    -0.806564007049953,
                    -0.812224085792241,
                    -0.80944908526672,
                    -0.800003558049947,
                    -0.670585250521041,
                    -0.750986130365404,
                    -0.814766332541869,
                    -0.581303975520991,
                    -0.659195404230577,
                    -0.566571591895223,
                    -0.76499707993372,
                ],
                [
                    0.0243443188596902,
                    0.0248520408569609,
                    0.025055738225313,
                    0.0249528129988155,
                    0.024884833137287,
                    0.0209950748671834,
                    0.0233418239394917,
                    0.0267817231232378,
                    0.0206581396749063,
                    0.02062833624101,
                    0.0176023373607526,
                    0.0248566151776792,
                ],
            ],
            [
                [
                    47.6635588328043,
                    47.6881294250488,
                    47.7072605133056,
                    47.6815816243489,
                    47.6765651702881,
                    47.2017490386963,
                    47.4113597869873,
                    47.2224323272705,
                    46.8611689249674,
                    47.0904954274495,
                    45.6525849660237,
                    46.8545650482177,
                ],
                [
                    2.67509635141309,
                    2.68184523741775,
                    2.69205027990038,
                    2.68069341151217,
                    2.70317845133357,
                    2.16664229313718,
                    2.49467595010742,
                    3.04853771486854,
                    1.97275847395725,
                    2.10986953422648,
                    1.44456846545441,
                    2.70673382402671,
                ],
                [
                    -0.352122929268506,
                    -0.353073932074167,
                    -0.354340476589603,
                    -0.352886585088877,
                    -0.360680108859545,
                    -0.311140108775426,
                    -0.339768033483368,
                    -0.420326584702606,
                    -0.313680664920585,
                    -0.30428820945722,
                    -0.221447879617865,
                    -0.376829031741981,
                ],
                [
                    0.00993380022308304,
                    0.00996139268126708,
                    0.00999676773690771,
                    0.00995571528096115,
                    0.0103021682281317,
                    0.009259238758102,
                    0.00982170523648682,
                    0.0125369933009055,
                    0.00998790790243677,
                    0.00907325698316053,
                    0.00678179721639615,
                    0.0112561083803273,
                ],
            ],
        ],
    ],
    dtype=np.float64,
)

# 13. Road Deterioration
dRoadDet = np.array(
    [
        [1, 0.02, 0, 0, 0, 0, 0],
        [3, 0, 1, 1, 134, 0.7947, 0.0054],
        [3, 0, 1, 1, 134, 0.7947, 0.0054],
        [1, 0.1, 0, 0, 0, 0, 0],
        [1, 0.1, 0, 0, 0, 0, 0],
        [1, 0.05, 0, 0, 0, 0, 0],
        [1, 0.05, 0, 0, 0, 0, 0],
    ],
    dtype=np.float64,
)

# 14. Roughness and Condition Category
iri_cc = np.array(
    [
        [1, 0.00, 3.50, 1],
        [1, 3.50, 5.50, 2],
        [1, 5.50, 8.50, 3],
        [1, 8.50, 12.00, 4],
        [1, 12.00, 500.00, 5],
        [2, 0.00, 2.50, 1],
        [2, 2.50, 4.50, 2],
        [2, 4.50, 9.00, 3],
        [2, 9.00, 14.00, 4],
        [2, 14.00, 500.00, 5],
        [3, 0.00, 3.50, 1],
        [3, 3.50, 5.50, 2],
        [3, 5.50, 9.50, 3],
        [3, 9.50, 14.00, 4],
        [3, 14.00, 500.00, 5],
        [4, 0.00, 6.00, 1],
        [4, 6.00, 9.00, 2],
        [4, 9.00, 13.50, 3],
        [4, 13.50, 18.00, 4],
        [4, 18.00, 500.00, 5],
        [5, 0.00, 8.00, 1],
        [5, 8.00, 11.00, 2],
        [5, 11.00, 15.50, 3],
        [5, 15.50, 20.00, 4],
        [5, 20.00, 500.00, 5],
        [6, 0.00, 3.50, 1],
        [6, 3.50, 5.00, 2],
        [6, 5.00, 8.00, 3],
        [6, 8.00, 12.00, 4],
        [6, 12.00, 500.00, 5],
        [7, 0.00, 6.00, 1],
        [7, 6.00, 9.00, 2],
        [7, 9.00, 13.50, 3],
        [7, 13.50, 18.00, 4],
        [7, 18.00, 500.00, 5],
    ],
    dtype=np.float64,
)


iri_cc_df = pd.DataFrame(data=iri_cc, columns=["SurfaceType", "RoughnessFrom", "RoughnessTo", "ConditionCategory"])

iri_cc_df[["SurfaceType", "ConditionCategory"]] = iri_cc_df[["SurfaceType", "ConditionCategory"]].astype(int)


def default_range(data):
    # from intervaltree import Interval, IntervalTree
    t = IntervalTree()
    for (l, u, v) in data:
        t[l:u] = v

    def lu(v):
        e = None
        for e in t[v]:
            break
        if e is None:
            print(v)
            print(t)
        return e.data

    return lu


traffic_range_lu = default_range(traffic_ranges_data)
lanes_lu = default_range(default_lanes)


def get_cc_from_iri_lu():
    def f(surface_type):
        data = np.squeeze(iri_cc[np.where(iri_cc[:, 0] == surface_type), 1:])
        return default_range(data)

    return {i: f(i) for i in range(1, 8)}


cc_from_iri_lu = get_cc_from_iri_lu()
