"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualServiceProvider = exports.VirtualService = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualServiceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, VirtualService);
            }
            throw error;
        }
        const providerConfig = props.virtualServiceProvider.bind(this);
        this.mesh = providerConfig.mesh;
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            virtualServiceName: this.physicalName,
            spec: {
                provider: providerConfig.virtualNodeProvider || providerConfig.virtualRouterProvider
                    ? {
                        virtualNode: providerConfig.virtualNodeProvider,
                        virtualRouter: providerConfig.virtualRouterProvider,
                    }
                    : undefined,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${this.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceArn = virtualServiceArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualServiceArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.virtualServiceName = cdk.Fn.select(2, this.parsedArn);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualService given its attributes
     */
    static fromVirtualServiceAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualServiceAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVirtualServiceAttributes);
            }
            throw error;
        }
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceName = attrs.virtualServiceName;
                this.mesh = attrs.mesh;
                this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualService`,
                    resourceName: this.virtualServiceName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualService = VirtualService;
_a = JSII_RTTI_SYMBOL_1;
VirtualService[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualService", version: "1.169.0" };
/**
 * Represents the properties needed to define the provider for a VirtualService
 */
class VirtualServiceProvider {
    /**
     * Returns a VirtualNode based Provider for a VirtualService
     */
    static virtualNode(virtualNode) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_IVirtualNode(virtualNode);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.virtualNode);
            }
            throw error;
        }
        return new VirtualServiceProviderImpl(virtualNode, undefined);
    }
    /**
     * Returns a VirtualRouter based Provider for a VirtualService
     */
    static virtualRouter(virtualRouter) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_IVirtualRouter(virtualRouter);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.virtualRouter);
            }
            throw error;
        }
        return new VirtualServiceProviderImpl(undefined, virtualRouter);
    }
    /**
     * Returns an Empty Provider for a VirtualService. This provides no routing capabilities
     * and should only be used as a placeholder
     */
    static none(mesh) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_IMesh(mesh);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.none);
            }
            throw error;
        }
        return new VirtualServiceProviderImpl(undefined, undefined, mesh);
    }
}
exports.VirtualServiceProvider = VirtualServiceProvider;
_b = JSII_RTTI_SYMBOL_1;
VirtualServiceProvider[_b] = { fqn: "@aws-cdk/aws-appmesh.VirtualServiceProvider", version: "1.169.0" };
class VirtualServiceProviderImpl extends VirtualServiceProvider {
    constructor(virtualNode, virtualRouter, mesh) {
        super();
        this.virtualNode = virtualNode;
        this.virtualRouter = virtualRouter;
        const providedMesh = this.virtualNode?.mesh ?? this.virtualRouter?.mesh ?? mesh;
        this.mesh = providedMesh;
    }
    bind(_construct) {
        return {
            mesh: this.mesh,
            virtualNodeProvider: this.virtualNode
                ? {
                    virtualNodeName: this.virtualNode.virtualNodeName,
                }
                : undefined,
            virtualRouterProvider: this.virtualRouter
                ? {
                    virtualRouterName: this.virtualRouter.virtualRouterName,
                }
                : undefined,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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