"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpGatewayRoutePathMatch = exports.HttpRoutePathMatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Defines HTTP route matching based on the URL path of the request.
 */
class HttpRoutePathMatch {
    /**
     * The value of the path must match the specified value exactly.
     * The provided `path` must start with the '/' character.
     *
     * @param path the exact path to match on
     */
    static exactly(path) {
        return new HttpRouteWholePathMatch({ exact: path });
    }
    /**
     * The value of the path must match the specified regex.
     *
     * @param regex the regex used to match the path
     */
    static regex(regex) {
        return new HttpRouteWholePathMatch({ regex: regex });
    }
    /**
     * The value of the path must match the specified prefix.
     *
     * @param prefix the value to use to match the beginning of the path part of the URL of the request.
     *   It must start with the '/' character. If provided as "/", matches all requests.
     *   For example, if your virtual service name is "my-service.local"
     *   and you want the route to match requests to "my-service.local/metrics", your prefix should be "/metrics".
     */
    static startsWith(prefix) {
        return new HttpRoutePrefixPathMatch(prefix);
    }
}
exports.HttpRoutePathMatch = HttpRoutePathMatch;
_a = JSII_RTTI_SYMBOL_1;
HttpRoutePathMatch[_a] = { fqn: "@aws-cdk/aws-appmesh.HttpRoutePathMatch", version: "1.169.0" };
class HttpRoutePrefixPathMatch extends HttpRoutePathMatch {
    constructor(prefix) {
        super();
        this.prefix = prefix;
        if (prefix && prefix[0] !== '/') {
            throw new Error(`Prefix Path for the match must start with \'/\', got: ${prefix}`);
        }
    }
    bind(_scope) {
        return {
            prefixPathMatch: this.prefix,
        };
    }
}
class HttpRouteWholePathMatch extends HttpRoutePathMatch {
    constructor(match) {
        super();
        this.match = match;
        if (match.exact && match.exact[0] !== '/') {
            throw new Error(`Exact Path for the match must start with \'/\', got: ${match.exact}`);
        }
    }
    bind(_scope) {
        return {
            wholePathMatch: this.match,
        };
    }
}
/**
 * Defines HTTP gateway route matching based on the URL path of the request.
 */
class HttpGatewayRoutePathMatch {
    /**
     * The value of the path must match the specified prefix.
     *
     * @param prefix the value to use to match the beginning of the path part of the URL of the request.
     *   It must start with the '/' character.
     *   When `rewriteTo` is provided, it must also end with the '/' character.
     *   If provided as "/", matches all requests.
     *   For example, if your virtual service name is "my-service.local"
     *   and you want the route to match requests to "my-service.local/metrics", your prefix should be "/metrics".
     * @param rewriteTo Specify either disabling automatic rewrite or rewriting to specified prefix path.
     *   To disable automatic rewrite, provide `''`.
     *   As a default, request's URL path is automatically rewritten to '/'.
     */
    static startsWith(prefix, rewriteTo) {
        return new HttpGatewayRoutePrefixPathMatch(prefix, rewriteTo);
    }
    /**
     * The value of the path must match the specified value exactly.
     * The provided `path` must start with the '/' character.
     *
     * @param path the exact path to match on
     * @param rewriteTo the value to substitute for the matched part of the path of the gateway request URL
     *   As a default, retains original request's URL path.
     */
    static exactly(path, rewriteTo) {
        return new HttpGatewayRouteWholePathMatch({ exact: path }, rewriteTo);
    }
    /**
     * The value of the path must match the specified regex.
     *
     * @param regex the regex used to match the path
     * @param rewriteTo the value to substitute for the matched part of the path of the gateway request URL
     *   As a default, retains original request's URL path.
     */
    static regex(regex, rewriteTo) {
        return new HttpGatewayRouteWholePathMatch({ regex }, rewriteTo);
    }
}
exports.HttpGatewayRoutePathMatch = HttpGatewayRoutePathMatch;
_b = JSII_RTTI_SYMBOL_1;
HttpGatewayRoutePathMatch[_b] = { fqn: "@aws-cdk/aws-appmesh.HttpGatewayRoutePathMatch", version: "1.169.0" };
class HttpGatewayRoutePrefixPathMatch extends HttpGatewayRoutePathMatch {
    constructor(prefixPathMatch, rewriteTo) {
        super();
        this.prefixPathMatch = prefixPathMatch;
        this.rewriteTo = rewriteTo;
        if (prefixPathMatch[0] !== '/') {
            throw new Error('Prefix path for the match must start with \'/\', '
                + `got: ${prefixPathMatch}`);
        }
        if (rewriteTo) {
            if (prefixPathMatch[prefixPathMatch.length - 1] !== '/') {
                throw new Error('When prefix path for the rewrite is specified, prefix path for the match must end with \'/\', '
                    + `got: ${prefixPathMatch}`);
            }
            if (rewriteTo[0] !== '/' || rewriteTo[rewriteTo.length - 1] !== '/') {
                throw new Error('Prefix path for the rewrite must start and end with \'/\', '
                    + `got: ${rewriteTo}`);
            }
        }
    }
    bind(_scope) {
        return {
            prefixPathMatch: this.prefixPathMatch,
            prefixPathRewrite: this.rewriteTo === undefined
                ? undefined
                : {
                    defaultPrefix: this.rewriteTo === '' ? 'DISABLED' : undefined,
                    value: this.rewriteTo === '' ? undefined : this.rewriteTo,
                },
        };
    }
}
class HttpGatewayRouteWholePathMatch extends HttpGatewayRoutePathMatch {
    constructor(wholePathMatch, exactPathRewrite) {
        super();
        this.wholePathMatch = wholePathMatch;
        this.exactPathRewrite = exactPathRewrite;
        if (wholePathMatch.exact && wholePathMatch.exact[0] !== '/') {
            throw new Error(`Exact Path for the match must start with \'/\', got: ${wholePathMatch.exact}`);
        }
        if (exactPathRewrite === '') {
            throw new Error('Exact Path for the rewrite cannot be empty. Unlike startsWith() method, no automatic rewrite on whole path match');
        }
        if (exactPathRewrite && exactPathRewrite[0] !== '/') {
            throw new Error(`Exact Path for the rewrite must start with \'/\', got: ${exactPathRewrite}`);
        }
    }
    bind(_scope) {
        return {
            wholePathMatch: this.wholePathMatch,
            wholePathRewrite: this.exactPathRewrite === undefined ? undefined : { exact: this.exactPathRewrite },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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