from random import randrange, seed, sample
from collections import Counter
from binteger import Bin

from subsets import DenseSet, DenseBox, DenseTernary
from subsets.max_cubes import MaxCubes
from subsets.max_cubes import MaxCubes_Dense2
from subsets.max_cubes import MaxCubes_Dense3


def test_QMC1():
    seed(123)
    for n in range(1, 12):
        P = DenseSet(n)
        for i in range(1000):
            P.set(randrange(2**n))

            if i < 10 or i % 10 == 0:
                S = MaxCubes(P)

                Sa = {}
                for a, u in S:
                    Sa.setdefault(a, []).append(u)

                print(len(S), len(Sa))
                for a in Sa:
                    # prec(u)
                    U = DenseSet(n)
                    for u in Sa[a]:
                        U.set(a)
                    # a + prec(u)
                    U.Not(a)
                    assert U <= P


def test_QMC1_random():
    for n in (4, 8, 12, 13):
        print("random", n)
        P = DenseSet(n)
        for x in range(2**n):
            if randrange(2):
                P.set(x)

        S = MaxCubes(P)
        S2 = MaxCubes_Dense2(P)
        S3 = MaxCubes_Dense3(P)
        assert set(S) == set(S2) == set(S3)

        Sa = {}
        for a, u in S:
            Sa.setdefault(a, []).append(u)

        for a in Sa:
            # prec(u)
            U = DenseSet(n)
            for u in Sa[a]:
                U.set(a)
            # a + prec(u)
            U.Not(a)
            assert U <= P


def test_implementations():
    seed(123)
    nmax = 12
    repmax = 2
    for n in range(1, nmax+1):
        for _ in range(repmax * nmax // n):
            db = DenseSet(n)
            full = list(range(2**n))

            for e in range(n+1):
                wt = randrange(2**e+1)

            xs = sample(full, wt)
            for x in xs:
                db.add(x)
            if randrange(2):
                db.do_Not()

            # Semi-dense QmC through binary dense sets
            pats = set()
            for a, u in MaxCubes(db):
                a = Bin(a, n)
                u = Bin(u, n)
                pat = ""
                for aa, uu in zip(a.str, u.str):
                    if uu == "1":
                        assert aa == "0"
                        pat += "*"
                    else:
                        pat += aa
                pats.add(pat)

            # Dense through generic DenseBox
            box = DenseBox([2] * n)
            for v in db.to_Bins():
                box.set(v.tuple)

            box.do_Sweep_AND_up_OR()
            box.do_Sweep_NOTAND_down()

            pats2 = set()
            for v in box:
                digits = []
                for i in range(n):
                    digits.append("*" if v % 3 == 2 else str(v % 3))
                    v //= 3
                pat = "".join(digits[::-1])
                pats2.add(pat)

            # Dense through optimized DenseTernary
            ter = DenseTernary(db)
            ter.do_MaxCubes()

            pats3 = set()
            for v in ter:
                digits = []
                for i in range(n):
                    digits.append("*" if v % 3 == 2 else str(v % 3))
                    v //= 3
                pat = "".join(digits[::-1])
                pats3.add(pat)

            assert pats == pats2
            assert pats == pats3


def time_imp_diff():
    AES = [
        0x63,0x7c,0x77,0x7b,0xf2,0x6b,0x6f,0xc5,0x30,0x01,0x67,0x2b,0xfe,0xd7,0xab,0x76,
        0xca,0x82,0xc9,0x7d,0xfa,0x59,0x47,0xf0,0xad,0xd4,0xa2,0xaf,0x9c,0xa4,0x72,0xc0,
        0xb7,0xfd,0x93,0x26,0x36,0x3f,0xf7,0xcc,0x34,0xa5,0xe5,0xf1,0x71,0xd8,0x31,0x15,
        0x04,0xc7,0x23,0xc3,0x18,0x96,0x05,0x9a,0x07,0x12,0x80,0xe2,0xeb,0x27,0xb2,0x75,
        0x09,0x83,0x2c,0x1a,0x1b,0x6e,0x5a,0xa0,0x52,0x3b,0xd6,0xb3,0x29,0xe3,0x2f,0x84,
        0x53,0xd1,0x00,0xed,0x20,0xfc,0xb1,0x5b,0x6a,0xcb,0xbe,0x39,0x4a,0x4c,0x58,0xcf,
        0xd0,0xef,0xaa,0xfb,0x43,0x4d,0x33,0x85,0x45,0xf9,0x02,0x7f,0x50,0x3c,0x9f,0xa8,
        0x51,0xa3,0x40,0x8f,0x92,0x9d,0x38,0xf5,0xbc,0xb6,0xda,0x21,0x10,0xff,0xf3,0xd2,
        0xcd,0x0c,0x13,0xec,0x5f,0x97,0x44,0x17,0xc4,0xa7,0x7e,0x3d,0x64,0x5d,0x19,0x73,
        0x60,0x81,0x4f,0xdc,0x22,0x2a,0x90,0x88,0x46,0xee,0xb8,0x14,0xde,0x5e,0x0b,0xdb,
        0xe0,0x32,0x3a,0x0a,0x49,0x06,0x24,0x5c,0xc2,0xd3,0xac,0x62,0x91,0x95,0xe4,0x79,
        0xe7,0xc8,0x37,0x6d,0x8d,0xd5,0x4e,0xa9,0x6c,0x56,0xf4,0xea,0x65,0x7a,0xae,0x08,
        0xba,0x78,0x25,0x2e,0x1c,0xa6,0xb4,0xc6,0xe8,0xdd,0x74,0x1f,0x4b,0xbd,0x8b,0x8a,
        0x70,0x3e,0xb5,0x66,0x48,0x03,0xf6,0x0e,0x61,0x35,0x57,0xb9,0x86,0xc1,0x1d,0x9e,
        0xe1,0xf8,0x98,0x11,0x69,0xd9,0x8e,0x94,0x9b,0x1e,0x87,0xe9,0xce,0x55,0x28,0xdf,
        0x8c,0xa1,0x89,0x0d,0xbf,0xe6,0x42,0x68,0x41,0x99,0x2d,0x0f,0xb0,0x54,0xbb,0x16
    ]

    SKINNY_8 = [
        0x65,0x4c,0x6a,0x42,0x4b,0x63,0x43,0x6b,0x55,0x75,0x5a,0x7a,0x53,0x73,0x5b,0x7b,
        0x35,0x8c,0x3a,0x81,0x89,0x33,0x80,0x3b,0x95,0x25,0x98,0x2a,0x90,0x23,0x99,0x2b,
        0xe5,0xcc,0xe8,0xc1,0xc9,0xe0,0xc0,0xe9,0xd5,0xf5,0xd8,0xf8,0xd0,0xf0,0xd9,0xf9,
        0xa5,0x1c,0xa8,0x12,0x1b,0xa0,0x13,0xa9,0x05,0xb5,0x0a,0xb8,0x03,0xb0,0x0b,0xb9,
        0x32,0x88,0x3c,0x85,0x8d,0x34,0x84,0x3d,0x91,0x22,0x9c,0x2c,0x94,0x24,0x9d,0x2d,
        0x62,0x4a,0x6c,0x45,0x4d,0x64,0x44,0x6d,0x52,0x72,0x5c,0x7c,0x54,0x74,0x5d,0x7d,
        0xa1,0x1a,0xac,0x15,0x1d,0xa4,0x14,0xad,0x02,0xb1,0x0c,0xbc,0x04,0xb4,0x0d,0xbd,
        0xe1,0xc8,0xec,0xc5,0xcd,0xe4,0xc4,0xed,0xd1,0xf1,0xdc,0xfc,0xd4,0xf4,0xdd,0xfd,
        0x36,0x8e,0x38,0x82,0x8b,0x30,0x83,0x39,0x96,0x26,0x9a,0x28,0x93,0x20,0x9b,0x29,
        0x66,0x4e,0x68,0x41,0x49,0x60,0x40,0x69,0x56,0x76,0x58,0x78,0x50,0x70,0x59,0x79,
        0xa6,0x1e,0xaa,0x11,0x19,0xa3,0x10,0xab,0x06,0xb6,0x08,0xba,0x00,0xb3,0x09,0xbb,
        0xe6,0xce,0xea,0xc2,0xcb,0xe3,0xc3,0xeb,0xd6,0xf6,0xda,0xfa,0xd3,0xf3,0xdb,0xfb,
        0x31,0x8a,0x3e,0x86,0x8f,0x37,0x87,0x3f,0x92,0x21,0x9e,0x2e,0x97,0x27,0x9f,0x2f,
        0x61,0x48,0x6e,0x46,0x4f,0x67,0x47,0x6f,0x51,0x71,0x5e,0x7e,0x57,0x77,0x5f,0x7f,
        0xa2,0x18,0xae,0x16,0x1f,0xa7,0x17,0xaf,0x01,0xb2,0x0e,0xbe,0x07,0xb7,0x0f,0xbf,
        0xe2,0xca,0xee,0xc6,0xcf,0xe7,0xc7,0xef,0xd2,0xf2,0xde,0xfe,0xd7,0xf7,0xdf,0xff
    ]

    PRESENT = 0xC,0x5,0x6,0xB,0x9,0x0,0xA,0xD,0x3,0xE,0xF,0x8,0x4,0x7,0x1,0x2

    for sbox in PRESENT, SKINNY_8, AES:
        if len(sbox) == 256:
            n = 8
        elif len(sbox) == 16:
            n = 4
        else:
            raise

        ddt = DenseSet(2*n)
        for x in range(2**n):
            for dx in range(2**n):
                dy = sbox[x] ^ sbox[x ^ dx]
                ddt.set((dx << n) | dy)

        print("ddt      ", ddt)
        ddt.do_Complement()
        print("ddt compl", ddt)
        S = MaxCubes(ddt)

        ans = 0
        for a, u in S:
            if a & u == 0:
                ans += 1
        assert ans == len(S)

        Sa = {}
        for a, u in S:
            Sa.setdefault(a, []).append(u)
        cnt = Counter(map(len, Sa.values()))
        print(len(S), "inequalities;", len(Sa), "unique a;")
        print()


if __name__ == '__main__':
    time_imp_diff()
