"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addForwarderToLogGroups = exports.addForwarder = exports.SUBSCRIPTION_FILTER_PREFIX = void 0;
const crypto = require("crypto");
const lambda = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const loglevel_1 = require("loglevel");
exports.SUBSCRIPTION_FILTER_PREFIX = "DatadogSubscriptionFilter";
// Change back to 'import { LambdaDestination } from "@aws-cdk/aws-logs-destinations";'
// once https://github.com/aws/aws-cdk/pull/14222 is merged and released.
const lambdaDestination_1 = require("./lambdaDestination");
function generateForwarderConstructId(forwarderArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda Forwarder");
    return "forwarder" + crypto.createHash("sha256").update(forwarderArn).digest("hex");
}
function generateSubscriptionFilterName(functionUniqueId, forwarderArn) {
    const subscriptionFilterValue = crypto
        .createHash("sha256")
        .update(functionUniqueId)
        .update(forwarderArn)
        .digest("hex");
    const subscriptionFilterValueLength = subscriptionFilterValue.length;
    const subscriptionFilterName = exports.SUBSCRIPTION_FILTER_PREFIX +
        subscriptionFilterValue.substring(subscriptionFilterValueLength - 8, subscriptionFilterValueLength);
    return subscriptionFilterName;
}
function getForwarder(scope, forwarderArn) {
    const forwarderConstructId = generateForwarderConstructId(forwarderArn);
    if (scope.node.tryFindChild(forwarderConstructId)) {
        return scope.node.tryFindChild(forwarderConstructId);
    }
    else {
        return lambda.Function.fromFunctionArn(scope, forwarderConstructId, forwarderArn);
    }
}
function addForwarder(scope, lambdaFunctions, forwarderArn) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new lambdaDestination_1.LambdaDestination(forwarder);
    lambdaFunctions.forEach((lam) => {
        const subscriptionFilterName = generateSubscriptionFilterName(cdk.Names.uniqueId(lam), forwarderArn);
        loglevel_1.default.debug(`Adding log subscription ${subscriptionFilterName} for ${lam.functionName}`);
        lam.logGroup.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarder = addForwarder;
function addForwarderToLogGroups(scope, logGroups, forwarderArn) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new lambdaDestination_1.LambdaDestination(forwarder);
    logGroups.forEach((group) => {
        const subscriptionFilterName = generateSubscriptionFilterName(cdk.Names.nodeUniqueId(group.node), forwarderArn);
        group.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarderToLogGroups = addForwarderToLogGroups;
//# sourceMappingURL=data:application/json;base64,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