'''
Construct airfoil with CST
'''
from typing import Tuple
import numpy as np
from numpy.linalg import lstsq

from scipy.special import factorial

from .basic import BasicSection, rotate, interp_from_curve, curve_curvature


#* ===========================================
#* CST sections
#* ===========================================

class Section(BasicSection):
    '''
    Section 3D curve generated by CST foil (upper & lower surface)
    
    Examples
    --------
    >>> sec = Section(thick=None, chord=1.0, twist=0.0, tail=0.0, lTwistAroundLE=True)
    
    Attributes
    ------------
    tail : float
        tail thickness (m)
    RLE : float
        relative leading edge radius
    te_angle : float
        trailing edge angle (degree)
    te_slope : float
        slope of the mean camber line at trailing edge (dy/dx)
    cst_u, cst_l : ndarray
        cst coefficients of the upper and lower surfaces
    refine_u, refine_l : {None, ndarray}
        cst coefficients of the refinement curve on upper and lower surfaces
    
    '''
    def __init__(self, thick=None, chord=1.0, twist=0.0, tail=0.0, lTwistAroundLE=True):

        super().__init__(thick=thick, chord=chord, twist=twist, lTwistAroundLE=lTwistAroundLE)

        self.tail = tail
        self.RLE = 0.0
        
        self.te_angle = 0.0     # trailing edge angle (degree)
        self.te_slope = 0.0     # slope of the mean camber line at trailing edge (dy/dx)

        #* 2D unit airfoil
        self.cst_u = np.zeros(1)
        self.cst_l = np.zeros(1)

        #* Refine airfoil
        self.refine_u = None
        self.refine_l = None

    def section(self, cst_u=None, cst_l=None, nn=1001, flip_x=False, projection=True) -> None:
        '''
        Generating the section (3D) by cst_foil. 
        First construct the 2D unit curve, then transform it to the 3D curve.

        Parameters
        ------------
        nn : int
            number of points in `xx`, `yy`, `yu`, and `yl`. 
        cst_u, cst_l : ndarray
            CST coefficients of upper and lower surfaces (optional)
        flip_x : bool
            whether flip `xx` in the reverse order, by default False.
        projection : bool
            whether keeps the projection length the same when rotating the section, by default True.
        
        Examples
        ------------
        >>> sec.section(cst_u=None, cst_l=None, nn=1001, flip_x=False, projection=True)
        '''
        #* Update CST parameters
        if isinstance(cst_u, np.ndarray) and isinstance(cst_l, np.ndarray):
            self.cst_u = cst_u.copy()
            self.cst_l = cst_l.copy()

        #* Construct airfoil with CST parameters
        self.xx, self.yu, self.yl, self.thick, self.RLE = cst_foil(
            nn, self.cst_u, self.cst_l, t=self.thick_set, tail=self.tail)
        
        #* Trailing edge information
        a1 = [self.xx[-1]-self.xx[-5], self.yu[-1]-self.yu[-5]]
        a2 = [self.xx[-1]-self.xx[-5], self.yl[-1]-self.yl[-5]]
        self.te_angle = np.arccos(np.dot(a1,a2)/np.linalg.norm(a1)/np.linalg.norm(a2))/np.pi*180.0
        self.te_slope = 0.5*((self.yu[-1]+self.yl[-1])-(self.yu[-5]+self.yl[-5]))/(self.xx[-1]-self.xx[-5])

        #* Refine the airfoil by incremental curves
        yu_i = np.zeros(nn)
        yl_i = np.zeros(nn)

        if isinstance(self.refine_u, np.ndarray):
            _, y_tmp = cst_curve(nn, self.refine_u, x=self.xx)
            yu_i += y_tmp

        if isinstance(self.refine_l, np.ndarray):
            _, y_tmp = cst_curve(nn, self.refine_l, x=self.xx)
            yl_i += y_tmp

        self.yu, self.yl = foil_increment_curve(self.xx, self.yu, self.yl, yu_i=yu_i, yl_i=yl_i, t=self.thick_set)

        #* Transform to 3D
        super().section(flip_x=flip_x, projection=projection)


class OpenSection(BasicSection):
    '''
    Section 3D curve generated by CST curve (open curve)
    
    Examples
    --------
    >>> sec = OpenSection(thick=None, chord=1.0, twist=0.0, lTwistAroundLE=True)
    
    Attributes
    ------------
    cst : ndarray
        cst coefficients of the curve
    refine: {None, ndarray}
        cst coefficients of the refinement curve 
    '''
    def __init__(self, thick=None, chord=1.0, twist=0.0, lTwistAroundLE=True):

        super().__init__(thick=thick, chord=chord, twist=twist, lTwistAroundLE=lTwistAroundLE)

        #* 2D unit curve
        self.cst = np.zeros(1)

        #* Refine airfoil
        self.refine = None

    def section(self, cst=None, nn=1001, flip_x=False, projection=True):
        '''
        Generating the section (3D) by cst_foil. 
        First construct the 2D unit curve, then transform it to the 3D curve.

        Parameters
        ------------
        nn : int
            number of points in `xx`, `yy`, `yu`, and `yl`. 
        cst : ndarray
            CST coefficients of upper and lower surfaces (optional)
        flip_x : bool
            whether flip `xx` in the reverse order, by default False.
        projection : bool
            whether keeps the projection length the same when rotating the section, by default True.
        
        Examples
        ------------
        >>> sec.section(cst=None, nn=1001, flip_x=False, projection=True)
        '''
        #* Update CST parameters
        if isinstance(cst, np.ndarray):
            self.cst = cst.copy()

        #* Construct curve with CST parameters
        self.xx, self.yy = cst_curve(nn, self.cst)

        #* Refine the geometry with an incremental curve
        if isinstance(self.refine, np.ndarray):
            _, y_i = cst_curve(nn, self.refine, x=self.xx)
            self.yy += y_i

        #* Apply thickness
        self.thick = np.max(self.yy, axis=0)
        if isinstance(self.thick_set, float):
            self.yy = self.yy/self.thick*self.thick_set
            self.thick = self.thick_set

        #* Transform to 3D
        super().section(flip_x=flip_x, projection=projection)


class RoundTipSection(BasicSection):
    '''
    Section 3D curve generated by CST foil and base shape (upper & lower surface)
    
    Suitable for round trailing edge foils or blades, or plates.
    
    Parameters
    -----------
    xLE, yLE, zLE:  float,
        coordinates of the leading edge
    chord: float
        chord length (m)
    thick: float
        maximum relative thickness
    twist: float
        twist angle (deg)
    tail: float
        actual thickness of TE (m)
    cst_u, cst_l : ndarray
        cst coefficients of the upper and lower surfaces
    base_le_radius: float
        relative radius of base shape function leading edge
    base_te_radius: float
        relative radius of base shape function trailing edge
    base_abs_thick: float
        actual thickness of the base shape
    base_le_ratio: float
        ratio of the leading  edge region
    base_te_ratio: float
        ratio of the trailing edge region
    aLE: float
        angle (deg) of the slope at leading  edge (a>0 => dy/dx>0)
    aTE: float
        angle (deg) of the slope at trailing edge (a<0 => dy/dx<0)
    i_split: {None, int}
        active when leading edge and trailing edge curves are intersected
    nn : int
        number of points in `xx`, `yy`, `yu`, and `yl`. 
    lTwistAroundLE : bool
        whether the twist center is LE, otherwise TE, by default True.
    
    '''
    def __init__(self, xLE: float, yLE: float, zLE: float, 
                chord: float, thick: float, twist: float, tail: float,
                cst_u: np.ndarray, cst_l: np.ndarray,
                base_le_ratio: float, base_te_ratio: float, base_abs_thick: float, 
                base_le_radius: float, base_te_radius: float,
                aLE=0.0, aTE=0.0, i_split=None, nn=501, lTwistAroundLE=False):

        super().__init__(thick=thick, chord=chord, twist=twist, lTwistAroundLE=lTwistAroundLE)
        
        self.xLE = xLE
        self.yLE = yLE
        self.zLE = zLE
        self.thick = thick
        self.tail = tail
        
        #* --------------------------------------------------
        #* Base shape (actual chord length)
        x_ref   = dist_clustcos(nn, a0=0.0079, a1=0.96, beta=2)
        
        if base_abs_thick > 0.0:
            
            x_, y_ = RoundTipSection.base_shape(x_ref, xLE, xLE+chord, base_le_ratio*chord, base_te_ratio*chord, 
                                                base_le_radius, base_te_radius, base_abs_thick/2.0, i_split=i_split)
            dy_    = RoundTipSection.base_camber(x_, a_LE=aLE, a_TE=aTE)

            # Scale to unit chord length
            self.xx = (x_ - xLE)/chord
            self.base_yu = (dy_ + y_)/chord
            self.base_yl = (dy_ - y_)/chord
        
        else:
            
            dy_ = RoundTipSection.base_camber(xLE+x_ref*chord, a_LE=aLE, a_TE=aTE)
            
            self.xx = x_ref
            self.base_yu = dy_/chord
            self.base_yl = dy_/chord
        
        #* Base shape thickness check
        base_tmax = np.max(self.base_yu-self.base_yl)
        if base_tmax > thick:
            print('Warning: base shape is thicker than tmax (%.3f > %.3f).'%(base_tmax*chord, thick*chord))
            print('         The base shape is scaled to tmax.')
            self.base_yu = dy_ + y_*thick/base_tmax
            self.base_yl = dy_ - y_*thick/base_tmax
        
        if base_tmax > tail/chord and tail>0.0:
            print('Warning: base shape is thicker than the specified tail thickness (%.3f > %.3f).'%(base_tmax*chord, tail))
            print('         The final tail thickness is about the base thickness.')
            
        #* --------------------------------------------------
        #* CST shape (actual chord length)
        self.cst_yu = np.zeros_like(self.xx)
        self.cst_yl = np.zeros_like(self.xx)
        
        if np.max(np.abs(cst_u))>1E-6 or np.max(np.abs(cst_l))>1E-6 > 0:
            
            self.cst_u = cst_u.copy()
            self.cst_l = cst_l.copy()
            
            cst_tail = max(0, tail/chord-base_tmax)
            
            _, self.cst_yu, self.cst_yl, _, _ = cst_foil(self.xx.shape[0], cst_u, cst_l,
                    x=self.xx, t=thick-base_tmax, tail=cst_tail)

        #* --------------------------------------------------
        #* Unit chord shape
        self.yu = self.base_yu + self.cst_yu
        self.yl = self.base_yl + self.cst_yl
        
        #* Calculate leading edge radius
        x_RLE = 0.005
        yu_RLE = interp_from_curve(x_RLE, self.xx, self.yu)
        yl_RLE = interp_from_curve(x_RLE, self.xx, self.yl)
        self.RLE, _ = find_circle_3p([0.0,0.0], [x_RLE,yu_RLE], [x_RLE,yl_RLE])

        
    @staticmethod
    def base_shape(x_ref: np.ndarray, x_LE: float, x_TE: float, l_LE: float, l_TE: float, 
                   r_LE: float, r_TE: float, h: float, i_split=None) -> Tuple[np.ndarray, np.ndarray]:
        '''
        Base shape function of wing sections.

        Parameters
        ------------
        x_ref: ndarray [nn]
            reference point distribution in [0,1]
        x_LE: float
            leading edge location
        x_TE: float
            trailing edge location
        l_LE: float
            length of leading  edge curve/ramp
        l_TE: float
            length of trailing edge curve/ramp
        r_LE: float
            relative radius of leading  edge
        r_TE: float
            relative radius of trailing edge
        h: float
            height
        i_split: {None, int}
            active when leading edge and trailing edge curves are intersected
        
        Notes
        -------
        Base shape: \n
        `_______________________________________________`   \n
        `(______________________________________________)`  \n
        
        Examples
        ---------
        >>> x, y = base_shape(x_ref, x_LE, x_TE, l_LE, l_TE, r_LE, r_TE, h, i_split=None)
        
        '''
        
        l0 = x_TE - x_LE
        
        if abs(l0)<=1e-10:
            return np.ones_like(x_ref)*x_LE, np.zeros_like(x_ref)
        
        x = x_ref*l0+x_LE
        y = np.ones_like(x)*h
        
        ratio_LE = l_LE/l0
        ratio_TE = l_TE/l0
        

        if l_LE+l_TE<=l0:
            
            i_LE = np.argmin(np.abs(x_ref-ratio_LE))+1
            i_TE = np.argmin(np.abs(x_ref-1+ratio_TE))-1
            y[:i_LE] = RoundTipSection.general_eqn(x_ref[:i_LE],   ratio_LE, r_LE, h)
            y[i_TE:] = RoundTipSection.general_eqn(1-x_ref[i_TE:], ratio_TE, r_TE, h)
            
            return x, y
            
        else:
            
            print('Warning: the specified length of LE and TE region %.2f > chord length %2f.'%(l_LE+l_TE, l0))
            print('         The LE and TE curves will intersect.')
            
            y_le = RoundTipSection.general_eqn(x_ref,   ratio_LE, r_LE, h)
            y_te = RoundTipSection.general_eqn(1-x_ref, ratio_TE, r_TE, h)
            i_IT = np.argmin(np.abs(y_le-y_te))

            #* Locate intersection point
            x_m = x_ref[i_IT]
            x_l = x_ref[i_IT-1] # y_le(x_l) < y_te(x_l)
            x_r = x_ref[i_IT+1] # y_le(x_r) > y_te(x_r)
            for _ in range(10):
                x_m = 0.5*(x_l+x_r)
                d_m = RoundTipSection.general_eqn(np.array([x_m]),   ratio_LE, r_LE, h) \
                    - RoundTipSection.general_eqn(np.array([1-x_m]), ratio_TE, r_TE, h)
                if d_m < -1e-10:
                    x_l = x_m
                elif d_m > 1e-10:
                    x_r = x_m
                else:
                    break
            
            y_m = RoundTipSection.general_eqn(np.array([x_m]), ratio_LE, r_LE, h)[0]
            
            if i_split == None:
                
                y = np.concatenate((y_le[:i_IT], y_te[i_IT:]))
                y[i_IT] = y_m
                
                return x, y
                
            else:
                
                print('         The intersect point is specified by i_split = %d'%(i_split))
                
                nn = x_ref.shape[0]
                x_le = dist_clustcos(i_split,      a0=0.01, a1=0.96, beta=2)*x_m
                x_te = dist_clustcos(nn-i_split+1, a0=0.05, a1=0.96)*(1.0-x_m)+x_m
                y_le = RoundTipSection.general_eqn(x_le,   ratio_LE, r_LE, h)
                y_te = RoundTipSection.general_eqn(1-x_te, ratio_TE, r_TE, h)
                
                xx = np.concatenate((x_le[:, np.newaxis], x_te[1:, np.newaxis]), axis=0)
                yy = np.concatenate((y_le[:, np.newaxis], y_te[1:, np.newaxis]), axis=0)
                xx = xx[:,0]*l0+x_LE
                yy = yy[:,0]
                xx[i_split-1] = x_m*l0+x_LE
                yy[i_split-1] = y_m

                return xx, yy

    @staticmethod
    def general_eqn(x: np.ndarray, l: float, rr: float, h: float) -> np.ndarray:
        '''
        General equations to define the leading edge semi-thickness, 
        the flat plate semi-thickness, the trailing edge closure semi-thickness,
        and the transverse radius of the sting fairing.

        Experimental Surface Pressure Data Obtained on 65° Delta Wing Across Reynolds Number
        and Mach Number Ranges (Volume 2—Small-Radius Leading Edges)

        https://ntrs.nasa.gov/archive/nasa/casi.ntrs.nasa.gov/19960025648.pdf


        Parameters
        ------------
        x: ndarray
            current location
        l: float
            range of x
        rr: float
            relative radius
        h: float
            height

        Examples
        -----------
        >>> phi = general_eqn(x, l, rr, t) # phi >= 0

        '''
        a = np.sqrt(2*rr*h)
        b = -15/8.*a + 3*h
        c = 5/4.*a - 3*h
        d = -3/8.*a + h
        
        xi  = x/l
        
        for i in range(xi.shape[0]):
            xi[i] = max(0.0, min(1.0, xi[i]))

        phi = a*np.sqrt(xi)+b*xi+c*np.power(xi,2)+d*np.power(xi,3)

        return phi

    @staticmethod
    def base_camber(x: np.ndarray, a_LE=0.0, a_TE=0.0) -> np.ndarray:
        '''
        Camber curve of the base shape function (in 3rd order spline)
        
        Parameters
        -----------
        x: ndarray
            actual x distribution
        a_LE: float
            angle (deg) of the slope at leading  edge (a>0 => dy/dx>0)
        a_TE: float
            angle (deg) of the slope at trailing edge (a<0 => dy/dx<0)

        Examples
        ----------
        >>> dy = base_camber(x, a_LE, a_TE)
        
        '''
        chord = x[-1]-x[0]
        x_ref = (x-x[0])/chord  # reference point distribution in [0,1]
        dy_LE = np.tan(a_LE/180.0*np.pi)*chord
        dy_TE = np.tan(a_TE/180.0*np.pi)*chord
        a  = dy_LE + dy_TE
        b  = -2*dy_LE-dy_TE
        c  = dy_LE
        dy = a*x_ref**3+b*x_ref**2+c*x_ref
        return dy


#* ===========================================
#* Foil functions
#* ===========================================

def foil_tcc(x: np.ndarray, yu: np.ndarray, yl: np.ndarray, info=True):
    '''
    Calculate thickness, curvature, camber distribution.
    
    Parameters
    ----------
    x, yu, yl: ndarray
        coordinates of the airfoil
    info: bool
        whether print warning information
    
    Returns
    --------
    thickness: ndarray
        thickness distribution
    curv_u, curv_l: ndarray
        curvature distribution
    camber: ndarray
        camber distribution

    Examples
    ----------
    >>> thickness, curv_u, curv_l, camber = foil_tcc(x, yu, yl, info=True)

    '''
    curv_u = curve_curvature(x, yu)
    curv_l = curve_curvature(x, yl)

    thickness = yu-yl
    camber = 0.5*(yu+yl)
    for i in range(x.shape[0]):
        if info and thickness[i]<0:
            print('Unreasonable Airfoil: negative thickness')

    return thickness, curv_u, curv_l, camber

def check_valid(x: np.ndarray, yu: np.ndarray, yl: np.ndarray, RLE=0.0, neg_t_cri=0.0) -> list:
    '''
    Check if the airfoil is reasonable by rules
    
    Parameters
    ----------
    x, yu, yl: ndarray
        coordinates of the airfoil
    RLE: float 
        the leading edge radius of this airfoil
    neg_t_cri: float
        critical value for checking negative thickness,
        e.g., neg_t_cri = -0.01, then only invalid when the thickness is smaller than -0.01
        
    Returns
    ---------
    rule_invalid: list
        0 means valid

    Rules
    ------------
    - 1. negative thickness
    - 2. maximum thickness point location
    - 3. extreme points of thickness
    - 4. maximum curvature
    - 5. maximum camber within x [0.2,0.7]
    - 6. RLE if provided
    - 7. convex LE

    Examples
    ---------
    >>> rule_invalid = check_valid(x, yu, yl, RLE=0.0)

    '''
    thickness, curv_u, curv_l, camber = foil_tcc(x, yu, yl, info=False)
    nn = x.shape[0]

    n_rule = 10
    rule_invalid = [0 for _ in range(n_rule)]

    #* Rule 1: negative thickness
    if np.min(thickness) < neg_t_cri:
        rule_invalid[0] = 1

    #* Rule 2: maximum thickness point location
    i_max = np.argmax(thickness)
    t0    = thickness[i_max]
    x_max = x[i_max]
    if x_max<0.15 or x_max>0.75:
        rule_invalid[1] = 1

    #* Rule 3: extreme points of thickness
    n_extreme = 0
    for i in range(nn-2):
        a1 = thickness[i+2]-thickness[i+1]
        a2 = thickness[i]-thickness[i+1]
        if a1*a2>=0.0:
            n_extreme += 1
    if n_extreme>2:
        rule_invalid[2] = 1

    #* Rule 4: maximum curvature
    cur_max_u = 0.0
    cur_max_l = 0.0
    for i in range(nn):
        if x[i]<0.1:
            continue
        cur_max_u = max(cur_max_u, abs(curv_u[i]))
        cur_max_l = max(cur_max_l, abs(curv_l[i]))

    if cur_max_u>5 or cur_max_l>5:
        rule_invalid[3] = 1

    #* Rule 5: Maximum camber within x [0.2,0.7]
    cam_max = 0.0
    for i in range(nn):
        if x[i]<0.2 or x[i]>0.7:
            continue
        cam_max = max(cam_max, abs(camber[i]))

    if cam_max>0.025:
        rule_invalid[4] = 1
    
    #* Rule 6: RLE
    if RLE>0.0 and RLE<0.005:
        rule_invalid[5] = 1

    if RLE>0.0 and RLE/t0<0.01:
        rule_invalid[5] = 1

    #* Rule 7: convex LE
    ii = int(0.1*nn)+1
    a0 = thickness[i_max]/x[i_max]
    au = yu[ii]/x[ii]/a0
    al = -yl[ii]/x[ii]/a0

    if au<1.0 or al<1.0:
        rule_invalid[6] = 1
    
    #if sum(rule_invalid)<=0:
    #    np.set_printoptions(formatter={'float': '{: 0.6f}'.format}, linewidth=100)
    #    print(np.array([x_max, n_extreme, cur_max_u, cur_max_l, cam_max, RLE/t0, au, al]))

    return rule_invalid

def normalize_foil(xu: np.ndarray, yu: np.ndarray, xl: np.ndarray, yl: np.ndarray):
    '''
    Transform the airfoil to a unit airfoil.

    Parameters
    ----------
    xu, yu, xl, yl: ndarray
        coordinates of the airfoil

    Returns
    --------
    xu_, yu_, xl_, yl_: ndarray
        coordinates of the airfoil
    twist: float
        twist angle (degree)
    chord: float
        chord length
    tail: float
        tail height relative to chord length

    Examples
    ---------
    >>> xu_, yu_, xl_, yl_, twist, chord, tail = normalize_foil(xu, yu, xl, yl)

    '''
    if abs(xu[0]-xl[0])>1e-6 or abs(yu[0]-yl[0])>1e-6:
        raise Exception('Two curves do not have the same leading edge')
    
    #* Transform
    xu_ = xu - xu[0]
    xl_ = xl - xl[0]
    yu_ = yu - yu[0]
    yl_ = yl - yl[0]

    #* Twist
    xTE   = 0.5*(xu_[-1]+xl_[-1])
    yTE   = 0.5*(yu_[-1]+yl_[-1])
    twist = np.arctan(yTE/xTE)*180/np.pi
    chord = np.sqrt(xTE**2+yTE**2)

    xu_, yu_, _ = rotate(xu_, yu_, np.zeros_like(xu_), angle=-twist, axis='Z')
    xl_, yl_, _ = rotate(xl_, yl_, np.zeros_like(xu_), angle=-twist, axis='Z')

    #* Scale
    yu_ = yu_ / xu_[-1]
    yl_ = yl_ / xl_[-1]
    xu_ = xu_ / xu_[-1]
    xl_ = xl_ / xl_[-1]
    
    #* Removing tail
    tail = abs(yu_[-1]) + abs(yl_[-1])

    for ip in range(yu_.shape[0]):
        yu_[ip] -= xu_[ip]*yu_[-1]  

    for ip in range(yl_.shape[0]):
        yl_[ip] -= xl_[ip]*yl_[-1]  

    return xu_, yu_, xl_, yl_, twist, chord, tail

def find_circle_3p(p1, p2, p3) -> Tuple[float, np.ndarray]:
    '''
    Determine the radius and origin of a circle by 3 points (2D)
    
    Parameters
    -----------
    p1, p2, p3: list or ndarray [2]
        coordinates of points, [x, y]
        
    Returns
    ----------
    R: float
        radius
    XC: ndarray [2]
        circle center

    Examples
    ----------
    >>> R, XC = find_circle_3p(p1, p2, p3)

    '''

    # http://ambrsoft.com/TrigoCalc/Circle3D.htm

    A = p1[0]*(p2[1]-p3[1]) - p1[1]*(p2[0]-p3[0]) + p2[0]*p3[1] - p3[0]*p2[1]
    if np.abs(A) <= 1E-20:
        raise Exception('Finding circle: 3 points in one line')
    
    p1s = p1[0]**2 + p1[1]**2
    p2s = p2[0]**2 + p2[1]**2
    p3s = p3[0]**2 + p3[1]**2

    B = p1s*(p3[1]-p2[1]) + p2s*(p1[1]-p3[1]) + p3s*(p2[1]-p1[1])
    C = p1s*(p2[0]-p3[0]) + p2s*(p3[0]-p1[0]) + p3s*(p1[0]-p2[0])
    D = p1s*(p3[0]*p2[1]-p2[0]*p3[1]) + p2s*(p1[0]*p3[1]-p3[0]*p1[1]) + p3s*(p2[0]*p1[1]-p1[0]*p2[1])

    x0 = -B/2/A
    y0 = -C/2/A
    R  = np.sqrt(B**2+C**2-4*A*D)/2/np.abs(A)

    '''
    x21 = p2[0] - p1[0]
    y21 = p2[1] - p1[1]
    x32 = p3[0] - p2[0]
    y32 = p3[1] - p2[1]

    if x21 * y32 - x32 * y21 == 0:
        raise Exception('Finding circle: 3 points in one line')

    xy21 = p2[0]*p2[0] - p1[0]*p1[0] + p2[1]*p2[1] - p1[1]*p1[1]
    xy32 = p3[0]*p3[0] - p2[0]*p2[0] + p3[1]*p3[1] - p2[1]*p2[1]
    
    y0 = (x32 * xy21 - x21 * xy32) / 2 * (y21 * x32 - y32 * x21)
    x0 = (xy21 - 2 * y0 * y21) / (2.0 * x21)
    R = np.sqrt(np.power(p1[0]-x0,2) + np.power(p1[1]-y0,2))
    '''

    return R, np.array([x0, y0])


#* ===========================================
#* CST foils
#* ===========================================

def cst_foil(nn: int, cst_u, cst_l, x=None, t=None, tail=0.0, xn1=0.5, xn2=1.0):
    '''
    Constructing upper and lower curves of an airfoil based on CST method

    CST: class shape transformation method (Kulfan, 2008)
    
    Parameters
    -----------
    nn: int
        total amount of points
    cst_u, cst_l: list or ndarray
        CST coefficients of the upper and lower surfaces
    x: ndarray [nn]
        x coordinates in [0,1] (optional)
    t: float
        specified relative maximum thickness (optional)
    tail: float
        relative tail thickness (optional)
    xn1, xn12: float
        CST parameters
        
    Returns
    --------
    x, yu, yl: ndarray
        coordinates
    t0: float
        actual relative maximum thickness
    R0: float
        leading edge radius
    
    Examples
    ---------
    >>> x_, yu, yl, t0, R0 = cst_foil(nn, cst_u, cst_l, x, t, tail)

    '''
    cst_u = np.array(cst_u)
    cst_l = np.array(cst_l)
    x_, yu = cst_curve(nn, cst_u, x=x, xn1=xn1, xn2=xn2)
    x_, yl = cst_curve(nn, cst_l, x=x, xn1=xn1, xn2=xn2)
    
    thick = yu-yl
    it = np.argmax(thick)
    t0 = thick[it]

    # Apply thickness constraint
    if t is not None:
        r  = (t-tail*x_[it])/t0
        t0 = t
        yu = yu * r
        yl = yl * r

    # Add tail
    for i in range(nn):
        yu[i] += 0.5*tail*x_[i]
        yl[i] -= 0.5*tail*x_[i]
        
    # Update t0 after adding tail
    if t is None:
        thick = yu-yl
        it = np.argmax(thick)
        t0 = thick[it]

    # Calculate leading edge radius
    x_RLE = 0.005
    yu_RLE = interp_from_curve(x_RLE, x_, yu)
    yl_RLE = interp_from_curve(x_RLE, x_, yl)
    R0, _ = find_circle_3p([0.0,0.0], [x_RLE,yu_RLE], [x_RLE,yl_RLE])

    return x_, yu, yl, t0, R0

def scale_cst(x: np.ndarray, yu: np.ndarray, yl: np.ndarray, cst_u, cst_l, t: float, tail=0.0):
    '''
    Scale CST coefficients, so that the airfoil has the maximum thickness of t. 

    Parameters
    -----------
    x, yu, yl: ndarray [nn]
        baseline airfoil. `x`, `yu`, `yl` must be directly generated by CST, without scaling.
    cst_u, cst_l: list or ndarray
        CST coefficients of the upper and lower surfaces
    t: float
        specified relative maximum thickness
    tail: float
        relative tail thickness (optional)

    Examples
    ---------
    >>> cst_u_new, cst_l_new = scale_cst(yu, yl, cst_u, cst_l, t)

    '''

    thick = yu - yl
    it = np.argmax(thick)
    t0 = thick[it]

    r  = (t-tail*x[it])/t0
    cst_u_new = np.array(cst_u) * r
    cst_l_new = np.array(cst_l) * r

    return cst_u_new, cst_l_new

def clustcos(i: int, nn: int, a0=0.0079, a1=0.96, beta=1.0) -> float:
    '''
    Point distribution on x-axis [0, 1]. (More points at both ends)
    
    Parameters
    ----------
    i: int
        index of current point (start from 0)
    nn: int
        total amount of points
    a0: float
        parameter for distributing points near x=0
    a1: float
        parameter for distributing points near x=1
    beta: float
        parameter for distribution points 

    Returns
    ---------
    float

    Examples
    ---------
    >>> c = clustcos(i, n, a0, a1, beta)

    '''
    aa = np.power((1-np.cos(a0*np.pi))/2.0, beta)
    dd = np.power((1-np.cos(a1*np.pi))/2.0, beta) - aa
    yt = i/(nn-1.0)
    a  = np.pi*(a0*(1-yt)+a1*yt)
    c  = (np.power((1-np.cos(a))/2.0,beta)-aa)/dd

    return c

def dist_clustcos(nn: int, a0=0.0079, a1=0.96, beta=1.0) -> np.ndarray:
    '''
    Point distribution on x-axis [0, 1]. (More points at both ends)

    Parameters
    ----------
    nn: int
        total amount of points
    a0: float
        parameter for distributing points near x=0
    a1: float
        parameter for distributing points near x=1
    beta: float
        parameter for distribution points 
    
    Examples
    ---------
    >>> xx = dist_clustcos(n, a0, a1, beta)

    '''
    aa = np.power((1-np.cos(a0*np.pi))/2.0, beta)
    dd = np.power((1-np.cos(a1*np.pi))/2.0, beta) - aa
    yt = np.linspace(0.0, 1.0, num=nn)
    a  = np.pi*(a0*(1-yt)+a1*yt)
    xx = (np.power((1-np.cos(a))/2.0,beta)-aa)/dd

    return xx

def cst_curve(nn: int, coef: np.array, x=None, xn1=0.5, xn2=1.0) -> Tuple[np.ndarray, np.ndarray]:
    '''
    Generating single curve based on CST method.

    CST: class shape transformation method (Kulfan, 2008)

    Parameters
    ----------
    nn: int
        total amount of points
    coef: ndarray
        CST coefficients
    x: ndarray [nn]
        coordinates of x distribution in [0,1] (optional)
    xn1, xn12: float
        CST parameters
    
    Returns
    --------
    x, y: ndarray
        coordinates
    
    Examples
    ---------
    >>> x, y = cst_curve(nn, coef, x, xn1, xn2)

    '''
    if x is None:
        x = np.zeros(nn)
        for i in range(nn):
            x[i] = clustcos(i, nn)
    elif x.shape[0] != nn:
        raise Exception('Specified point distribution has different size %d as input nn %d'%(x.shape[0], nn))
    
    n_cst = coef.shape[0]
    y = np.zeros(nn)
    for ip in range(nn):
        s_psi = 0.0
        for i in range(n_cst):
            xk_i_n = factorial(n_cst-1)/factorial(i)/factorial(n_cst-1-i)
            s_psi += coef[i]*xk_i_n * np.power(x[ip],i) * np.power(1-x[ip],n_cst-1-i)

        C_n1n2 = np.power(x[ip],xn1) * np.power(1-x[ip],xn2)
        y[ip] = C_n1n2*s_psi

    y[0] = 0.0
    y[-1] = 0.0

    return x, y


#* ===========================================
#* Fitting a curve/foil with CST
#* ===========================================

def cst_foil_fit(xu: np.ndarray, yu: np.ndarray, xl: np.ndarray, yl: np.ndarray,
                n_cst=7, xn1=0.5, xn2=1.0) -> Tuple[np.ndarray, np.ndarray]:
    '''
    Using CST method to fit an airfoil

    Parameters
    ----------
    xu, yu, xl, yl: ndarray
        coordinates
    n_cst: int
        number of CST coefficients
    xn1, xn2: float
        CST parameters
        
    Returns
    -------
    cst_u, cst_l: ndarray
        CST coefficients

    Examples
    ---------
    >>> cst_u, cst_l = cst_foil_fit(xu, yu, xl, yl, n_cst=7, xn1=0.5, xn2=1.0)

    Notes
    -----
    This function allows the airfoil has non-zero tail thickness.
    Also allows the airfoil chord length not equals to one.
    But yu[0] yl[0] should be 0.

    '''
    cst_u = fit_curve(xu, yu, n_cst=n_cst, xn1=xn1, xn2=xn2)
    cst_l = fit_curve(xl, yl, n_cst=n_cst, xn1=xn1, xn2=xn2)
    return cst_u, cst_l

def fit_curve(x: np.ndarray, y: np.ndarray, n_cst=7, xn1=0.5, xn2=1.0):
    '''
    Using least square method to fit a CST curve.
    
    Parameters
    ----------
    x, y: ndarray
        coordinates
    n_cst: int
        number of CST coefficients
    xn1, xn2: float
        CST parameters
        
    Returns
    -------
    coef: ndarray [n_cst]
        CST coefficients
        
    Notes
    -----
    y[0] should be 0.

    Examples
    ---------
    >>> coef = fit_curve(x, y, n_cst=7, xn1=0.5, xn2=1.0)

    '''
    # Array A: A[nn, n_cst], nn=len(x).
    # Array b: b[nn].

    nn = x.shape[0]
    L  = x[-1] - x[0]   # type: float
    x_ = (x-x[0])/L     # scaling x to 0~1
    y_ = (y-y[0])/L     # scaling according to L #! This means y[0] should be 0
    b  = y_.copy()

    for ip in range(nn):
        b[ip] -= x_[ip]*y_[-1]  # removing tail

    A = np.zeros((nn, n_cst))
    for ip in range(nn):
        C_n1n2 = np.power(x_[ip],xn1) * np.power(1-x_[ip],xn2)
        for i in range(n_cst):
            xk_i_n = factorial(n_cst-1)/factorial(i)/factorial(n_cst-1-i)
            A[ip][i] = xk_i_n * np.power(x_[ip],i) * np.power(1-x_[ip],n_cst-1-i) * C_n1n2

    solution = lstsq(A, b, rcond=None)

    return solution[0]

def fit_curve_with_twist(x: np.ndarray, y: np.ndarray, n_cst=7, 
                         xn1=0.5, xn2=1.0) -> Tuple[np.ndarray, float, float, float]:
    '''
    Using least square method to fit a CST curve

    Parameters
    ----------
    x, y: ndarray
        coordinates
    n_cst: int
        number of CST coefficients
    xn1, xn2: float
        CST parameters

    Returns
    -------
    coef: ndarray [n_cst]
        CST coefficients 
    chord: float
        distance between two ends of the curve
    twist: float
        degree, +z axis
    thick: float
        maximum relative thickness

    Examples
    ---------
    >>> coef, chord, twist, thick = fit_curve_with_twist(x, y, n_cst, xn1, xn2)

    '''   
    chord = np.sqrt((x[0]-x[-1])**2+(y[0]-y[-1])**2)
    twist = np.arctan((y[-1]-y[0])/(x[-1]-x[0]))*180/np.pi

    x_ = (x - x[0])/chord
    y_ = (y - y[0])/chord
    x_, y_, _ = rotate(x_, y_, np.zeros_like(x_), angle=-twist, axis='Z')
    thick = np.max(y_, axis=0)

    coef = fit_curve(x_, y_, n_cst=n_cst, xn1=xn1, xn2=xn2)
    
    return coef, chord, twist, thick

def fit_curve_partial(x: np.ndarray, y: np.ndarray, n_cst=7, ip0=0, ip1=0,
            ic0=0, ic1=0, xn1=0.5, xn2=1.0):
    '''
    Using least square method to fit a part of a unit curve

    Parameters
    ----------
    x, y: ndarray
        coordinates
    n_cst: int
        number of CST coefficients
    ip0, ip1: int
        index of the partial curve x[ip0:ip1] 
    ic0, ic1: int
        index of the CST parameters cst[ic0:ic1] that are not 0
    xn1, xn2: float
        CST parameters
        
    Returns
    -------
    coef: ndarray [n_cst]
        CST coefficients

    Examples
    ---------
    >>> coef = fit_curve_partial(x: np.array, y: np.array, ip0=0, ip1=0, n_cst=7, xn1=0.5, xn2=1.0)

    '''
    # Array A: A[nn, n_cst], nn=len(x).
    # Array b: b[nn].
    
    ip0 = max(0, ip0)
    if ip1 <= ip0:
        ip1 = x.shape[0]

    ic0 = max(0, ic0)
    if ic1 <= ic0:
        ic1 = n_cst

    #* Fit the partial curve
    A = np.zeros((ip1-ip0, ic1-ic0))
    for ip in range(ip0, ip1):
        C_n1n2 = np.power(x[ip],xn1) * np.power(1-x[ip],xn2)
        for i in range(ic0,ic1):
            xk_i_n = factorial(n_cst-1)/factorial(i)/factorial(n_cst-1-i)
            A[ip-ip0][i-ic0] = xk_i_n * np.power(x[ip],i) * np.power(1-x[ip],n_cst-1-i) * C_n1n2

    solution = lstsq(A, y[ip0:ip1], rcond=None)
    
    coef = np.zeros(n_cst)
    for i in range(ic0, ic1):
        coef[i] = solution[0][i-ic0]

    return coef


#* ===========================================
#* Modification of a curve/foil
#* ===========================================

def foil_bump_modify(x: np.ndarray, yu: np.ndarray, yl: np.ndarray,
            xc: float, h: float, s: float, side=1, n_cst=0,
            return_cst=False, keep_tmax=True):
    '''
    Add bumps on the airfoil

    Parameters
    ----------
    x, yu, yl: ndarray
        coordinates of the airfoil
    xc: float
        x of the bump center
    h: float
        relative height of the bump (to maximum thickness)
    s: float
        span of the bump
    side: int
        +1/-1 upper/lower side of the airfoil
    n_cst: int 
        if specified (>0), then use CST to fit the new foil
    return_cst: bool
        if True, also return cst_u, cst_l when n_cst > 0
    keep_tmax: bool
        if True, keep the maximum thickness unchanged scale the opposite side of 'side' to keep thickness

    Returns
    -------
    yu_new, yl_new: ndarray
        coordinates
    cst_u, cst_l: ndarray
        CST coefficients
    
    Examples
    ---------
    >>> yu_new, yl_new (, cst_u, cst_l) = foil_bump_modify(
    >>>         x: np.array, yu: np.array, yl: np.array, 
    >>>         xc: float, h: float, s: float, side=1,
    >>>         n_cst=0, return_cst=False, keep_tmax=True)

    '''
    yu_new = yu.copy()
    yl_new = yl.copy()
    t0 = np.max(yu_new-yl_new)

    if xc<0.1 or xc>0.9:
        kind = 'H'
    else:
        kind = 'G'

    if side > 0:
        yu_new = yu_new + bump_function(x, xc, h*t0, s, kind=kind)
    else:
        yl_new = yl_new + bump_function(x, xc, h*t0, s, kind=kind)

    if keep_tmax:

        it = np.argmax(yu_new-yl_new)
        tu = np.abs(yu_new[it])
        tl = np.abs(yl_new[it])

        #* Scale the opposite side
        if side > 0:
            rl = (t0-tu)/tl
            yl_new = rl * np.array(yl_new)
        else:
            ru = (t0-tl)/tu
            yu_new = ru * np.array(yu_new)

        t0 = None

    if n_cst > 0:
        # CST reverse
        tail = yu[-1] - yl[-1]
        cst_u, cst_l = cst_foil_fit(x, yu_new, x, yl_new, n_cst=n_cst)
        _, yu_new, yl_new, _, _ = cst_foil(x.shape[0], cst_u, cst_l, x=x, t=t0, tail=tail)
    else:
        cst_u = None
        cst_l = None
    
    if return_cst:
        return yu_new, yl_new, cst_u, cst_l
    else:
        return yu_new, yl_new

def foil_increment(x, yu, yl, cst_u=None, cst_l=None, t=None) -> Tuple[np.ndarray, np.ndarray]:
    '''
    Add cst curve by incremental curves

    Parameters
    ----------
    x, yu, yl: ndarray
        coordinates of the baseline airfoil
    cst_u, cst_l: {None, ndarray} 
        CST coefficients of incremental upper curve
    t: {None, float}
        relative maximum thickness

    Returns
    -------
    yu_new, yl_new: ndarray
        coordinates

    Examples
    ---------
    >>> yu_new, yl_new = foil_increment(x, yu, yl, cst_u, cst_l, t=None)

    '''
    nn = len(x)

    if cst_u is not None:
        _, yu_i = cst_curve(nn, cst_u, x=x)
    else:
        yu_i = None

    if cst_u is not None:
        _, yl_i = cst_curve(nn, cst_l, x=x)
    else:
        yl_i = None

    yu_new, yl_new = foil_increment_curve(x, yu, yl, yu_i=yu_i, yl_i=yl_i, t=t)

    return yu_new, yl_new

def foil_increment_curve(x: np.ndarray, yu: np.ndarray, yl: np.ndarray,
                         yu_i=None, yl_i=None, t=None) -> Tuple[np.ndarray, np.ndarray]:
    '''
    Add cst curve by incremental curves

    Parameters
    -----------
    x, yu, yl : ndarray
        coordinates of the baseline airfoil.
    yu_i, yl_i : {None, ndarray}
        if not None, coordinates of the incremental curves.
    t : {None, float}
        relative maximum thickness.

    Returns
    ---------
    yu_, yl_ : ndarray
        coordinates of the new airfoil.
        
    Examples
    ---------
    >>> yu_, yl_ = foil_increment_curve(x, yu, yl, yu_i, yl_i, t=None)
    
    '''
    nn = len(x)

    if not isinstance(yu_i, np.ndarray):
        yu_i = np.zeros(nn)

    if not isinstance(yl_i, np.ndarray):
        yl_i = np.zeros(nn)

    x_   = x.copy()
    yu_  = yu.copy()
    yl_  = yl.copy()

    # Remove tail
    tail = yu_[-1] - yl_[-1]
    if tail > 0.0:
        yu_ = yu_ - 0.5*tail*x_
        yl_ = yl_ + 0.5*tail*x_

    # Add incremental curves
    yu_  = yu_ + yu_i
    yl_  = yl_ + yl_i

    thick = yu_-yl_
    it = np.argmax(thick)
    t0 = thick[it]

    # Apply thickness constraint
    if t is not None:
        r  = (t-tail*x[it])/t0
        yu_ = yu_ * r
        yl_ = yl_ * r

    # Add tail
    if tail > 0.0:
        yu_ = yu_ + 0.5*tail*x_
        yl_ = yl_ - 0.5*tail*x_

    return yu_, yl_

def bump_function(x: np.ndarray, xc: float, h: float, s: float, kind='G') -> np.ndarray:
    '''
    A bump distribution [x, y].

    Parameters
    -----------
    x, y: ndarray
        current curve, x in [0,1]
    xc: float
        x of the bump center
    h: float
        height of the bump
    s: float
        span of the bump
    kind: str
        bump function type.
        'G':   Gaussian, less cpu cost
        'H':   Hicks-Henne, better when near leading edge

    Returns
    ---------
    y_bump : ndarray
        coordinates

    Examples
    ---------
    >>> y_bump = bump_function(x, xc, h, s, kind)

    '''
    y_bump = np.zeros_like(x)

    if xc<=0 or xc>=1:
        print('Bump location not valid (0,1): xc = %.3f'%(xc))
        return y_bump

    if 'G' in kind:

        for i in range(x.shape[0]):
            if xc-s<0.0 and x[i]<xc:
                sigma = xc/3.5
            elif  xc+s>1.0 and x[i]>xc:
                sigma = (1.0-xc)/3.5
            else:
                sigma = s/6.0
            aa = -np.power(x[i]-xc,2)/2.0/sigma**2
            y_bump[i] += h*np.exp(aa)

    else:
        
        s0 = np.log(0.5)/np.log(xc) 

        Pow = 1
        span = 1.0
        hm = np.abs(h)
        while Pow<100 and span>s:
            x1  = -1.0
            x2  = -1.0
            for i in range(0, 201):
                xx = i*0.005
                rr = np.pi*np.power(xx,s0)
                yy = hm * np.power(np.sin(rr),Pow)
                if yy > 0.01*hm and x1<0.0 and xx<xc:
                    x1 = xx
                if yy < 0.01*hm and x2<0.0 and xx>xc:
                    x2 = xx
            if x2 < 0.0:
                x2 = 1.0
            
            span = x2 - x1
            Pow = Pow + 1

        for i in range(len(x)):
            rr = np.pi*np.power(x[i],s0)
            dy = h*np.power(np.sin(rr),Pow)
            y_bump[i] += dy

    return y_bump


