"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assertions_1 = require("aws-cdk-lib/assertions");
const cf = require("aws-cdk-lib/aws-cloudfront");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib/");
const lib_1 = require("../lib");
test('Cloudfront Distribution Included', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow'
                })
            ],
        },
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow'
                })
            ],
        },
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Cloudfront With Custom Role', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
        role: new aws_iam_1.Role(stack, 'myRole', { roleName: 'testRole', assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com') })
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.objectLike({
        Role: {
            "Fn::GetAtt": [
                "myRoleE60D68E8",
                "Arn"
            ]
        }
    }));
});
test('Basic Site Setup', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    },
                })
            ],
        },
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    },
                })
            ],
        },
    }));
});
test('Basic Site Setup with Custom Role', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
        role: new aws_iam_1.Role(stack, 'myRole', { roleName: 'testRole', assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com') }),
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.objectLike({
        Role: {
            "Fn::GetAtt": [
                "myRoleE60D68E8",
                "Arn"
            ]
        }
    }));
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    }
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*",
                        "s3:DeleteObject*",
                        "s3:PutObject*",
                        "s3:Abort*"
                    ],
                    Condition: {
                        StringEquals: {
                            "aws:PrincipalArn": {
                                "Fn::GetAtt": [
                                    "myRoleE60D68E8",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    }
                })
            ],
        }
    }));
});
test('Basic Site Setup with Undefined Role', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
        role: undefined
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.objectLike({
        Runtime: "python3.7",
        Role: {
            "Fn::GetAtt": [
                "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265",
                "Arn"
            ]
        }
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    },
                })
            ],
        },
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Cloudfront With Custom Defined Behaviors', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        cfBehaviors: [
            {
                isDefaultBehavior: true,
                allowedMethods: cf.CloudFrontAllowedMethods.ALL,
                forwardedValues: {
                    queryString: true,
                    cookies: { forward: 'all' },
                    headers: ['*'],
                },
            },
            {
                pathPattern: '/virtual-path',
                allowedMethods: cf.CloudFrontAllowedMethods.GET_HEAD,
                cachedMethods: cf.CloudFrontAllowedCachedMethods.GET_HEAD,
            },
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            CacheBehaviors: [
                assertions_1.Match.objectLike({
                    AllowedMethods: ['GET', 'HEAD'],
                    CachedMethods: ['GET', 'HEAD'],
                    PathPattern: '/virtual-path',
                }),
            ],
            DefaultCacheBehavior: {
                AllowedMethods: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
                ForwardedValues: {
                    Cookies: { Forward: 'all' },
                    Headers: ['*'],
                    QueryString: true,
                },
                TargetOriginId: 'origin1',
            },
        },
    }));
});
test('Cloudfront With Custom Security Policy', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
        securityPolicy: cf.SecurityPolicyProtocol.TLS_V1_2_2019,
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
                MinimumProtocolVersion: 'TLSv1.2_2019',
            },
        },
    }));
});
test('Cloudfront With Custom SSL Method', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
        sslMethod: cf.SSLMethod.VIP,
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'vip',
            },
        },
    }));
});
test('Basic Site Setup, IP Filter', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Condition: {
                        IpAddress: {
                            'aws:SourceIp': [
                                '1.1.1.1',
                            ],
                        },
                    },
                    Effect: 'Allow',
                    Principal: {
                        "AWS": "*"
                    },
                })
            ],
        },
    }));
});
test('Create From Hosted Zone', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
    template.hasResourceProperties('AWS::S3::BucketPolicy', assertions_1.Match.objectLike({
        PolicyDocument: {
            Statement: [
                assertions_1.Match.objectLike({
                    Action: 's3:GetObject',
                    Effect: 'Allow'
                })
            ],
        },
    }));
});
test('Create From Hosted Zone with subdomain', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
        subdomain: 'myhost',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'myhost.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Create From Hosted Zone with Custom Role', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
        role: new aws_iam_1.Role(stack, 'myRole', { roleName: 'testRole', assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com') })
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.objectLike({
        Role: {
            "Fn::GetAtt": [
                "myRoleE60D68E8",
                "Arn"
            ]
        }
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    template.hasResource('Custom::CDKBucketDeployment', {});
    template.hasResourceProperties('AWS::CloudFront::Distribution', assertions_1.Match.objectLike({
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, Block Public Enabled', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy')
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL
    });
    const template = assertions_1.Template.fromStack(stack);
    // THEN
    template.hasResourceProperties('AWS::S3::Bucket', assertions_1.Match.objectLike({
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true,
        },
    }));
});
/*
test('Basic Site Setup, URL Output Enabled With Name', () => {
  const stack = new Stack();
  const exportName = 'test-export-name';

  // WHEN
  new SPADeploy(stack, 'spaDeploy', {})
    .createBasicSite({
      indexDoc: 'index.html',
      websiteFolder: 'website',
      exportWebsiteUrlOutput: true,
      exportWebsiteUrlName: exportName,
    });

  const template = Template.fromStack(stack);

  console.log(template.toJSON());

  // THEN
  template.hasOutput(exportName, {});
});

test('Basic Site Setup, URL Output Enabled With No Name', () => {
  const stack = new Stack();
  const exportName = 'test-export-name';

  // WHEN
  new SPADeploy(stack, 'spaDeploy', {})
    .createBasicSite({
      indexDoc: 'index.html',
      websiteFolder: 'website',
      exportWebsiteUrlOutput: true,
    });

  // THEN
  expectCDK(stack).notTo(haveOutput({
    exportName,
  }));
});

test('Basic Site Setup, URL Output Not Enabled', () => {
  const stack = new Stack();
  const exportName = 'test-export-name';

  // WHEN
  new SPADeploy(stack, 'spaDeploy', {})
    .createBasicSite({
      indexDoc: 'index.html',
      websiteFolder: 'website',
      exportWebsiteUrlOutput: false,
    });

  // THEN
  expectCDK(stack).notTo(haveOutput({
    exportName,
  }));
});*/
//# sourceMappingURL=data:application/json;base64,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