"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_route53_patterns_1 = require("aws-cdk-lib/aws-route53-patterns");
const aws_route53_targets_1 = require("aws-cdk-lib/aws-route53-targets");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3deploy = require("aws-cdk-lib/aws-s3-deployment");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
class SPADeploy extends constructs_1.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true || typeof config.blockPublicAccess !== 'undefined') {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                throw new Error('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        //The below "reinforces" the IAM Role's attached policy, it's not required but it allows for customers using permission boundaries to write into the bucket.
        if (config.role) {
            bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    "s3:GetObject*",
                    "s3:GetBucket*",
                    "s3:List*",
                    "s3:DeleteObject*",
                    "s3:PutObject*",
                    "s3:Abort*"
                ],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [bucket.arnForObjects('*'), bucket.bucketArn],
                conditions: {
                    StringEquals: {
                        'aws:PrincipalArn': config.role.roleArn,
                    },
                },
                principals: [new aws_iam_1.AnyPrincipal()]
            }));
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.sslMethod !== 'undefined') {
            cfConfig.aliasConfiguration.sslMethod = config.sslMethod;
        }
        if (typeof config.securityPolicy !== 'undefined') {
            cfConfig.aliasConfiguration.securityPolicy = config.securityPolicy;
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.subdomain ? `${config.subdomain}.${config.zoneName}` : config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website.
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            role: config.role,
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                throw new Error('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new aws_cdk_lib_1.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     It will also setup error forwarding and unauth forwarding back to indexDoc.
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
            role: config.role,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.distributionDomainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     configured by using the details in the hosted zone provided.
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const domainName = config.subdomain ? `${config.subdomain}.${config.zoneName}` : config.zoneName;
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            role: config.role,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: domainName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        if (!config.subdomain) {
            new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
                zone,
                recordNames: [`www.${config.zoneName}`],
                targetDomain: config.zoneName,
            });
        }
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
_a = JSII_RTTI_SYMBOL_1;
SPADeploy[_a] = { fqn: "cdk-spa-deploy.SPADeploy", version: "2.0.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3BhLWRlcGxveS1jb25zdHJ1Y3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzcGEtZGVwbG95LWNvbnN0cnVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLCtEQU9vQztBQUNwQyxpREFBa0Y7QUFDbEYseURBQTRFO0FBQzVFLCtFQUE2RTtBQUM3RSwyRUFBaUU7QUFDakUseUVBQW1FO0FBQ25FLDZDQUF3QztBQUN4QywwREFBMEQ7QUFDMUQseUNBQTBDO0FBQzFDLDJDQUF1QztBQTBDdkMsTUFBYSxTQUFVLFNBQVEsc0JBQVM7SUFHcEMsWUFBWSxLQUFnQixFQUFFLEVBQVMsRUFBRSxNQUF1QjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksT0FBTyxNQUFNLEtBQUssV0FBVyxFQUFFO1lBQ2pDLElBQUksQ0FBQyxZQUFZLEdBQUcsTUFBTSxDQUFDO1NBQzVCO2FBQU07WUFDTCxJQUFJLENBQUMsWUFBWSxHQUFHO2dCQUNsQixhQUFhLEVBQUUsS0FBSztnQkFDcEIsUUFBUSxFQUFFLEtBQUs7YUFDaEIsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssV0FBVyxDQUFDLE1BQXNCLEVBQUUsZUFBd0I7UUFDbEUsTUFBTSxZQUFZLEdBQU87WUFDdkIsb0JBQW9CLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDckMsb0JBQW9CLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDckMsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsS0FBSyxJQUFJLEVBQUU7WUFDNUMsWUFBWSxDQUFDLFVBQVUsR0FBRyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDO1NBQzFEO1FBRUQsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsS0FBSyxJQUFJLElBQUksZUFBZSxLQUFLLElBQUksSUFBSSxPQUFPLE1BQU0sQ0FBQyxpQkFBaUIsS0FBSyxXQUFXLEVBQUU7WUFDdEgsWUFBWSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQztZQUN0QyxJQUFJLE9BQU8sTUFBTSxDQUFDLGlCQUFpQixLQUFLLFdBQVcsRUFBRTtnQkFDbkQsWUFBWSxDQUFDLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQzthQUMzRDtTQUNGO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFFbEUsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsS0FBSyxJQUFJLElBQUksZUFBZSxLQUFLLEtBQUssRUFBRTtZQUNwRSxJQUFJLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEtBQUssV0FBVyxFQUFFO2dCQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7YUFDeEU7WUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLHlCQUFlLEVBQUUsQ0FBQztZQUMzQyxZQUFZLENBQUMsZUFBZSxFQUFFLENBQUM7WUFDL0IsWUFBWSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN4QyxZQUFZLENBQUMsWUFBWSxDQUFDLEdBQUcsTUFBTSxDQUFDLFNBQVMsSUFBSSxDQUFDLENBQUM7WUFDbkQsWUFBWSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7Z0JBQ3JDLGNBQWMsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU07YUFDekMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQzFDO1FBRUQsNEpBQTRKO1FBQzVKLElBQUksTUFBTSxDQUFDLElBQUksRUFBRTtZQUNmLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDdEIsSUFBSSx5QkFBZSxDQUFDO2dCQUNsQixPQUFPLEVBQUU7b0JBQ1AsZUFBZTtvQkFDZixlQUFlO29CQUNmLFVBQVU7b0JBQ1Ysa0JBQWtCO29CQUNsQixlQUFlO29CQUNmLFdBQVc7aUJBQ1o7Z0JBQ0QsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztnQkFDcEIsU0FBUyxFQUFFLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDO2dCQUN4RCxVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFO3dCQUNaLGtCQUFrQixFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTztxQkFDeEM7aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUFFLENBQUMsSUFBSSxzQkFBWSxFQUFFLENBQUM7YUFDakMsQ0FBQyxDQUNMLENBQUM7U0FDSDtRQUVELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7T0FFRztJQUNLLFdBQVcsQ0FBQyxhQUF1QixFQUFFLE1BQVUsRUFBRSxjQUFvQyxFQUFFLElBQTZCO1FBQzFILE1BQU0sUUFBUSxHQUFPO1lBQ25CLGFBQWEsRUFBRTtnQkFDYjtvQkFDRSxjQUFjLEVBQUU7d0JBQ2QsY0FBYyxFQUFFLGFBQWE7d0JBQzdCLG9CQUFvQixFQUFFLGNBQWM7cUJBQ3JDO29CQUNELFNBQVMsRUFBRSxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ25GO2FBQ0Y7WUFDRCx3RkFBd0Y7WUFDeEYsbUJBQW1CLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLEdBQUc7b0JBQ2QsZ0JBQWdCLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUM7b0JBQ25GLFlBQVksRUFBRSxHQUFHO2lCQUNsQjtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsR0FBRztvQkFDZCxnQkFBZ0IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQztvQkFDbkYsWUFBWSxFQUFFLEdBQUc7aUJBQ2xCLENBQUM7U0FDSCxDQUFDO1FBRUYsSUFBSSxPQUFPLE1BQU0sQ0FBQyxjQUFjLEtBQUssV0FBVyxJQUFJLE9BQU8sTUFBTSxDQUFDLFNBQVMsS0FBSyxXQUFXLEVBQUU7WUFDM0YsUUFBUSxDQUFDLGtCQUFrQixHQUFHO2dCQUM1QixVQUFVLEVBQUUsTUFBTSxDQUFDLGNBQWM7Z0JBQ2pDLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUzthQUN4QixDQUFDO1NBQ0g7UUFDRCxJQUFJLE9BQU8sTUFBTSxDQUFDLFNBQVMsS0FBSyxXQUFXLEVBQUU7WUFDM0MsUUFBUSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDO1NBQzFEO1FBRUQsSUFBSSxPQUFPLE1BQU0sQ0FBQyxjQUFjLEtBQUssV0FBVyxFQUFFO1lBQ2hELFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxjQUFjLEdBQUcsTUFBTSxDQUFDLGNBQWMsQ0FBQztTQUNwRTtRQUVELElBQUksT0FBTyxNQUFNLENBQUMsUUFBUSxLQUFLLFdBQVcsSUFBSSxPQUFPLElBQUksS0FBSyxXQUFXLEVBQUU7WUFDekUsUUFBUSxDQUFDLGlCQUFpQixHQUFHLGtDQUFpQixDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRTtnQkFDdEUsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxNQUFNLENBQUMsU0FBUyxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQzthQUN6RixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7Ozs7SUFHTSxlQUFlLENBQUMsTUFBc0I7UUFDM0MsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFdEQsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3RELE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN0RCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUk7WUFDakIsaUJBQWlCLEVBQUUsYUFBYTtTQUNqQyxDQUFDLENBQUM7UUFFSCxNQUFNLGVBQWUsR0FBTztZQUMxQixXQUFXLEVBQUUsd0JBQXdCO1lBQ3JDLEtBQUssRUFBRSxhQUFhLENBQUMsZ0JBQWdCO1NBQ3RDLENBQUM7UUFFRixJQUFJLE1BQU0sQ0FBQyxzQkFBc0IsS0FBSyxJQUFJLEVBQUU7WUFDMUMsSUFBSSxPQUFPLE1BQU0sQ0FBQyxvQkFBb0IsS0FBSyxXQUFXLElBQUksTUFBTSxDQUFDLG9CQUFvQixLQUFLLEVBQUUsRUFBRTtnQkFDNUYsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRkFBaUYsQ0FBQyxDQUFDO2FBQ3BHO1lBQ0QsZUFBZSxDQUFDLFVBQVUsR0FBRyxNQUFNLENBQUMsb0JBQW9CLENBQUM7U0FDMUQ7UUFFRCxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztRQUU1QyxPQUFPLEVBQUUsYUFBYSxFQUFFLENBQUM7SUFDM0IsQ0FBQzs7Ozs7SUFHTSx3QkFBd0IsQ0FBQyxNQUFzQjtRQUNwRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNyRCxNQUFNLGNBQWMsR0FBRyxJQUFJLHFDQUFvQixDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLGFBQWEsQ0FBQyxVQUFVLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUMxSSxNQUFNLFlBQVksR0FBRyxJQUFJLDBDQUF5QixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUMsQ0FBQztRQUU1SSxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDdEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3RELGlCQUFpQixFQUFFLGFBQWE7WUFDaEMsaUdBQWlHO1lBQ2pHLFlBQVk7WUFDWixpQkFBaUIsRUFBRSxDQUFDLEdBQUcsRUFBRSxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUMvQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUk7U0FDbEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSx1QkFBUyxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUN2QyxXQUFXLEVBQUUsMkJBQTJCO1lBQ3hDLEtBQUssRUFBRSxZQUFZLENBQUMsc0JBQXNCO1NBQzNDLENBQUMsQ0FBQztRQUVILE9BQU8sRUFBRSxhQUFhLEVBQUUsWUFBWSxFQUFFLENBQUM7SUFDekMsQ0FBQzs7Ozs7SUFHTSx3QkFBd0IsQ0FBQyxNQUF1QjtRQUNyRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNyRCxNQUFNLElBQUksR0FBRyx3QkFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLEVBQUUsVUFBVSxFQUFFLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3hGLE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsTUFBTSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7UUFDakcsTUFBTSxJQUFJLEdBQUcsSUFBSSxnREFBdUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzVELFVBQVUsRUFBRSxJQUFJO1lBQ2hCLFVBQVU7WUFDVixNQUFNLEVBQUUsV0FBVztTQUNwQixDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxJQUFJLHFDQUFvQixDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLGFBQWEsQ0FBQyxVQUFVLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUMxSSxNQUFNLFlBQVksR0FBRyxJQUFJLDBDQUF5QixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7UUFFbEosSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3RELE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN0RCxpQkFBaUIsRUFBRSxhQUFhO1lBQ2hDLGlHQUFpRztZQUNqRyxZQUFZO1lBQ1osSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1lBQ2pCLGlCQUFpQixFQUFFLENBQUMsR0FBRyxFQUFFLElBQUksTUFBTSxDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQ2hELENBQUMsQ0FBQztRQUVILElBQUkscUJBQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ3pCLElBQUk7WUFDSixVQUFVLEVBQUUsVUFBVTtZQUN0QixNQUFNLEVBQUUsMEJBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxzQ0FBZ0IsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUNuRSxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRTtZQUNyQixJQUFJLG9DQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtnQkFDaEMsSUFBSTtnQkFDSixXQUFXLEVBQUUsQ0FBQyxPQUFPLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDdkMsWUFBWSxFQUFFLE1BQU0sQ0FBQyxRQUFRO2FBQ2hDLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxFQUFFLGFBQWEsRUFBRSxZQUFZLEVBQUUsQ0FBQztJQUN6QyxDQUFDOztBQTVOTCw4QkE2TkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xyXG4gIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24sXHJcbiAgVmlld2VyQ2VydGlmaWNhdGUsXHJcbiAgT3JpZ2luQWNjZXNzSWRlbnRpdHksXHJcbiAgQmVoYXZpb3IsXHJcbiAgU1NMTWV0aG9kLFxyXG4gIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wsXHJcbn0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQnO1xyXG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIFJvbGUsIEFueVByaW5jaXBhbCwgRWZmZWN0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XHJcbmltcG9ydCB7IEhvc3RlZFpvbmUsIEFSZWNvcmQsIFJlY29yZFRhcmdldCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzJztcclxuaW1wb3J0IHsgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcclxuaW1wb3J0IHsgSHR0cHNSZWRpcmVjdCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzLXBhdHRlcm5zJztcclxuaW1wb3J0IHsgQ2xvdWRGcm9udFRhcmdldCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzLXRhcmdldHMnO1xyXG5pbXBvcnQgeyBDZm5PdXRwdXQgfSBmcm9tICdhd3MtY2RrLWxpYic7XHJcbmltcG9ydCBzM2RlcGxveT0gcmVxdWlyZSgnYXdzLWNkay1saWIvYXdzLXMzLWRlcGxveW1lbnQnKTtcclxuaW1wb3J0IHMzID0gcmVxdWlyZSgnYXdzLWNkay1saWIvYXdzLXMzJyk7XHJcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xyXG5cclxuZXhwb3J0IGludGVyZmFjZSBTUEFEZXBsb3lDb25maWcge1xyXG4gIHJlYWRvbmx5IGluZGV4RG9jOnN0cmluZyxcclxuICByZWFkb25seSBlcnJvckRvYz86c3RyaW5nLFxyXG4gIHJlYWRvbmx5IHdlYnNpdGVGb2xkZXI6IHN0cmluZyxcclxuICByZWFkb25seSBjZXJ0aWZpY2F0ZUFSTj86IHN0cmluZyxcclxuICByZWFkb25seSBjZkJlaGF2aW9ycz86IEJlaGF2aW9yW10sXHJcbiAgcmVhZG9ubHkgY2ZBbGlhc2VzPzogc3RyaW5nW10sXHJcbiAgcmVhZG9ubHkgZXhwb3J0V2Vic2l0ZVVybE91dHB1dD86Ym9vbGVhbixcclxuICByZWFkb25seSBleHBvcnRXZWJzaXRlVXJsTmFtZT86IHN0cmluZyxcclxuICByZWFkb25seSBibG9ja1B1YmxpY0FjY2Vzcz86czMuQmxvY2tQdWJsaWNBY2Nlc3NcclxuICByZWFkb25seSBzc2xNZXRob2Q/OiBTU0xNZXRob2QsXHJcbiAgcmVhZG9ubHkgc2VjdXJpdHlQb2xpY3k/OiBTZWN1cml0eVBvbGljeVByb3RvY29sLFxyXG4gIHJlYWRvbmx5IHJvbGU/OlJvbGUsXHJcbn1cclxuXHJcbmV4cG9ydCBpbnRlcmZhY2UgSG9zdGVkWm9uZUNvbmZpZyB7XHJcbiAgcmVhZG9ubHkgaW5kZXhEb2M6c3RyaW5nLFxyXG4gIHJlYWRvbmx5IGVycm9yRG9jPzpzdHJpbmcsXHJcbiAgcmVhZG9ubHkgY2ZCZWhhdmlvcnM/OiBCZWhhdmlvcltdLFxyXG4gIHJlYWRvbmx5IHdlYnNpdGVGb2xkZXI6IHN0cmluZyxcclxuICByZWFkb25seSB6b25lTmFtZTogc3RyaW5nLFxyXG4gIHJlYWRvbmx5IHN1YmRvbWFpbj86IHN0cmluZyxcclxuICByZWFkb25seSByb2xlPzogUm9sZSxcclxufVxyXG5cclxuZXhwb3J0IGludGVyZmFjZSBTUEFHbG9iYWxDb25maWcge1xyXG4gIHJlYWRvbmx5IGVuY3J5cHRCdWNrZXQ/OmJvb2xlYW4sXHJcbiAgcmVhZG9ubHkgaXBGaWx0ZXI/OmJvb2xlYW4sXHJcbiAgcmVhZG9ubHkgaXBMaXN0PzpzdHJpbmdbXSxcclxuICByZWFkb25seSByb2xlPzpSb2xlLFxyXG59XHJcblxyXG5leHBvcnQgaW50ZXJmYWNlIFNQQURlcGxveW1lbnQge1xyXG4gIHJlYWRvbmx5IHdlYnNpdGVCdWNrZXQ6IHMzLkJ1Y2tldCxcclxufVxyXG5cclxuZXhwb3J0IGludGVyZmFjZSBTUEFEZXBsb3ltZW50V2l0aENsb3VkRnJvbnQgZXh0ZW5kcyBTUEFEZXBsb3ltZW50IHtcclxuICByZWFkb25seSBkaXN0cmlidXRpb246IENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24sXHJcbn1cclxuXHJcbmV4cG9ydCBjbGFzcyBTUEFEZXBsb3kgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xyXG4gICAgZ2xvYmFsQ29uZmlnOiBTUEFHbG9iYWxDb25maWc7XHJcblxyXG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6c3RyaW5nLCBjb25maWc/OlNQQUdsb2JhbENvbmZpZykge1xyXG4gICAgICBzdXBlcihzY29wZSwgaWQpO1xyXG5cclxuICAgICAgaWYgKHR5cGVvZiBjb25maWcgIT09ICd1bmRlZmluZWQnKSB7XHJcbiAgICAgICAgdGhpcy5nbG9iYWxDb25maWcgPSBjb25maWc7XHJcbiAgICAgIH0gZWxzZSB7XHJcbiAgICAgICAgdGhpcy5nbG9iYWxDb25maWcgPSB7XHJcbiAgICAgICAgICBlbmNyeXB0QnVja2V0OiBmYWxzZSxcclxuICAgICAgICAgIGlwRmlsdGVyOiBmYWxzZSxcclxuICAgICAgICB9O1xyXG4gICAgICB9XHJcbiAgICB9XHJcblxyXG4gICAgLyoqXHJcbiAgICAgKiBIZWxwZXIgbWV0aG9kIHRvIHByb3ZpZGUgYSBjb25maWd1cmVkIHMzIGJ1Y2tldFxyXG4gICAgICovXHJcbiAgICBwcml2YXRlIGdldFMzQnVja2V0KGNvbmZpZzpTUEFEZXBsb3lDb25maWcsIGlzRm9yQ2xvdWRGcm9udDogYm9vbGVhbikge1xyXG4gICAgICBjb25zdCBidWNrZXRDb25maWc6YW55ID0ge1xyXG4gICAgICAgIHdlYnNpdGVJbmRleERvY3VtZW50OiBjb25maWcuaW5kZXhEb2MsXHJcbiAgICAgICAgd2Vic2l0ZUVycm9yRG9jdW1lbnQ6IGNvbmZpZy5lcnJvckRvYyxcclxuICAgICAgICBwdWJsaWNSZWFkQWNjZXNzOiB0cnVlLFxyXG4gICAgICB9O1xyXG5cclxuICAgICAgaWYgKHRoaXMuZ2xvYmFsQ29uZmlnLmVuY3J5cHRCdWNrZXQgPT09IHRydWUpIHtcclxuICAgICAgICBidWNrZXRDb25maWcuZW5jcnlwdGlvbiA9IHMzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRDtcclxuICAgICAgfVxyXG5cclxuICAgICAgaWYgKHRoaXMuZ2xvYmFsQ29uZmlnLmlwRmlsdGVyID09PSB0cnVlIHx8IGlzRm9yQ2xvdWRGcm9udCA9PT0gdHJ1ZSB8fCB0eXBlb2YgY29uZmlnLmJsb2NrUHVibGljQWNjZXNzICE9PSAndW5kZWZpbmVkJykge1xyXG4gICAgICAgIGJ1Y2tldENvbmZpZy5wdWJsaWNSZWFkQWNjZXNzID0gZmFsc2U7XHJcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcuYmxvY2tQdWJsaWNBY2Nlc3MgIT09ICd1bmRlZmluZWQnKSB7XHJcbiAgICAgICAgICBidWNrZXRDb25maWcuYmxvY2tQdWJsaWNBY2Nlc3MgPSBjb25maWcuYmxvY2tQdWJsaWNBY2Nlc3M7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcblxyXG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdXZWJzaXRlQnVja2V0JywgYnVja2V0Q29uZmlnKTtcclxuXHJcbiAgICAgIGlmICh0aGlzLmdsb2JhbENvbmZpZy5pcEZpbHRlciA9PT0gdHJ1ZSAmJiBpc0ZvckNsb3VkRnJvbnQgPT09IGZhbHNlKSB7XHJcbiAgICAgICAgaWYgKHR5cGVvZiB0aGlzLmdsb2JhbENvbmZpZy5pcExpc3QgPT09ICd1bmRlZmluZWQnKSB7XHJcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1doZW4gSVAgRmlsdGVyIGlzIHRydWUgdGhlbiB0aGUgSVAgTGlzdCBpcyByZXF1aXJlZCcpO1xyXG4gICAgICAgIH1cclxuXHJcbiAgICAgICAgY29uc3QgYnVja2V0UG9saWN5ID0gbmV3IFBvbGljeVN0YXRlbWVudCgpO1xyXG4gICAgICAgIGJ1Y2tldFBvbGljeS5hZGRBbnlQcmluY2lwYWwoKTtcclxuICAgICAgICBidWNrZXRQb2xpY3kuYWRkQWN0aW9ucygnczM6R2V0T2JqZWN0Jyk7XHJcbiAgICAgICAgYnVja2V0UG9saWN5LmFkZFJlc291cmNlcyhgJHtidWNrZXQuYnVja2V0QXJufS8qYCk7XHJcbiAgICAgICAgYnVja2V0UG9saWN5LmFkZENvbmRpdGlvbignSXBBZGRyZXNzJywge1xyXG4gICAgICAgICAgJ2F3czpTb3VyY2VJcCc6IHRoaXMuZ2xvYmFsQ29uZmlnLmlwTGlzdCxcclxuICAgICAgICB9KTtcclxuXHJcbiAgICAgICAgYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koYnVja2V0UG9saWN5KTtcclxuICAgICAgfVxyXG5cclxuICAgICAgLy9UaGUgYmVsb3cgXCJyZWluZm9yY2VzXCIgdGhlIElBTSBSb2xlJ3MgYXR0YWNoZWQgcG9saWN5LCBpdCdzIG5vdCByZXF1aXJlZCBidXQgaXQgYWxsb3dzIGZvciBjdXN0b21lcnMgdXNpbmcgcGVybWlzc2lvbiBib3VuZGFyaWVzIHRvIHdyaXRlIGludG8gdGhlIGJ1Y2tldC5cclxuICAgICAgaWYgKGNvbmZpZy5yb2xlKSB7XHJcbiAgICAgICAgYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXHJcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xyXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcclxuICAgICAgICAgICAgICAgIFwiczM6R2V0T2JqZWN0KlwiLFxyXG4gICAgICAgICAgICAgICAgXCJzMzpHZXRCdWNrZXQqXCIsXHJcbiAgICAgICAgICAgICAgICBcInMzOkxpc3QqXCIsXHJcbiAgICAgICAgICAgICAgICBcInMzOkRlbGV0ZU9iamVjdCpcIixcclxuICAgICAgICAgICAgICAgIFwiczM6UHV0T2JqZWN0KlwiLFxyXG4gICAgICAgICAgICAgICAgXCJzMzpBYm9ydCpcIlxyXG4gICAgICAgICAgICAgIF0sXHJcbiAgICAgICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXHJcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmFybkZvck9iamVjdHMoJyonKSwgYnVja2V0LmJ1Y2tldEFybl0sXHJcbiAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xyXG4gICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XHJcbiAgICAgICAgICAgICAgICAgICdhd3M6UHJpbmNpcGFsQXJuJzogY29uZmlnLnJvbGUucm9sZUFybixcclxuICAgICAgICAgICAgICAgIH0sXHJcbiAgICAgICAgICAgICAgfSxcclxuICAgICAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXVxyXG4gICAgICAgICAgICB9KVxyXG4gICAgICAgICk7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIHJldHVybiBidWNrZXQ7XHJcbiAgICB9XHJcblxyXG4gICAgLyoqXHJcbiAgICAgKiBIZWxwZXIgbWV0aG9kIHRvIHByb3ZpZGUgY29uZmlndXJhdGlvbiBmb3IgY2xvdWRmcm9udFxyXG4gICAgICovXHJcbiAgICBwcml2YXRlIGdldENGQ29uZmlnKHdlYnNpdGVCdWNrZXQ6czMuQnVja2V0LCBjb25maWc6YW55LCBhY2Nlc3NJZGVudGl0eTogT3JpZ2luQWNjZXNzSWRlbnRpdHksIGNlcnQ/OkRuc1ZhbGlkYXRlZENlcnRpZmljYXRlKSB7XHJcbiAgICAgIGNvbnN0IGNmQ29uZmlnOmFueSA9IHtcclxuICAgICAgICBvcmlnaW5Db25maWdzOiBbXHJcbiAgICAgICAgICB7XHJcbiAgICAgICAgICAgIHMzT3JpZ2luU291cmNlOiB7XHJcbiAgICAgICAgICAgICAgczNCdWNrZXRTb3VyY2U6IHdlYnNpdGVCdWNrZXQsXHJcbiAgICAgICAgICAgICAgb3JpZ2luQWNjZXNzSWRlbnRpdHk6IGFjY2Vzc0lkZW50aXR5LFxyXG4gICAgICAgICAgICB9LFxyXG4gICAgICAgICAgICBiZWhhdmlvcnM6IGNvbmZpZy5jZkJlaGF2aW9ycyA/IGNvbmZpZy5jZkJlaGF2aW9ycyA6IFt7IGlzRGVmYXVsdEJlaGF2aW9yOiB0cnVlIH1dLFxyXG4gICAgICAgICAgfSxcclxuICAgICAgICBdLFxyXG4gICAgICAgIC8vIFdlIG5lZWQgdG8gcmVkaXJlY3QgYWxsIHVua25vd24gcm91dGVzIGJhY2sgdG8gaW5kZXguaHRtbCBmb3IgYW5ndWxhciByb3V0aW5nIHRvIHdvcmtcclxuICAgICAgICBlcnJvckNvbmZpZ3VyYXRpb25zOiBbe1xyXG4gICAgICAgICAgZXJyb3JDb2RlOiA0MDMsXHJcbiAgICAgICAgICByZXNwb25zZVBhZ2VQYXRoOiAoY29uZmlnLmVycm9yRG9jID8gYC8ke2NvbmZpZy5lcnJvckRvY31gIDogYC8ke2NvbmZpZy5pbmRleERvY31gKSxcclxuICAgICAgICAgIHJlc3BvbnNlQ29kZTogMjAwLFxyXG4gICAgICAgIH0sXHJcbiAgICAgICAge1xyXG4gICAgICAgICAgZXJyb3JDb2RlOiA0MDQsXHJcbiAgICAgICAgICByZXNwb25zZVBhZ2VQYXRoOiAoY29uZmlnLmVycm9yRG9jID8gYC8ke2NvbmZpZy5lcnJvckRvY31gIDogYC8ke2NvbmZpZy5pbmRleERvY31gKSxcclxuICAgICAgICAgIHJlc3BvbnNlQ29kZTogMjAwLFxyXG4gICAgICAgIH1dLFxyXG4gICAgICB9O1xyXG5cclxuICAgICAgaWYgKHR5cGVvZiBjb25maWcuY2VydGlmaWNhdGVBUk4gIT09ICd1bmRlZmluZWQnICYmIHR5cGVvZiBjb25maWcuY2ZBbGlhc2VzICE9PSAndW5kZWZpbmVkJykge1xyXG4gICAgICAgIGNmQ29uZmlnLmFsaWFzQ29uZmlndXJhdGlvbiA9IHtcclxuICAgICAgICAgIGFjbUNlcnRSZWY6IGNvbmZpZy5jZXJ0aWZpY2F0ZUFSTixcclxuICAgICAgICAgIG5hbWVzOiBjb25maWcuY2ZBbGlhc2VzLFxyXG4gICAgICAgIH07XHJcbiAgICAgIH1cclxuICAgICAgaWYgKHR5cGVvZiBjb25maWcuc3NsTWV0aG9kICE9PSAndW5kZWZpbmVkJykge1xyXG4gICAgICAgIGNmQ29uZmlnLmFsaWFzQ29uZmlndXJhdGlvbi5zc2xNZXRob2QgPSBjb25maWcuc3NsTWV0aG9kO1xyXG4gICAgICB9XHJcblxyXG4gICAgICBpZiAodHlwZW9mIGNvbmZpZy5zZWN1cml0eVBvbGljeSAhPT0gJ3VuZGVmaW5lZCcpIHtcclxuICAgICAgICBjZkNvbmZpZy5hbGlhc0NvbmZpZ3VyYXRpb24uc2VjdXJpdHlQb2xpY3kgPSBjb25maWcuc2VjdXJpdHlQb2xpY3k7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIGlmICh0eXBlb2YgY29uZmlnLnpvbmVOYW1lICE9PSAndW5kZWZpbmVkJyAmJiB0eXBlb2YgY2VydCAhPT0gJ3VuZGVmaW5lZCcpIHtcclxuICAgICAgICBjZkNvbmZpZy52aWV3ZXJDZXJ0aWZpY2F0ZSA9IFZpZXdlckNlcnRpZmljYXRlLmZyb21BY21DZXJ0aWZpY2F0ZShjZXJ0LCB7XHJcbiAgICAgICAgICBhbGlhc2VzOiBbY29uZmlnLnN1YmRvbWFpbiA/IGAke2NvbmZpZy5zdWJkb21haW59LiR7Y29uZmlnLnpvbmVOYW1lfWAgOiBjb25maWcuem9uZU5hbWVdLFxyXG4gICAgICAgIH0pO1xyXG4gICAgICB9XHJcblxyXG4gICAgICByZXR1cm4gY2ZDb25maWc7XHJcbiAgICB9XHJcblxyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcclxuICAgIHB1YmxpYyBjcmVhdGVCYXNpY1NpdGUoY29uZmlnOlNQQURlcGxveUNvbmZpZyk6IFNQQURlcGxveW1lbnQge1xyXG4gICAgICBjb25zdCB3ZWJzaXRlQnVja2V0ID0gdGhpcy5nZXRTM0J1Y2tldChjb25maWcsIGZhbHNlKTtcclxuXHJcbiAgICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHRoaXMsICdCdWNrZXREZXBsb3ltZW50Jywge1xyXG4gICAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQoY29uZmlnLndlYnNpdGVGb2xkZXIpXSxcclxuICAgICAgICByb2xlOiBjb25maWcucm9sZSxcclxuICAgICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogd2Vic2l0ZUJ1Y2tldCxcclxuICAgICAgfSk7XHJcblxyXG4gICAgICBjb25zdCBjZm5PdXRwdXRDb25maWc6YW55ID0ge1xyXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnVGhlIHVybCBvZiB0aGUgd2Vic2l0ZScsXHJcbiAgICAgICAgdmFsdWU6IHdlYnNpdGVCdWNrZXQuYnVja2V0V2Vic2l0ZVVybCxcclxuICAgICAgfTtcclxuXHJcbiAgICAgIGlmIChjb25maWcuZXhwb3J0V2Vic2l0ZVVybE91dHB1dCA9PT0gdHJ1ZSkge1xyXG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLmV4cG9ydFdlYnNpdGVVcmxOYW1lID09PSAndW5kZWZpbmVkJyB8fCBjb25maWcuZXhwb3J0V2Vic2l0ZVVybE5hbWUgPT09ICcnKSB7XHJcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1doZW4gT3V0cHV0IFVSTCBhcyBBV1MgRXhwb3J0IHByb3BlcnR5IGlzIHRydWUgdGhlbiB0aGUgb3V0cHV0IG5hbWUgaXMgcmVxdWlyZWQnKTtcclxuICAgICAgICB9XHJcbiAgICAgICAgY2ZuT3V0cHV0Q29uZmlnLmV4cG9ydE5hbWUgPSBjb25maWcuZXhwb3J0V2Vic2l0ZVVybE5hbWU7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1VSTCcsIGNmbk91dHB1dENvbmZpZyk7XHJcblxyXG4gICAgICByZXR1cm4geyB3ZWJzaXRlQnVja2V0IH07XHJcbiAgICB9XHJcblxyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXHJcbiAgICBwdWJsaWMgY3JlYXRlU2l0ZVdpdGhDbG91ZGZyb250KGNvbmZpZzpTUEFEZXBsb3lDb25maWcpOiBTUEFEZXBsb3ltZW50V2l0aENsb3VkRnJvbnQge1xyXG4gICAgICBjb25zdCB3ZWJzaXRlQnVja2V0ID0gdGhpcy5nZXRTM0J1Y2tldChjb25maWcsIHRydWUpO1xyXG4gICAgICBjb25zdCBhY2Nlc3NJZGVudGl0eSA9IG5ldyBPcmlnaW5BY2Nlc3NJZGVudGl0eSh0aGlzLCAnT3JpZ2luQWNjZXNzSWRlbnRpdHknLCB7IGNvbW1lbnQ6IGAke3dlYnNpdGVCdWNrZXQuYnVja2V0TmFtZX0tYWNjZXNzLWlkZW50aXR5YCB9KTtcclxuICAgICAgY29uc3QgZGlzdHJpYnV0aW9uID0gbmV3IENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24odGhpcywgJ2Nsb3VkZnJvbnREaXN0cmlidXRpb24nLCB0aGlzLmdldENGQ29uZmlnKHdlYnNpdGVCdWNrZXQsIGNvbmZpZywgYWNjZXNzSWRlbnRpdHkpKTtcclxuXHJcbiAgICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHRoaXMsICdCdWNrZXREZXBsb3ltZW50Jywge1xyXG4gICAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQoY29uZmlnLndlYnNpdGVGb2xkZXIpXSxcclxuICAgICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogd2Vic2l0ZUJ1Y2tldCxcclxuICAgICAgICAvLyBJbnZhbGlkYXRlIHRoZSBjYWNoZSBmb3IgLyBhbmQgaW5kZXguaHRtbCB3aGVuIHdlIGRlcGxveSBzbyB0aGF0IGNsb3VkZnJvbnQgc2VydmVzIGxhdGVzdCBzaXRlXHJcbiAgICAgICAgZGlzdHJpYnV0aW9uLFxyXG4gICAgICAgIGRpc3RyaWJ1dGlvblBhdGhzOiBbJy8nLCBgLyR7Y29uZmlnLmluZGV4RG9jfWBdLFxyXG4gICAgICAgIHJvbGU6IGNvbmZpZy5yb2xlLFxyXG4gICAgICB9KTtcclxuXHJcbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ2Nsb3VkZnJvbnQgZG9tYWluJywge1xyXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnVGhlIGRvbWFpbiBvZiB0aGUgd2Vic2l0ZScsXHJcbiAgICAgICAgdmFsdWU6IGRpc3RyaWJ1dGlvbi5kaXN0cmlidXRpb25Eb21haW5OYW1lLFxyXG4gICAgICB9KTtcclxuXHJcbiAgICAgIHJldHVybiB7IHdlYnNpdGVCdWNrZXQsIGRpc3RyaWJ1dGlvbiB9O1xyXG4gICAgfVxyXG5cclxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcclxuICAgIHB1YmxpYyBjcmVhdGVTaXRlRnJvbUhvc3RlZFpvbmUoY29uZmlnOkhvc3RlZFpvbmVDb25maWcpOiBTUEFEZXBsb3ltZW50V2l0aENsb3VkRnJvbnQge1xyXG4gICAgICBjb25zdCB3ZWJzaXRlQnVja2V0ID0gdGhpcy5nZXRTM0J1Y2tldChjb25maWcsIHRydWUpO1xyXG4gICAgICBjb25zdCB6b25lID0gSG9zdGVkWm9uZS5mcm9tTG9va3VwKHRoaXMsICdIb3N0ZWRab25lJywgeyBkb21haW5OYW1lOiBjb25maWcuem9uZU5hbWUgfSk7XHJcbiAgICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBjb25maWcuc3ViZG9tYWluID8gYCR7Y29uZmlnLnN1YmRvbWFpbn0uJHtjb25maWcuem9uZU5hbWV9YCA6IGNvbmZpZy56b25lTmFtZTtcclxuICAgICAgY29uc3QgY2VydCA9IG5ldyBEbnNWYWxpZGF0ZWRDZXJ0aWZpY2F0ZSh0aGlzLCAnQ2VydGlmaWNhdGUnLCB7XHJcbiAgICAgICAgaG9zdGVkWm9uZTogem9uZSxcclxuICAgICAgICBkb21haW5OYW1lLFxyXG4gICAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXHJcbiAgICAgIH0pO1xyXG5cclxuICAgICAgY29uc3QgYWNjZXNzSWRlbnRpdHkgPSBuZXcgT3JpZ2luQWNjZXNzSWRlbnRpdHkodGhpcywgJ09yaWdpbkFjY2Vzc0lkZW50aXR5JywgeyBjb21tZW50OiBgJHt3ZWJzaXRlQnVja2V0LmJ1Y2tldE5hbWV9LWFjY2Vzcy1pZGVudGl0eWAgfSk7XHJcbiAgICAgIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHRoaXMsICdjbG91ZGZyb250RGlzdHJpYnV0aW9uJywgdGhpcy5nZXRDRkNvbmZpZyh3ZWJzaXRlQnVja2V0LCBjb25maWcsIGFjY2Vzc0lkZW50aXR5LCBjZXJ0KSk7XHJcblxyXG4gICAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudCh0aGlzLCAnQnVja2V0RGVwbG95bWVudCcsIHtcclxuICAgICAgICBzb3VyY2VzOiBbczNkZXBsb3kuU291cmNlLmFzc2V0KGNvbmZpZy53ZWJzaXRlRm9sZGVyKV0sXHJcbiAgICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHdlYnNpdGVCdWNrZXQsXHJcbiAgICAgICAgLy8gSW52YWxpZGF0ZSB0aGUgY2FjaGUgZm9yIC8gYW5kIGluZGV4Lmh0bWwgd2hlbiB3ZSBkZXBsb3kgc28gdGhhdCBjbG91ZGZyb250IHNlcnZlcyBsYXRlc3Qgc2l0ZVxyXG4gICAgICAgIGRpc3RyaWJ1dGlvbixcclxuICAgICAgICByb2xlOiBjb25maWcucm9sZSxcclxuICAgICAgICBkaXN0cmlidXRpb25QYXRoczogWycvJywgYC8ke2NvbmZpZy5pbmRleERvY31gXSxcclxuICAgICAgfSk7XHJcblxyXG4gICAgICBuZXcgQVJlY29yZCh0aGlzLCAnQWxpYXMnLCB7XHJcbiAgICAgICAgem9uZSxcclxuICAgICAgICByZWNvcmROYW1lOiBkb21haW5OYW1lLFxyXG4gICAgICAgIHRhcmdldDogUmVjb3JkVGFyZ2V0LmZyb21BbGlhcyhuZXcgQ2xvdWRGcm9udFRhcmdldChkaXN0cmlidXRpb24pKSxcclxuICAgICAgfSk7XHJcblxyXG4gICAgICBpZiAoIWNvbmZpZy5zdWJkb21haW4pIHtcclxuICAgICAgICBuZXcgSHR0cHNSZWRpcmVjdCh0aGlzLCAnUmVkaXJlY3QnLCB7XHJcbiAgICAgICAgICAgIHpvbmUsXHJcbiAgICAgICAgICAgIHJlY29yZE5hbWVzOiBbYHd3dy4ke2NvbmZpZy56b25lTmFtZX1gXSxcclxuICAgICAgICAgICAgdGFyZ2V0RG9tYWluOiBjb25maWcuem9uZU5hbWUsXHJcbiAgICAgICAgfSk7ICAgICAgICAgIFxyXG4gICAgICB9XHJcblxyXG4gICAgICByZXR1cm4geyB3ZWJzaXRlQnVja2V0LCBkaXN0cmlidXRpb24gfTtcclxuICAgIH1cclxufVxyXG4iXX0=