# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flask_firebase_admin']

package_data = \
{'': ['*']}

install_requires = \
['firebase-admin>=4.0.0,<5.0.0', 'flask>=1.0.0,<2.0.0']

setup_kwargs = {
    'name': 'flask-firebase-admin',
    'version': '0.2.3',
    'description': 'Firebase for Flask',
    'long_description': '# Flask Firebase Admin\n\nAdd Firebase (a Firebase Admin app) to a Flask application.\n\n[![PyPI Version](https://img.shields.io/pypi/v/flask-firebase-admin.svg)](https://pypi.org/project/flask-firebase-admin/)\n[![Tests](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/test.yaml/badge.svg)](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/test.yaml)\n[![codecov](https://codecov.io/gh/andrewrosss/flask-firebase-admin/branch/master/graph/badge.svg?token=JM7PL13H59)](https://codecov.io/gh/andrewrosss/flask-firebase-admin)\n[![Type Check](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/type-check.yaml/badge.svg)](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/type-check.yaml)\n[![Code Style](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/lint.yaml/badge.svg)](https://github.com/andrewrosss/flask-firebase-admin/actions/workflows/lint.yaml)\n\n## Installation\n\n```bash\npip install flask-firebase-admin\n```\n\n## Quickstart\n\nIn the simplest case, let\'s protect a route, specifically, we\'ll require a user to provide a firebase jwt to one of our routes:\n\n```python\nfrom flask import Flask, request\nfrom flask_firebase_admin import FirebaseAdmin\n\napp = Flask(__name__)\nfirebase = FirebaseAdmin(app)  # uses GOOGLE_APPLICATION_CREDENTIALS\n\n@app.route("/unprotected")\ndef unprotected():\n    return {"message": "Hello anonymous user!"}\n\n@app.route("/protected")\n@firebase.jwt_required  # This route now requires authorization via firebase jwt\ndef protected():\n    # By default JWT payload is stored under request.jwt_payload\n    return {"message": f"Hello {request.jwt_payload[\'email\']}!"}\n\nif __name__ == "__main__":\n    app.run(debug=True)\n```\n\nAssuming the code above is located in a module named `app.py`, start the Flask application:\n\n```bash\nGOOGLE_APPLICATION_CREDENTIALS="/path/to/service_account.json" python app.py\n```\n\nAnd in a separate terminal window, ping the unprotected route:\n\n```bash\n$ curl http://127.0.0.1:5000/unprotected\n{\n  "message": "Hello anonymous user!"\n}\n```\n\nLooks good. Now the protected route:\n\n```bash\n$ curl http://127.0.0.1:5000/protected\n{\n  "error": {\n    "message": "No credentials provided"\n  }\n}\n```\n\nOK, makes sense. Now with some credentials:\n\n```bash\n$ TOKEN="your-firebase-token ..."\n$ curl -H "Authorization: Bearer ${TOKEN}" http://127.0.0.1:5000/protected\n{\n  "message": "Hello <your@email.com>!"\n}\n```\n\nExcellent. We now have a application with routes (one route) which require the user to provide their Firebase JWT to access!\n\nInternally the `jwt_required` method provided by the `FirebaseAdmin` object calls the `firebase_admin.auth.verify_id_token` function which returns a dictionary of key-value pairs parsed from the decoded JWT. This dictionary is accessible via the `request` object provided by flask, specifically, this information is attached to the `request.jwt_payload` attribute by default.\n\n## Configuration\n\n> **Note:** The following groups of configuration parameters are mutually exclusive:\n>\n> - `FIREBASE_ADMIN_APP` (This config parameter takes precendence and if specified then configuration from the other group is ignored)\n>\n>   ***\n>\n> - `FIREBASE_ADMIN_CREDENTIAL`\n>\n>   `FIREBASE_ADMIN_OPTIONS`\n>\n>   `FIREBASE_ADMIN_NAME`\n>\n>   `FIREBASE_ADMIN_RAISE_IF_APP_EXISTS`\n\nThe `FirebaseAdmin` object can be configured in the following ways:\n\n- `FIREBASE_ADMIN_CREDENTIAL`\n\n  **Defaults to `None`**. This is the credential passed to the call to `firebase_admin.initialize_app`. When this parameter is None the Firebase Admin application tries to initialize using `GOOGLE_APPLICATION_CREDENTIALS`. If initializing the Firebase Admin app with the `GOOGLE_APPLICATION_CREDENTIALS` environment variable is undesirebale, credentials can be created manually, for example:\n\n  ```python\n  app = Flask(__name__)\n  app.config["FIREBASE_ADMIN_CREDENTIAL"] = credentials.Certificate("/path/to/key.json")\n  firebase = FirebaseAdmin(app)  # no longer uses GOOGLE_APPLICATION_CREDENTIALS\n  ```\n\n  Or perhaps something like:\n\n  ```python\n  app = Flask(__name__)\n  cert = {\n      "type": "service_account",\n      "project_id": os.getenv("PROJECT_ID"),\n      "private_key_id": os.getenv("PRIVATE_KEY_ID"),\n      "private_key": os.getenv("PRIVATE_KEY"),\n      "client_email": os.getenv("CLIENT_EMAIL"),\n      "client_id": os.getenv("CLIENT_ID"),\n      "auth_uri": "https://accounts.google.com/o/oauth2/auth",\n      "token_uri": "https://oauth2.googleapis.com/token",\n      "auth_provider_x509_cert_url": "https://www.googleapis.com/oauth2/v1/certs",\n      "client_x509_cert_url": os.getenv("CLIENT_X509_CERT_URL"),\n  }\n  app.config["FIREBASE_ADMIN_CREDENTIAL"] = credentials.Certificate(cert)\n  firebase = FirebaseAdmin(app)  # no longer uses GOOGLE_APPLICATION_CREDENTIALS\n  ```\n\n- `FIREBASE_ADMIN_OPTIONS`\n\n  **Defaults to `None`**. This config is passed directly to `admin.initialize_app()` as the second `options` argument. From the Firebase Admin SDK docs: _A dictionary of configuration options (optional). Supported options include **databaseURL**, **storageBucket**, **projectId**, **databaseAuthVariableOverride**, **serviceAccountId** and **httpTimeout**. If httpTimeout is not set, the SDK uses a default timeout of 120 seconds._\n\n- `FIREBASE_ADMIN_NAME`\n\n  **Defaults to `\'[DEFAULT]\'`**. This config is passed directly to `admin.initialize_app()` as the third `name` argument.\n\n- `FIREBASE_ADMIN_AUTHORIZATION_SCHEME`\n\n  **Defaults to `\'Bearer\'`**. This is the authorization scheme expected by the `FirebaseAdmin` object. Changing this parameter changes the format of the auth header that is required by the client. For example, if we set this to `\'JWT\'` we would then need to include an authorization header of the form: `Authorization: JWT <token>` when making requests to protected routes.\n\n- `FIREBASE_ADMIN_CHECK_REVOKED`\n\n  **Defaults to `True`**. This parameter is passed as the `check_revoked` argument in the call to `firebase_admin.auth.verify_id_token()`.\n\n- `FIREBASE_ADMIN_PAYLOAD_ATTR`\n\n  **Defaults to `\'jwt_payload\'`**. This is attribute on the flask `request` object from which we can access the JWT payload data. If we were to change this to, say, `\'jwt\'` we would then access the JWT payload using `request.jwt`\n\n- `FIREBASE_ADMIN_RAISE_IF_APP_EXISTS`\n\n  **Defaults to `True`**. Internally, `flask-firebase-admin` calls `admin.initialize_app()`, if the app with the configured name already exists the Firebase Admin SDK raises a `ValueError` exception. When this config variable is set to `False`, `flask-firebase-admin` will catch this error, get, and subsequently use the existing admin app by the given name.\n\n- `FIREBASE_ADMIN_APP`\n\n  **Defaults to `None`**. This is a way to explicity provided the `FirebaseAdmin` extension with a particular firebase admin app to use. For example:\n\n  ```python\n  import firebase_admin\n  from flask import Flask\n  from flask_firebase_admin import FirebaseAdmin\n\n  # elsewhere ...\n  default_admin_app = firebase_admin.initialize_app()\n  other_admin_app = firebase_admin.initialize_app(other_creds, other_options, other_name)\n\n  # then ...\n  app = Flask(__name__)\n  app.config["FIREBASE_ADMIN_APP"] = other_admin_app\n\n  # now firebase.jwt_required will use other_admin_app for authentication\n  firebase = FirebaseAdmin(app)\n  ```\n\nAn example using more of the available configuration:\n\n```python\nfrom flask import Flask, request\nfrom firebase_admin import credentials\nfrom flask_firebase_admin import FirebaseAdmin\n\napp = Flask(__name__)\napp.config["FIREBASE_ADMIN_CREDENTIAL"] = credentials.Certificate("/path/to/key.json")\napp.config["FIREBASE_ADMIN_AUTHORIZATION_SCHEME"] = "JWT"\napp.config["FIREBASE_ADMIN_CHECK_REVOKED"] = False  # don\'t check for revoked tokens\napp.config["FIREBASE_ADMIN_PAYLOAD_ATTR"] = "firebase_jwt"\n\n# initialized with credentials defined above, not GOOGLE_APPLICATION_CREDENTIALS\nfirebase = FirebaseAdmin(app)\n\n@app.route("/unprotected")\ndef unprotected():\n    return {"message": "Hello anonymous user!"}\n\n@app.route("/protected")\n@firebase.jwt_required  # This route now requires authorization via firebase jwt\ndef protected():\n    # we now access the JWT payload using request.firebase_jwt\n    return {"message": f"Hello {request.firebase_jwt[\'email\']}!"}\n\nif __name__ == "__main__":\n    app.run(debug=True)\n```\n\nTo call the `/protected` route we have to update our auth header that we sent originally:\n\n```bash\n$ TOKEN="your-firebase-token ..."\n$ curl -H "Authorization: JWT ${TOKEN}" http://127.0.0.1:5000/protected\n{\n  "message": "Hello <your@email.com>!"\n}\n```\n\n## Extras\n\nFor convenience, the modules in the `firebase_admin` package are aliased as class-level attributes on the `FirebaseAdmin` object. For example:\n\n```python\nfrom flask import Flask\nfrom flask_firebase_admin import FirebaseAdmin\n\napp = Flask(__name__)\nfirebase = FirebaseAdmin(app)\ndb = firebase.firestore.client()  # <-- connect firestore client\n\n@app.route("/unprotected")\ndef unprotected():\n    return {"message": "Hello anonymous user!"}\n\n@app.route("/protected")\n@firebase.jwt_required\ndef protected():\n    # do stuff in firestore using the db object defined above.\n    ...\n\nif __name__ == "__main__":\n    app.run(debug=True)\n```\n',
    'author': 'Andrew Ross',
    'author_email': 'andrew.ross.mail@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/andrewrosss/flask-firebase-admin',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
