#    Copyright 2015-2017 ARM Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
This class sublclasses :mod:`trappy.plotter.StaticPlot.StaticPlot` to
implement a line plot.
"""
from __future__ import unicode_literals
from __future__ import division
from __future__ import print_function

from trappy.plotter import AttrConf
from trappy.plotter.StaticPlot import StaticPlot

class LinePlot(StaticPlot):
    """
    This class uses :mod:`trappy.plotter.Constraint.Constraint` to
    represent different permutations of input parameters. These
    constraints are generated by creating an instance of
    :mod:`trappy.plotter.Constraint.ConstraintManager`.

    :param traces: The input data
    :type traces: a list of :mod:`trappy.trace.FTrace`,
        :mod:`trappy.trace.SysTrace`, :mod:`trappy.trace.BareTrace`
        or :mod:`pandas.DataFrame` or a single instance of them.

    :param column: specifies the name of the column to
           be plotted.
    :type column: (str, list(str))

    :param templates: TRAPpy events

        .. note::

                This is not required if a :mod:`pandas.DataFrame` is
                used

    :type templates: :mod:`trappy.base.Base`

    :param filters: Filter the column to be plotted as per the
        specified criteria. For Example:
        ::

            filters =
                    {
                        "pid": [ 3338 ],
                        "cpu": [0, 2, 4],
                    }
    :type filters: dict

    :param per_line: Used to control the number of graphs
        in each graph subplot row
    :type per_line: int

    :param concat: Draw all the pivots on a single graph
    :type concat: bool

    :param fill: Fill the area under the plots
    :type fill: bool

    :param permute: Draw one plot for each of the traces specified
    :type permute: bool

    :param drawstyle: This argument is forwarded to the matplotlib
        corresponding :func:`matplotlib.pyplot.plot` call

        drawing style.

        .. note::

            step plots are not currently supported for filled
            graphs

    :param xlim: A tuple representing the upper and lower xlimits
    :type xlim: tuple

    :param ylim: A tuple representing the upper and lower ylimits
    :type ylim: tuple

    :param title: A title describing all the generated plots
    :type title: str

    :param style: Created pre-styled graphs loaded from
        :mod:`trappy.plotter.AttrConf.MPL_STYLE`
    :type style: bool

    :param signals: A string of the type event_name:column to indicate
        the value that needs to be plotted.  You can add an additional
        parameter to specify the color of the lin in rgb:
        "event_name:column:color".  The color is specified as a comma
        separated list of rgb values, from 0 to 255 or from 0x0 to
        0xff.  E.g. 0xff,0x0,0x0 is red and 100,40,32 is brown.

        .. note::

            - Only one of `signals` or both `templates` and
              `columns` should be specified
            - Signals format won't work for :mod:`pandas.DataFrame`
              input

    :type signals: str

    """

    def __init__(self, traces, templates=None, **kwargs):
        # Default keys, each can be overridden in kwargs

        super(LinePlot, self).__init__(
            traces=traces,
            templates=templates,
            **kwargs)

        self._check_add_scatter()

    def set_defaults(self):
        """Sets the default attrs"""
        super(LinePlot, self).set_defaults()
        self._attr["scatter"] = AttrConf.PLOT_SCATTER
        self._attr["fill"] = AttrConf.FILL

    def _check_add_scatter(self):
        """Check if a scatter plot is needed
        and augment the forwarded args accordingly"""

        if self._attr["scatter"]:
            self._attr["args_to_forward"]["linestyle"] = ""
            self._attr["args_to_forward"]["marker"] = "o"
            if "point_size" in self._attr:
                self._attr["args_to_forward"]["markersize"] = \
                    self._attr["point_size"]

    def fill_line(self, axis, line_2d, cmap_index):
        """Fill the area under a line"""
        drawstyle = line_2d.get_drawstyle()
        if drawstyle.startswith("steps"):
            # This has been fixed in upstream matplotlib
            raise UserWarning("matplotlib does not support fill for step plots")

        xdat, ydat = line_2d.get_data(orig=False)
        axis.fill_between(
            xdat,
            axis.get_ylim()[0],
            ydat,
            facecolor=self._cmap.cmap(cmap_index),
            alpha=AttrConf.ALPHA)

    def plot_axis(self, axis, series_list, permute, concat, args_to_forward):
        """Internal Method called to plot data (series_list) on a given axis"""
        for i, (constraint, pivot) in enumerate(series_list):
            result = constraint.result
            line_2d_list = axis.plot(
                result[pivot].index,
                result[pivot].values,
                color=self._cmap.cmap(i),
                **args_to_forward
            )

            if self._attr["fill"]:
                self.fill_line(axis, line_2d_list[0], i)

            axis.set_title(self.make_title(constraint, pivot, permute, concat))

            self.add_to_legend(i, line_2d_list[0], constraint, pivot, concat, permute)
