# -*- coding: utf-8 -*-
# Copyright (c) 2016 Sqreen. All Rights Reserved.
# Please refer to our terms for more information: https://www.sqreen.io/terms.html
""" Config module
"""

import os
from os.path import isfile, expanduser

try:
    # python2
    from ConfigParser import RawConfigParser
except ImportError:
    # python3
    from configparser import RawConfigParser

from .env import get


HOME_FILE_PATH = expanduser("~/.sqreen.ini")


CONFIG_DEFAULT_VALUE = {
    # The backend URL
    "URL": "https://back.sqreen.io",

    # Should the client check the rules signatures
    "RULES_VERIFY_SIGNATURE": True,

    # Log configuration
    "LOG_LEVEL": "CRITICAL",
    "LOG_LOCATION": None,

    # Should sqreen protection run while application is tested?
    "RUN_IN_TEST": False,

    # Should sqreen report callback performance on newrelic?
    "REPORT_PERF_NR": False,
}


class Config(object):
    """ Config class to parse several sources of config and merge them

    This try to load configuration by different ways.
    1. Default value
    2. By file, which overrides whatever result we found in 1:
      a. From path in environment variable SQREEN_CONFIG_FILE
      b. From local directory sqreen.ini
      c. From home in ~/.sqreen.ini
    3. From Django configuration directly
    4. From the environment, which overrides whatever result we found in 1 or 2.
    """

    FILE_ENV_VAR = "SQREEN_CONFIG_FILE"
    HOME_FILE_PATH = HOME_FILE_PATH

    def __init__(self, default_values=None):
        self.config = {}

        if default_values is None:
            default_values = CONFIG_DEFAULT_VALUE

        self.default_values = default_values

        self.loaders = [
            self.load_from_default_values,
            self.load_from_file,
            self.load_from_env,
        ]

    def load(self):
        """ Call each loaders and update the config at the end
        """
        base_config = {}

        for loader in self.loaders:
            loaded = loader()

            if loaded:
                base_config.update(loaded)

        self.config = base_config

    def load_from_default_values(self):
        """ Returns default values
        """
        return self.default_values

    def load_from_file(self):
        """ Load from ONE config file, the file path is explained in class
        docstring
        """

        file_path = self._file_path_from_env() or self._file_path_from_home() \
            or self._file_path_from_local()

        if not file_path:
            return {}

        config = RawConfigParser()
        config.read(file_path)

        config_dict = {}
        for option in config.options('sqreen'):
            config_dict[option.upper()] = config.get('sqreen', option)

        return config_dict

    @staticmethod
    def load_from_env():
        """ Load configuration from os environment variables, variables
        must be prefixed with SQREEN_ to be detected.
        """
        env_config = {}
        for env_var in os.environ:
            if env_var.startswith('SQREEN_'):
                env_config[env_var.lstrip('SQREEN_')] = os.environ[env_var]

        return env_config

    def _file_path_from_env(self):
        """ Return file path if os environement was set and file exists
        """
        path = get(self.FILE_ENV_VAR)

        if path and isfile(path):
            return path

    def _file_path_from_home(self):
        """ Return file path if file exists in home directory
        """
        if HOME_FILE_PATH and isfile(HOME_FILE_PATH):
            return HOME_FILE_PATH

    @staticmethod
    def _file_path_from_local():
        """ Return file path if file exists locally on the project
        """
        if isfile("sqreen.ini"):
            return os.path.join(os.getcwd(), "sqreen.ini")

    def __getitem__(self, name):
        return self.config[name]
