"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(stack, 'runners', {});
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(stack, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(stack, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(stack, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   stack, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   stack,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *     defaultProviderLabel: 'my-codebuild',
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        const defaultProvider = this.getDefaultProvider();
        if (!defaultProvider) {
            throw new Error(`No provider was found for the default label "${this.props.defaultProviderLabel}"`);
        }
        else {
            this.defaultProvider = defaultProvider;
        }
        this.orchestrator = this.stateMachine();
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    getDefaultProvider() {
        for (const provider of this.providers) {
            if ((this.props.defaultProviderLabel || 'codebuild') == provider.label) {
                return provider;
            }
        }
        return null;
    }
    stateMachine() {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: this.deleteRunner(),
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${provider.label}`), providerTask);
            if (this.defaultProvider == provider) {
                providerChooser.otherwise(providerTask);
            }
        }
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        return new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
            },
            timeout: cdk.Duration.seconds(30),
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const providers = this.providers.map(provider => {
            return {
                type: provider.constructor.name,
                label: provider.label,
                vpcArn: provider.vpc && provider.vpc.vpcArn,
                securityGroup: provider.securityGroup && provider.securityGroup.securityGroupId,
                roleArn: provider.grantPrincipal.grantPrincipal.roleArn,
            };
        });
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                PROVIDERS: JSON.stringify(providers),
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                SETUP_FUNCTION_URL: this.setupUrl,
            },
            timeout: cdk.Duration.minutes(3),
        });
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${cdk.Stack.of(this).region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
            },
            timeout: cdk.Duration.minutes(3),
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        // this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.1.1" };
//# sourceMappingURL=data:application/json;base64,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